/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: SymbolTable.h,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.5 $	$Date: 95/11/01 07:57:00 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *  Stores the functions available to get info from a molecule
 *  Calls the atom selection parser to create a parse tree
 *
 ***************************************************************************/
#ifndef SYMBOLTABLE_H
#define SYMBOLTABLE_H

#include <stddef.h>
#include "String.h"
#include "ResizeArray.h"

// originally this was designed as a template, however, not all
// compilers handle templates w/ member functions, so we do this
// instead
#define SYMBOLTABLE_TEMP_HDR 
#define SYMBOLTABLE_TEMP_FCTN 
#define SYMBOLTABLE_TEMP_REF
// for the template version they should be
//#define SYMBOLTABLE_TEMP_HDR template <class T>
//#define SYMBOLTABLE_TEMP_FCTN T::
//#define SYMBOLTABLE_TEMP_REF <T>

// these are used to catch errors during the regex compile
void symboltable_regex_reset(void);
int symboltable_regex_check(const char ** = NULL);
void symboltable_regex_swap(void);

typedef struct symbol_data; // forward declarations
SYMBOLTABLE_TEMP_HDR class ParseTree;

// the data strcutures must be broken up into two parts, one
// that is not template dependent, for use in the parser, and
// one that is template dependent, to catch the use of member
// functions.
//   The template independent part keeps track of the mapping from
// regex to function index, the dependent part converts from
// index to the given member function

// KEYWORD and SINGLEWORD must point to member functions of T
//  (eg, 'resname' and 'backbone')
// FUNCTION must point to global functions (eg, 'cos' and 'sqrt')
typedef struct SymbolTableName {
   enum symtype {IS_INT, IS_FLOAT, IS_STRING};
   enum symdesc {NOTHING, KEYWORD, FUNCTION, SINGLEWORD, STRINGFCTN};
   GString pattern;
   GString visible; // name w/o the regex characters
   Regex *regex;   // need to do your own destructor for this,

   SymbolTableName(void);
   SymbolTableName(const char *new_pattern, const char *visible,
		   symdesc new_desc,
		   symtype new_takes, symtype new_returns);
   ~SymbolTableName(void);

   symdesc is_a;
   symtype takes_a;
   symtype returns_a;
   
   // The Regex class has no provision for a copy, so I can't do
   // everything via the ResizeArray template.  Thus, if this class
   // used, the regex must explicitly be deallocated, via 'unset'
   // the 'set' can also return the latest error message
   int set(const char **err = NULL); // recompiles the string into regex
   void unset(void);                 // deletes the regex
};

// this keeps track of the template dependent mapping from index
// to member function
SYMBOLTABLE_TEMP_HDR
class SymbolTableElement {
public:   
  // these acccess (extract) the data
   union {
      void *fctn;     // the function pointer
      int (SYMBOLTABLE_TEMP_FCTN*keyword_int)(int, int*, int *);
      int (SYMBOLTABLE_TEMP_FCTN*keyword_double)(int, double*, int *);
      int (SYMBOLTABLE_TEMP_FCTN*keyword_string)(int, GString*, int *);
      int (SYMBOLTABLE_TEMP_FCTN*keyword_single)(int, int *);
      int (SYMBOLTABLE_TEMP_FCTN*keyword_stringfctn)(int, const char**, int *, int, int *);
   };
  // these set the data -- note functions and string functions are not included
   union {
     void *set_fctn; // set this to NULL if there is nothing else
     int (SYMBOLTABLE_TEMP_FCTN*set_keyword_int)(int, int*, int *);
     int (SYMBOLTABLE_TEMP_FCTN*set_keyword_double)(int, double*, int *);
     int (SYMBOLTABLE_TEMP_FCTN*set_keyword_string)(int, GString*, int *);
     int (SYMBOLTABLE_TEMP_FCTN*set_keyword_single)(int, int *, int *);
   };


   SymbolTableElement(void);  // need for use in a ResizeArray
   // for global functions (like 'cos')
   SymbolTableElement(void *new_fctn);
   // for member functions which have values (like 'resname')
   // first element in the functions are number of atoms, second is for the
   // data, and third the boolean flags indicating which to get/ set
   SymbolTableElement(int (SYMBOLTABLE_TEMP_FCTN*new_fctn)(int, int *, int *),
		      int (SYMBOLTABLE_TEMP_FCTN*set_fctn)(int, int *, int *));
   SymbolTableElement(int (SYMBOLTABLE_TEMP_FCTN*new_fctn)(int, double *, 
							   int *),
		      int (SYMBOLTABLE_TEMP_FCTN*set_fctn)(int, double *,
							   int *));
   SymbolTableElement(int (SYMBOLTABLE_TEMP_FCTN*new_fctn)(int, GString *, 
							   int *),
		      int (SYMBOLTABLE_TEMP_FCTN*set_fctn)(int, GString *,
							   int *));
   SymbolTableElement(int (SYMBOLTABLE_TEMP_FCTN*new_fctn)(int argc, 
						const char **argv,
						int *strtype,
						int num, int *flgs),
		      int (SYMBOLTABLE_TEMP_FCTN*set_fctn)(int argc, 
						const char **argv,
						int *strtype,
						int num, int *flgs));

   // for boolean member functions which modify the current selection 
   // (like 'waters'); first function gets, second sets
   SymbolTableElement( int (SYMBOLTABLE_TEMP_FCTN*new_fctn)(int, int *),
		       int (SYMBOLTABLE_TEMP_FCTN*set_fctn)(int, int*, int*));
};


// this tracks the names and functions needed to parse a selection
// for the given class
SYMBOLTABLE_TEMP_HDR
class SymbolTable {
// protected:
 public:
   ResizeArray<SymbolTableElement SYMBOLTABLE_TEMP_REF *> fctns;
   ResizeArray<SymbolTableName *> names;
 public:
   SymbolTable(void);
   virtual ~SymbolTable(void);

   // add functions and keywords ...
   int add_function(const char *pattern, const char *visible, void *fctn,
		    SymbolTableName::symtype takes,
		    SymbolTableName::symtype returns);
   // more could be added, if needed
   int add_function(const char *pattern, const char *visible,
		    double (*fctn)(double)) {
      return add_function(pattern, visible, fctn, SymbolTableName::IS_FLOAT,
			  SymbolTableName::IS_FLOAT);
   }
   int add_function(const char *pattern, const char *visible,
		    double (*fctn)(int)) {
      return add_function(pattern, visible, fctn, SymbolTableName::IS_FLOAT,
			  SymbolTableName::IS_INT);
   }



   // when keywords are called, they need to fill the passed array
   // the 'set' routines set from the passed array
   int add_keyword(const char *pattern, const char *visible,
	   int (SYMBOLTABLE_TEMP_FCTN*)(int num, int *data, int *flgs),
	   int (SYMBOLTABLE_TEMP_FCTN*)(int num, int *data, int *flgs));
   int add_keyword(const char *pattern, const char *visible,
	   int (SYMBOLTABLE_TEMP_FCTN*)(int num, double *data, int *flgs),
	   int (SYMBOLTABLE_TEMP_FCTN*)(int num, double *data, int *flgs));
   int add_keyword(const char *pattern, const char *visible,
	   int (SYMBOLTABLE_TEMP_FCTN*)(int num, GString *data, int *flgs),
	   int (SYMBOLTABLE_TEMP_FCTN*)(int num, GString *data, int *flgs));


   // these are for things like 'waters', which take no parameters
   int add_singleword(const char *pattern, const char *visible,
	      int (SYMBOLTABLE_TEMP_FCTN*)(int num, int *flgs),
	      int (SYMBOLTABLE_TEMP_FCTN*)(int num, int *data, int *flgs));

   // these are for "smart" selections which do all the parsing themselves
   // and _modify_ the flags
   // the "set" functions will probably never be used (there for
   // completeness, but never tested)
   int add_stringfctn(const char *pattern, const char *visible,
		      int (SYMBOLTABLE_TEMP_FCTN*)(int argc, 
						   const char **argv, 
						   int *types,
						   int num, int *flgs),
		      int (SYMBOLTABLE_TEMP_FCTN*)(int argc, 
						   const char **argv, 
						   int *types,
						   int num, int *flgs)
		      );

   // find keyword/function matching the name and return function index, or -1
   int find_attribute( char *attrib);

   // returns '1' if the variable can be changed/ modified
   int is_changeable(int fctnidx);

   // these return the data for a keyword, converting to the appropriate
   // type as needed, or 0 or "" if the function is not a keyword
   void extract_keyword_info(int fctnidx, int numatoms, int *iptr,
			     int *flgs, int sorted = 0,
			     SymbolTableName::symtype sort_as =
			     SymbolTableName::IS_INT);
   void extract_keyword_info(int fctnidx, int numatoms, double *dptr,
			     int *flgs, int sorted = 0,
			     SymbolTableName::symtype sort_as =
			     SymbolTableName::IS_FLOAT);
   void extract_keyword_info(int fctnidx, int numatoms, GString *sptr,
			     int *flgs, int sorted = 0,
			     SymbolTableName::symtype sort_as =
			     SymbolTableName::IS_STRING);

  // same as above, but I do the search for the fctn index and
  // return 0, or return 1
   int extract_keyword_info(char *name, int numatoms, int *iptr,
			    int *flgs, int sorted = 0,
			    SymbolTableName::symtype sort_as =
			    SymbolTableName::IS_INT);
   int extract_keyword_info(char *name, int numatoms, double *dptr,
			    int *flgs, int sorted = 0,
			    SymbolTableName::symtype sort_as =
			    SymbolTableName::IS_FLOAT);
   int extract_keyword_info(char *name, int numatoms, GString *sptr,
			    int *flgs, int sorted = 0,
			    SymbolTableName::symtype sort_as =
			    SymbolTableName::IS_STRING);

   // these set the data associated with a keyword
   // (this one is also used to set boolean values)
   void set_keyword_info(int fctnidx, int numatoms, int     *data, int *flgs);
   void set_keyword_info(int fctnidx, int numatoms, double  *data, int *flgs);
   void set_keyword_info(int fctnidx, int numatoms, GString *data, int *flgs);
   // and the named counterparts
   int set_keyword_info(char *name, int numatoms, int     *data, int *flgs);
   int set_keyword_info(char *name, int numatoms, double  *data, int *flgs);
   int set_keyword_info(char *name, int numatoms, GString *data, int *flgs);
   // this next two calls convert to GString and call the right fctn
   void set_keyword_info(int fctnidx, int numatoms, char **data, int *flgs);
   int set_keyword_info(char *name, int numatoms, char **data, int *flgs);
   
   // finally, this returns the ParseTree corresponding to a given
   // string, or NULL
   ParseTree SYMBOLTABLE_TEMP_REF *parse(char *s);
   ParseTree SYMBOLTABLE_TEMP_REF *parse(const char *s);

};

#endif

