/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: RadianceDisplayDevice.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 96/03/24 07:07:20 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *  Writes to the format for Radiance.  For more information about that
 * package, see http://radsite.lbl.gov/radiance/HOME.html .  It provides
 * conversion programs to go from its format to something normal (eg,
 * ra_ps, ra_tiff, and ra_gif).
 *
 ***************************************************************************/

#include <stdio.h>
#include <string.h>
#include <math.h>
#include "RadianceDisplayDevice.h"
#include "Stack.h"
#include "Matrix4.h"
#include "DispCmds.h"
#include "Inform.h"
#include "utilities.h"

#define DEFAULT_RADIUS 0.002
#define DASH_LENGTH 0.02

// Be careful when you modify the coordinates.  To make things view the
// right way, I have to rotate everything around the (x,y,z) = (1,1,1)
// vector so that x->z, y->x, and z->y

#define ORDER(x,y,z) -z, -x, y

///////////////////////// constructor and destructor

// constructor ... initialize some variables
RadianceDisplayDevice::RadianceDisplayDevice(char *nm) : FileRenderer(nm) {
	
    MSGDEBUG(1,"Creating RadianceDisplayDevice ..." << sendmsg);
    
    strcpy(renderCommand,"oconv %s > %s.oct; rview -pe 100 -vp -3.5 0 0 "
	   "-vd 1 0 0 %s.oct");
    strcpy(fname,"plot.rad");
}
               
//destructor
RadianceDisplayDevice::~RadianceDisplayDevice(void) { }

///////////////////////// protected nonvirtual routines

// draw a point
void RadianceDisplayDevice::point(float spdata[]) {
  float vec[3];

  // transform the world coordinates
  (transMat.top()).multpoint3d(spdata, vec);
   
  // draw the sphere
  set_color(currentColor[0], currentColor[1], currentColor[2]);
  fprintf(rad_file, " sphere ball\n0\n0\n4"
	            "%7f %7f %7f %7f\n",
	            ORDER(vec[0], vec[1], vec[2]), 
	            float(lineWidth) * DEFAULT_RADIUS);
}

// draw a sphere
void RadianceDisplayDevice::sphere(float spdata[]) {
  
  float vec[3];
  float radius;
    
  // transform the world coordinates
  (transMat.top()).multpoint3d(spdata, vec);
  radius = scale_radius(spdata[3]);
   
  // draw the sphere
  set_color(currentColor[0], currentColor[1], currentColor[2]);
  fprintf(rad_file, " sphere ball\n0\n0\n4 "
	            "%7f %7f %7f %7f\n",
	            ORDER(vec[0], vec[1], vec[2]), 
	            radius);
}

// draw a line (cylinder) from a to b
void RadianceDisplayDevice::line(float *a, float*b) {
    int i, j, test;
    float dirvec[3], unitdirvec[3];
    float from[3], to[3], tmp1[3], tmp2[3];
    float len;
    
    if(lineStyle == ::SOLIDLINE ) {
  
        // transform the world coordinates
        (transMat.top()).multpoint3d(a, from);
        (transMat.top()).multpoint3d(b, to);
    
        // draw the cylinder
	set_color(currentColor[0], currentColor[1], currentColor[2]);
	fprintf(rad_file, " cylinder cyl\n0\n0\n7 ");
        fprintf(rad_file, "%7f %7f %7f ", 
		ORDER(from[0], from[1], from[2])); // first point
        fprintf(rad_file, "%7f %7f %7f ", 
		ORDER(to[0], to[1], to[2])); // second point
        fprintf(rad_file, "%7f\n", float(lineWidth)*DEFAULT_RADIUS); // radius
        
    } else if (lineStyle == ::DASHEDLINE ) {
        
         // transform the world coordinates
        (transMat.top()).multpoint3d(a, tmp1);
        (transMat.top()).multpoint3d(b, tmp2);

        // how to create a dashed line
        for(i=0;i<3;i++) {
            dirvec[i] = tmp2[i] - tmp1[i];  // vector from a to b
        }
        len = sqrtf( dirvec[0]*dirvec[0] + dirvec[1]*dirvec[1] + dirvec[2]*dirvec[2] );
        for(i=0;i<3;i++) {
            unitdirvec[i] = dirvec[i] / sqrtf(len);  // unit vector pointing from a to b
        }
           
        test = 1;
        i = 0;

        while( test == 1 ) {
            for(j=0;j<3;j++) {
                from[j] = tmp1[j] + (2*i)*DASH_LENGTH*unitdirvec[j];
                to[j] = tmp1[j] + (2*i + 1)*DASH_LENGTH*unitdirvec[j];
            }
            if( fabsf(tmp1[0] - to[0]) >= fabsf(dirvec[0]) ) {
                for(j=0;j<3;j++) {
                    to[j] = tmp2[j];
                }
                test = 0;
            }
    
            // draw the cylinder
	    set_color(currentColor[0], currentColor[1], currentColor[2]);
	    fprintf(rad_file, " cylinder cyl\n0\n0\n7 ");
	    // first point
	    fprintf(rad_file, "%7f %7f %7f ", 
		    ORDER(from[0], from[1], from[2]));
	    // second point
	    fprintf(rad_file, "%7f %7f %7f ", 
		    ORDER(to[0], to[1], to[2])); 
	    // radius
	    fprintf(rad_file, "%7f\n", float(lineWidth)*DEFAULT_RADIUS); 

            i++;
        }
    } else {
        msgErr << "RadianceDisplayDevice: Unknown line style " << lineStyle << sendmsg;
    }

}

// draw a cylinder
void RadianceDisplayDevice::cylinder(float *a, float *b, float r) {

  float vec1[3], vec2[3];
  
  // transform the world coordinates
  (transMat.top()).multpoint3d(a, vec1);
  (transMat.top()).multpoint3d(b, vec2);
    
  // draw the cylinder

  set_color(currentColor[0], currentColor[1], currentColor[2]);
  fprintf(rad_file, " cylinder cyl\n0\n0\n7 ");
  // first point
  fprintf(rad_file, "%7f %7f %7f ", 
	  ORDER(vec1[0], vec1[1], vec1[2]));
  // second point
  fprintf(rad_file, "%7f %7f %7f ", 
	  ORDER(vec2[0], vec2[1], vec2[2])); 
  // radius
  fprintf(rad_file, "%7f\n", scale_radius(r));

  // and fill in the ends
  float normal[3];
  subtract(normal, vec1, vec2);
  normalize(normal);

  // one end
  set_color(currentColor[0], currentColor[1], currentColor[2]);
  fprintf(rad_file, " ring cyl_end\n0\n0\n8 ");
  fprintf(rad_file, "%7f %7f %7f ",         // location
	  ORDER(vec1[0], vec1[1], vec1[2]));
  fprintf(rad_file, "%7f %7f %7f ",         // normal
	  ORDER(normal[0], normal[1], normal[2]));
  fprintf(rad_file, "0 %7f\n", scale_radius(r)); // radii

  // the other end
  normal[0] = -normal[0];
  normal[1] = -normal[1];
  normal[2] = -normal[2];
  set_color(currentColor[0], currentColor[1], currentColor[2]);
  fprintf(rad_file, " ring cyl_end\n0\n0\n8 ");
  fprintf(rad_file, "%7f %7f %7f ",         // location
	  ORDER(vec2[0], vec2[1], vec2[2]));
  fprintf(rad_file, "%7f %7f %7f ",         // normal
	  ORDER(normal[0], normal[1], normal[2]));
  fprintf(rad_file, "0 %7f\n", scale_radius(r)); // radii
  
}

// draw a cone
void RadianceDisplayDevice::cone(float *a, float *b, float r) {

  float vec1[3], vec2[3];
  
  // transform the world coordinates
  (transMat.top()).multpoint3d(a, vec1);
  (transMat.top()).multpoint3d(b, vec2);
    
  set_color(currentColor[0], currentColor[1], currentColor[2]);
  fprintf(rad_file, " cone a_cone\n0\n0\n8 ");
  // first point
  fprintf(rad_file, "%7f %7f %7f ", 
	  ORDER(vec1[0], vec1[1], vec1[2]));
  // second point
  fprintf(rad_file, "%7f %7f %7f ", 
	  ORDER(vec2[0], vec2[1], vec2[2])); 
  // radius
  fprintf(rad_file, "%7f %7f\n", scale_radius(r), 0.0);
}

// draw a triangle
void RadianceDisplayDevice::triangle(float *a, float *b, float *c, float *, 
float *, float *) {

  float vec1[3], vec2[3], vec3[3];
  
  // transform the world coordinates
  (transMat.top()).multpoint3d(a, vec1);
  (transMat.top()).multpoint3d(b, vec2);
  (transMat.top()).multpoint3d(c, vec3);

  // draw the triangle

  set_color(currentColor[0], currentColor[1], currentColor[2]);

  fprintf(rad_file, " polygon poly\n0\n0\n9 "); // triangle
  fprintf(rad_file, "%7f %7f %7f ", 
	  ORDER(vec1[0], vec1[1], vec1[2])); // point one
  fprintf(rad_file, "%7f %7f %7f ", 
	  ORDER(vec2[0], vec2[1], vec2[2])); // point two
  fprintf(rad_file, "%7f %7f %7f\n", 
	  ORDER(vec3[0], vec3[1], vec3[2])); // point three
}

// draw a square
void RadianceDisplayDevice::square(float *, float *a, float *b, float *c, float *d) {
  
  float vec1[3], vec2[3], vec3[3], vec4[3];
  
  // transform the world coordinates
  (transMat.top()).multpoint3d(a, vec1);
  (transMat.top()).multpoint3d(b, vec2);
  (transMat.top()).multpoint3d(c, vec3);
  (transMat.top()).multpoint3d(d, vec4);

  // draw the square

  set_color(currentColor[0], currentColor[1], currentColor[2]);

  fprintf(rad_file, " polygon poly\n0\n0\n12 "); // triangle
  fprintf(rad_file, "%7f %7f %7f ", 
	  ORDER(vec1[0], vec1[1], vec1[2])); // point one
  fprintf(rad_file, "%7f %7f %7f ", 
	  ORDER(vec2[0], vec2[1], vec2[2])); // point two
  fprintf(rad_file, "%7f %7f %7f ", 
	  ORDER(vec3[0], vec3[1], vec3[2])); // point three
  fprintf(rad_file, "%7f %7f %7f\n", 
	  ORDER(vec4[0], vec4[1], vec4[2])); // point four

  
}

///////////////////// the color routines

// given the color, see if it has been made before and, if not, create 
// a new color index.  When finished, print the color index
void RadianceDisplayDevice::set_color(float r, float g, float b)
{
  int num = red.num();
  for (int i=0; i<num; i++) {
    if (r == red[i] && g == green[i] && b == blue[i]) {
      break;
    }
  }
  if (i == num) { // create a new color category
    red.append(r);
    blue.append(b);
    green.append(g);
    // define it for radiance
    fprintf(rad_file, "void plastic color%d\n0\n0\n5 ", i);
    fprintf(rad_file, "%f %f %f .05 .05\n", r, g, b);
  }
  // the color is 'i' so print it
  fprintf(rad_file, "color%d ", i);
}

///////////////////// public virtual routines

// initialize the file for output
int RadianceDisplayDevice::prepare3D(int ) {

  if(!(rad_file = fopen(fname,"w"))) {
    msgErr << "Cannot open file " << fname << " for Radiance output" << sendmsg;
    Initialized = FALSE;
  } else {
  
    Initialized = TRUE;
    rad_filename = stringdup(fname);

    // clear out the r/g/b arrays
    red.remove(-1,-1);
    green.remove(-1,-1);
    blue.remove(-1,-1);
    
    // write the light sources
    fprintf(rad_file, "void dielectric invisible\n0\n0\n5 1 1 1 1 0\n");
    //    fprintf(rad_file, "void light bright\n0\n0\n3 100 100 100\n");
    fprintf(rad_file, "void illum bright\n1 invisible\n0\n"
	    "3 10000 10000 10000\n");
    
    // do this instead of the right way (see later)
    fprintf(rad_file, "bright sphere fixture\n0\n0\n4  -10 0 0  .01\n");
    
    // background color is black until I figure out how to set it
    // interactively.  I'm thinking of having a glowing sphere or plane


    /* The light code doesn't work because at this point I don't have the
       correct transformation matrix, so I'll leave only the one light source
       defined above. */
    /*
    {
      float vec[3];
      for (int i=0; i<DISP_LIGHTS; i++) {
	if (lightDefined[i] && lightOn[i]) {
	  msgInfo << "Light is at: " << i << " " << lightPos[i][0]
		  << " " << lightPos[i][1] << " " << lightPos[i][2] << sendmsg;
	  (transMat.top()).multpoint3d(lightPos[i], vec);
	  fprintf(rad_file,
		  "bright sphere fixture\n0\n0\n4 %f %f %f .01\n",
		  ORDER( 10 * vec[0], 10 * vec[1], 10 * vec[2]));
	}
      }
    }
    */
  }
  return Initialized;
}

    
// clean up after yourself
void RadianceDisplayDevice::update(int) {

  if(Initialized) {
    fclose(rad_file);
    if(rad_filename) delete [] rad_filename;
    msgInfo << "Radiance file generation finished" << sendmsg;
  }
}

