/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: ParseTree.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.10 $	$Date: 96/03/23 05:09:43 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Given the parse tree created by a SymbolTable, evaluate it and return
 * the selection
 *
 ***************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>     // for pow
#include "AtomParser.h"  // for atomparser_node definition
#include "y.tab.h"  // for FLOAT, INT, and WORD
#include "ParseTree.h"
#include "Inform.h"

// this was converted from a templated version; see ParseTree.h

const char *parsetree_nodestring;

// do the string and numeric compares
#define compare_numeric_macro(switchcase, symbol)			      \
   case switchcase:							      \
   l->convert(SymbolTableName::IS_FLOAT);				      \
   r->convert(SymbolTableName::IS_FLOAT);				      \
   ldval = l->dval;							      \
   rdval = r->dval;							      \
   flg = flgs;								      \
   for (i=num-1; i>=0; i--) {					      \
      *flg = *flg && (*ldval symbol *rdval);			              \
      ldval++; rdval++; flg++;		                                      \
   }									      \
break;

#define compare_string_macro(switchcase, symbol)			      \
   case switchcase:							      \
   l->convert(SymbolTableName::IS_STRING);				      \
   r->convert(SymbolTableName::IS_STRING);				      \
   lsptr = l->sval;							      \
   rsptr = r->sval;							      \
   flg = flgs;								      \
   for (i=num-1; i>=0; i--) {					              \
      *flg = *flg && (*lsptr symbol *rsptr);			              \
      lsptr++; rsptr++; flg++;		                                      \
   }									      \
break;
   

///////////////// the ParseTree
PARSETREE_TEMP_HDR
ParseTree PARSETREE_TEMP_REF::
ParseTree(/*const*/ SymbolTable PARSETREE_TEMP_REF *parser, atomparser_node *parse_tree)
{
   tree = parse_tree;
   table = parser;
   selected_array = NULL;
   num_selected = 0;
}
PARSETREE_TEMP_HDR
ParseTree PARSETREE_TEMP_REF::
~ParseTree(void)
{
   if (selected_array != NULL) delete [] selected_array;
   delete tree;
}

PARSETREE_TEMP_HDR   
void ParseTree PARSETREE_TEMP_REF::eval_compare(atomparser_node *node, int num, int *flgs)
{
   int i;
   double *ldval, *rdval;
   GString *lsptr, *rsptr;
   int *flg;
   // get the data on the left and right
   symbol_data *l = eval(node->left, num, flgs);
   symbol_data *r = eval(node->right, num, flgs);

//   printf("--1> %d %d\n", l->num, r->num);
   l->extend(num);  // FIX THIS! (don't need lots of check if 1 element)
   r->extend(num);
//   printf("--2> %d %d\n", l->num, r->num);
   
   switch(node->ival) {
      compare_numeric_macro(NLT, < )
      compare_numeric_macro(NLE, <= )
      compare_numeric_macro(NEQ, == )
      compare_numeric_macro(NGE, >= )
      compare_numeric_macro(NGT, > )
      compare_numeric_macro(NNE, != )
      compare_string_macro(SLT, < )
      compare_string_macro(SLE, <= )
      compare_string_macro(SEQ, == )
      compare_string_macro(SGE, >= )
      compare_string_macro(SGT, > )
      compare_string_macro(SNE, != )
    case MATCH: {
       l->convert(SymbolTableName::IS_STRING);
       r->convert(SymbolTableName::IS_STRING);
       lsptr = l->sval;
       rsptr = r->sval;
       flg = flgs;
       symboltable_regex_reset();
       symboltable_regex_swap();
       Regex *r = NULL;
       GString first;
       if (num > 0) {
	  first = *rsptr;
       }
       for (i=0; i<num; i++) {
	  if (i==0 || *rsptr != first) {
	     if (r) delete r;
	     r = new Regex(*rsptr);
	     if (!symboltable_regex_check()) {
		msgErr << "Bad regex, ignoring it: " << (const char *)(*rsptr)
		       << sendmsg;
		delete r;
		r = NULL;
	     }
	     first = *rsptr;
	  }
	  if (r) {
	     *flg = *flg && (r->match(*lsptr, strlen(*lsptr)) != -1);
	  } else {
	     *flg = 0;
	  }
	  lsptr++; rsptr++; flg++;
	  
       }
       if (r) {
	  delete r;
       }
       symboltable_regex_swap();
       break;
       // done with match search
    }
    default:
      printf("All operators should have been defined!\n");
   }
   delete l;
   delete r;
}

// place to do +, -, *, and /
PARSETREE_TEMP_HDR   
symbol_data *
ParseTree PARSETREE_TEMP_REF::eval_mathop(atomparser_node *node, int num, int *flgs)
{
   symbol_data *l = eval(node->left, num, flgs);
   symbol_data *r = eval(node->right, num, flgs);
   l->extend(num);
   r->extend(num);
   l->convert(SymbolTableName::IS_FLOAT);
   r->convert(SymbolTableName::IS_FLOAT);
   symbol_data *tmp = new symbol_data(SymbolTableName::IS_FLOAT, num);
   int i;
   switch (node->node_type) {
    case ADD:
      for (i=num-1; i>=0; i--) {
	 if (flgs[i]) tmp->dval[i] = l->dval[i] + r->dval[i];
      }
      break;
    case SUB:
      for (i=num-1; i>=0; i--) {
	 if (flgs[i]) tmp->dval[i] = l->dval[i] - r->dval[i];
      }
      break;
    case MULT:
      for (i=num-1; i>=0; i--) {
	 if (flgs[i]) tmp->dval[i] = l->dval[i] * r->dval[i];
      }
      break;
    case DIV:
      for (i=num-1; i>=0; i--) {
	 if (flgs[i]) tmp->dval[i] = l->dval[i] / r->dval[i];
      }
      break;
    case MOD:  // fake mod
      for (i=num-1; i>=0; i--) {
	 if (flgs[i]) {
	    tmp->dval[i] = fmod(l->dval[i],r->dval[i]);
	 }
      }
      break;
    case EXP:
      for (i=num-1; i>=0; i--) {
	 if (flgs[i]) {
	    tmp->dval[i] = pow(l->dval[i], r->dval[i]);
	 }
      }
   }
       
   delete l;
   delete r;
   return tmp;
}

// to make life easier on the reader..
typedef int (*int_fctn_int)(int);
typedef int (*int_fctn_double)(double);
typedef int (*int_fctn_string)(const GString& );
typedef double (*double_fctn_int)(int);
typedef double (*double_fctn_double)(double);
typedef double (*double_fctn_string)(const GString&);
typedef GString (*string_fctn_int)(int);
typedef GString (*string_fctn_double)(double);
typedef GString (*string_fctn_string)(const GString&);

// simple little macro for doing the loop
#define eval_fctn_macro for (i=num-1; i>=0; i--) if (flgs[i])

PARSETREE_TEMP_HDR   
symbol_data *ParseTree PARSETREE_TEMP_REF::eval_function( atomparser_node *node, int num,
				       int *flgs)
{
   SymbolTableName::symtype takes_a = table->names[node->extra_type]->takes_a;
   SymbolTableName::symtype returns_a = table->names[node->extra_type]->returns_a;

   // get the values to give to the function
   symbol_data *inp = eval(node->left, num, flgs);

   // convert to the expected data type
   inp->convert(takes_a);

   // set up space for the return
   symbol_data *ret = new symbol_data(returns_a, num);
   
   // now do the evaluations (there are nine cases)
   parsetree_nodestring = (const char *) node->sele.s;
   void *fctn = table->fctns[node->extra_type]->fctn;
   int i;
   if (takes_a == SymbolTableName::IS_INT) {
      if (returns_a == SymbolTableName::IS_INT) {
	 eval_fctn_macro ret->ival[i] = ((int_fctn_int) fctn)(inp->ival[i]);
      } else if (returns_a == SymbolTableName::IS_FLOAT) {
	 eval_fctn_macro ret->dval[i] = ((double_fctn_int) fctn)(inp->ival[i]);
      } else {
	 eval_fctn_macro ret->sval[i] = ((string_fctn_int) fctn)(inp->ival[i]);
      }
   } else if (takes_a == SymbolTableName::IS_FLOAT) {
      if (returns_a == SymbolTableName::IS_INT) {
	 eval_fctn_macro ret->ival[i] = ((int_fctn_double) fctn)(inp->dval[i]);
      } else if (returns_a == SymbolTableName::IS_FLOAT) {
	 eval_fctn_macro ret->dval[i] = ((double_fctn_double) fctn)
	                                                     (inp->dval[i]);
      } else {
	 eval_fctn_macro ret->sval[i] = ((string_fctn_double) fctn)
	                                                     (inp->dval[i]);
      }
   } else {
      if (returns_a == SymbolTableName::IS_INT) {
	 eval_fctn_macro ret->ival[i] = ((int_fctn_string) fctn)(inp->sval[i]);
      } else if (returns_a == SymbolTableName::IS_FLOAT) {
	 eval_fctn_macro ret->dval[i] = ((double_fctn_string) fctn)
	                                                     (inp->sval[i]);
      } else {
	 eval_fctn_macro ret->sval[i] = ((string_fctn_string) fctn)
	                                                     (inp->sval[i]);
      }
   }
   delete inp;
   return ret;
}

// This puts the task of doing the selection inside the function
// For example: sequence APW "T.*A"
// The function converts the linked list into an array of const char *
// and of fields,  0 == raw, 1 == single quote, 2 == double quote
// if this is the start of a "to" then the fields are
// and of fields,  3 == raw, 4 == single quote, 5 == double quote
// the function modifies the selection as it pleases, 
// so it _can_ override the current flags.  Please be careful.
PARSETREE_TEMP_HDR
void ParseTree PARSETREE_TEMP_REF::eval_stringfctn(atomparser_node *node,
				   int num, int *flgs)
{
   // first find out how many elements exist
   int count = 0;
   parsetree_nodestring = (const char *) node->sele.s;
   for (atomparser_node *left = node->left;
	left != NULL;
	left = left -> left) {
      count++;
   }
   if (count == 0) return;

   // now populate the pointers
   char **argv= (char **) malloc(count * sizeof(char *));
   int *types = new int[count];
   int i=0;
   for (left = node->left; left != NULL; left = left -> left, i++) {
      // get the string type (single, double, raw
      switch(left->sele.st)
      {
       case RAW_STRING: {
	  types[i] = 0;
	  argv[i] = (char *) ((const char *) left -> sele.s);
	  break;
       }
       case SQ_STRING: {
	  types[i] = 1;
	  argv[i] = (char *) ((const char *) left -> sele.s);
	  break;
       }
       case DQ_STRING: {
	  types[i] = 2;
	  argv[i] = (char *) ((const char *) left -> sele.s);
	  break;
       }
      }
      if (left->extra_type != -1) { // then it is a "through" search
	 types[i] += 3;
      }
   }

   // and call the function
   int *tmp_flgs = new int[num];
   memcpy(tmp_flgs, flgs, num * sizeof(int));
   parsetree_nodestring = (const char *) node->sele.s;
   SymbolTableElement PARSETREE_TEMP_REF *fctn = (table->fctns[node->extra_type]);
#if PARSETREE_TEMP == 1
   (obj->*(fctn->keyword_stringfctn))(count, argv, types,
				      num, tmp_flgs);
#else
#ifdef ARCH_AIX3
   (fctn->keyword_stringfctn)(count, (const char **)(void *)argv,
			      types, num, tmp_flgs);
#else
   (fctn->keyword_stringfctn)(count, argv, types, num, tmp_flgs);
#endif
#endif
   for (i = num-1; i>=0; i--) {
      if (flgs[i]) flgs[i] = tmp_flgs[i];
   }
   delete [] tmp_flgs;
   delete [] types;
   free(argv);
}


// find all the elements in 'flgs' which have the same "field" value
// as elements in 'subselect'
#define same_search_macro(field) 					      \
for (int i=0; i<num; i++) {         /* search everything */     	      \
   if (flgs[i]) {                   /* make sure it is already selected	*/    \
      flgs[i] = 0;							      \
      for (int j=0; j<num; j++) {   /* search the subselection	*/	      \
	 if (subselect[j]) {						      \
	    if (tmp->field[i] == tmp2->field[j]) {  /* is 'same' */	      \
	       flgs[i] = 1;						      \
	       break;							      \
	    }								      \
	 }								      \
      }									      \
   }									      \
}



// this does things like: same resname as name CA 
// 1) evalute the expression (m atoms)
// 2) get the keyword information (n atoms)
// 3) do an n*m search for the 'same' values
PARSETREE_TEMP_HDR
void ParseTree PARSETREE_TEMP_REF::eval_same( atomparser_node *node, int num, int *flgs)
{
   // 1) evaluate the sub-selection
   int *subselect = new int[num];
   {
      int *tmp = subselect;
      for (int i=num-1; i>=0; i--) {
	 *tmp++=1;
      }
      symbol_data *tmpret;
      if (tmpret = eval(node->left, num, subselect)) {
	 delete [] subselect;
	 msgErr << "eval of a 'same' returned data when it shouldn't have" << sendmsg;
	 return;
      }
   }
   // at this point, only the sub selection is defined
   
   // Step 2) get the keyword information
   
   // a) make space for the return type
   SymbolTableName::symtype has_type = table->names[node->extra_type]
                                         -> returns_a;
   symbol_data *tmp, *tmp2;
   tmp = new symbol_data(has_type, num);
   tmp2 = new symbol_data(has_type, num);
   
   // b) get the data (masked by the info passed by flgs)
   SymbolTableElement PARSETREE_TEMP_REF *fctn= (table->fctns[node->extra_type]);
   switch (has_type) {
    case SymbolTableName::IS_INT:   
#if PARSETREE_TEMP == 1
      (obj->*(fctn->keyword_int))(num, tmp->ival, flgs);
      (obj->*(fctn->keyword_int))(num, tmp2->ival, subselect);
#else
      (fctn->keyword_int)(num, tmp->ival, flgs);
      (fctn->keyword_int)(num, tmp2->ival, subselect);
#endif
      break;
    case SymbolTableName::IS_FLOAT: 
#if PARSETREE_TEMP == 1
      (obj->*(fctn->keyword_double))(num, tmp->dval, flgs); 
      (obj->*(fctn->keyword_double))(num, tmp2->dval, subselect); 
#else
      (fctn->keyword_double)(num, tmp->dval, flgs);
      (fctn->keyword_double)(num, tmp2->dval, subselect); 
#endif
      break;
    case SymbolTableName::IS_STRING:
#if PARSETREE_TEMP == 1
      (obj->*(fctn->keyword_string))(num, tmp->sval, flgs);
      (obj->*(fctn->keyword_string))(num, tmp2->sval, subselect);
#else
      (fctn->keyword_string)(num, tmp->sval, flgs);
      (fctn->keyword_string)(num, tmp2->sval, subselect);
#endif
      break;
   }
   
   // and Step 3) find the 'same' value
   {
      switch (has_type) {
       case SymbolTableName::IS_INT: {
	  same_search_macro(ival);
	  break;
       }
       case SymbolTableName::IS_FLOAT: {
	  same_search_macro(dval);
	  break;
       }
       case SymbolTableName::IS_STRING: {
	  same_search_macro(sval);
	  break;
       }
       } // end of switch
   }
   delete tmp;
   delete tmp2;
   delete [] subselect;
}

// here's where I get things like: name CA N C O
// and: mass
PARSETREE_TEMP_HDR
symbol_data *ParseTree PARSETREE_TEMP_REF::eval_key( atomparser_node *node, int num, int *flgs)
{
   // make space for the return type
   SymbolTableName::symtype has_type = table->names[node->extra_type]
                                         -> returns_a;
   symbol_data *tmp;
   tmp = new symbol_data(has_type, num);

   parsetree_nodestring = (const char *) node->sele.s;
   switch (has_type) {
    case SymbolTableName::IS_INT:
      table->extract_keyword_info(node->extra_type, num, tmp->ival, flgs);
      break;
    case SymbolTableName::IS_FLOAT:
      table->extract_keyword_info(node->extra_type, num, tmp->dval, flgs);
      break;
    case SymbolTableName::IS_STRING:
      table->extract_keyword_info(node->extra_type, num, tmp->sval, flgs);
      break;
   }

   // Now that I have the data, I can do one of two things
   // Either it is a list, in which case there is data off the
   // left, or it returns the data itself

   // if there is a list coming off the left, then I have
   // name CA N     ===> (name='CA' and name='N')
   // chain 1 to 3  ===> (name>='1' and name<='3'
   int *newflgs = new int[num];   // have to do this since the parameters
   for (int i=num-1; i>=0; i--) { // in the selection are 'or'ed together
      newflgs[i] = 0;
   }
			      
   if (node->left) {
      atomparser_node *left = node->left;
      while (left) {
	 if (left->extra_type == -1) { // then it is normal
//	    printf("Normal\n"); fflush(stdout);
	    switch(has_type) {
	     case SymbolTableName::IS_INT:
	       {
//		  printf("Int\n"); fflush(stdout);
		  int ival = atoi(left->sele.s);
		  for (int i=num-1; i>=0; i--) {
		     if (flgs[i]) newflgs[i] |= (ival == tmp->ival[i]);
		  }
	       }
	       break;
	     case SymbolTableName::IS_FLOAT:
	       {
//		  printf("Double\n"); fflush(stdout);
		  double dval = atof(left->sele.s);
		  double delta = dval / 1000;
		  if (delta < 0) delta = -delta;
		  for (int i=num-1; i>=0; i--) {
		     if (flgs[i]) newflgs[i] |= (dval-delta < tmp->dval[i] &&
						 dval+delta > tmp->dval[i]);
		  }
	       }
	       break;
//	     default: printf("Defaulting on a loan\n"); fflush(stdout);
	     case SymbolTableName::IS_STRING:
	       {
		  switch (left->sele.st) {
		   case SQ_STRING:
		   case RAW_STRING:
		     {
//			printf("Doing string as single quotes\n");
			for (int i=num-1; i>=0; i--) {
			   if (flgs[i]) {
			      newflgs[i] |= (left->sele.s == tmp->sval[i]);
			   }
			}
		     }
		     break;
		   case DQ_STRING:
		   default:
		     {
			symboltable_regex_reset();
			symboltable_regex_swap();
			// A regex like "H" would match 'H', 'H21',
			// 'OH2', etc.  I force the match to be
			// complete with the ^ and $.  The parenthesis \(\)
			// are to avoid turning C\|O into ^C\|O$
			// and the double \ is to escape the string escape
			// mechanism.  Ain't this grand?
			GString temps = "^\\("+left->sele.s+"\\)$";
			Regex r(temps, 1);  // 1 for fast compile
			char *str;
#ifndef ARCH_AIX3
			int i = symboltable_regex_check(&str);
#else
			int i = symboltable_regex_check(
				(const char **)(void *)(&str));
#endif
			symboltable_regex_swap();
			if (!i) {
			   printf("Bad regex, ignoring it: %s %s\n",
				  left->sele.s, str);
			   break;
			}
//			printf("Doing double.\n");
			for (i=num-1; i>=0; i--) {
			   if (flgs[i]) newflgs[i] |= (r.match(tmp->sval[i],
					strlen(tmp->sval[i])) != -1);
			} // end loop
		     } // end check for DQ_STRING
		  } // end based on string type
	       }
	    } // end based on keyword type
	 } else {  // do a 'through' search
	    switch(has_type) {
	     case SymbolTableName::IS_INT:
	       {
		  int ltval = atoi(left->sele.s);
		  int gtval = atoi(left->left->sele.s);
		  for (int i=num-1; i>=0; i--) {
		     if (flgs[i]) newflgs[i] |=
		                          (( ltval <= tmp->ival[i])
		                       && ( gtval >= tmp->ival[i]));
		  }
	       }
	       break;
	     case SymbolTableName::IS_FLOAT:
	       {
		  double ltval = atof(left->sele.s);
		  double gtval = atof(left->left->sele.s);
		  for (int i=num-1; i>=0; i--) {
		     if (flgs[i]) newflgs[i] |=
		                         (( ltval <= tmp->dval[i])
		                       && ( gtval >= tmp->dval[i]));
		  }
	       }
	       break;
	     default:
	       {
		  // no way to do regex with < or >, so do exact
		  for (int i=num-1; i>=0; i--) {
		     if (flgs[i]) newflgs[i] |=
		                (flgs[i] && ( left->sele.s <= tmp->sval[i])
		                && (left->left->sele.s >= tmp->sval[i]));

		  }
	       }
	    } // end checking type
	    left = left -> left;  // need to bypass that 2nd one
	 } // end both possible ways
	 left = left -> left;
      } // end going down the left side
      // get the flgs info back together
      {
	 for (int i=num-1; i>=0; i--) {
	    if (flgs[i]) flgs[i] = newflgs[i];
	 }
	 delete [] newflgs;
      }
      delete tmp;
      return NULL;
   } else {
      // if there isn't a list, then I have something like
      // mass + 5 < 7
      // so just return the data
      return tmp;
   }
}

PARSETREE_TEMP_HDR
void ParseTree PARSETREE_TEMP_REF::eval_single(atomparser_node *node, int num, int *flgs)
{
   SymbolTableElement PARSETREE_TEMP_REF *fctn = table->fctns[node->extra_type];
   parsetree_nodestring = (const char *) node->sele.s;
#if PARSETREE_TEMP == 1
   (obj->*(fctn->keyword_single))(num, flgs);
#else
   (fctn->keyword_single)(num, flgs);
#endif
}

PARSETREE_TEMP_HDR
void ParseTree PARSETREE_TEMP_REF::eval_within(atomparser_node *node, int num, int *flgs)
{
   // find the atoms in the rest of the selection
   int *others = new int[num];
   int *tmp = others;
   for (int i=num-1; i>=0; i--) {
      *tmp++ = 1;
   }
   symbol_data *tmpret;
   if (tmpret = eval(node->left, num, others)) {
      delete [] others;
      printf("eval of a 'within' returned data when it shouldn't have\n");
      return;
   }
   
   // get the X, Y, and Z coordinates
   int x = -1, y = -1, z = -1;
   // which means I need to find the elements that return 'x', 'y', and 'z'
   for (i=table->names.num()-1; i>=0; i--) {
      if (table->names[i]->regex->match("x", 1) != -1) x = i;
      if (table->names[i]->regex->match("y", 1) != -1) y = i;
      if (table->names[i]->regex->match("z", 1) != -1) z = i;
   }
   if (x == -1) {
      msgErr << "Couldn't find 'x' coordinates in 'within' search!" << sendmsg;
   }
   if (y == -1) {
      msgErr << "Couldn't find 'y' coordinates in 'within' search!" << sendmsg;
   }
   if (z == -1) {
      msgErr << "Couldn't find 'z' coordinates in 'within' search!" << sendmsg;
   }
   if (x == -1 || y == -1 || z == -1) {
      return;
   }
   if (table->names[x]->returns_a != SymbolTableName::IS_FLOAT) {
      msgErr << "'x' needs to return a double for 'within' search!" << sendmsg;

   }
   if (table->names[y]->returns_a != SymbolTableName::IS_FLOAT) {
      msgErr << "'y' needs to return a double for 'within' search!" << sendmsg;
      return;
   }
   if (table->names[z]->returns_a != SymbolTableName::IS_FLOAT) {
      msgErr << "'z' needs to return a double for 'within' search!" << sendmsg;
      return;
   }
   // Now I need to get the coordinates
   int *tmp2 = new int[num];
   SymbolTableElement PARSETREE_TEMP_REF *fctn = table->fctns[node->extra_type];

   // for x
//   printf("Getting 'x'\n"); fflush(stdout);
   for (i=num-1, tmp = tmp2; i>=0; i--) *tmp++ = 1;  // reset the flags
   // (this should to be reset every time since I cannot guarentee the
   // function won't change it)
   fctn = table->fctns[x];                            // get the 'x' function
   double *xcoord = new double[num];                 // space for results
#if PARSETREE_TEMP == 1
   (obj->*(fctn->keyword_double))(num, xcoord, tmp2);// and get the coords
#else
   (fctn->keyword_double)(num, xcoord, tmp2);
#endif
   // for y
//   printf("Getting 'y'\n"); fflush(stdout);
   for (i=num-1, tmp = tmp2; i>=0; i--) *tmp++ = 1;
   fctn = table->fctns[y];
   double *ycoord = new double[num];
#if PARSETREE_TEMP == 1
   (obj->*(fctn->keyword_double))(num, ycoord, tmp2);
#else
   (fctn->keyword_double)(num, ycoord, tmp2);
#endif
   // for z
//   printf("Getting 'z'\n"); fflush(stdout);
   for (i=num-1, tmp = tmp2; i>=0; i--) *tmp++ = 1;
   fctn = table->fctns[z];
   double *zcoord = new double[num];
#if PARSETREE_TMP == 1
   (obj->*(fctn->keyword_double))(num, zcoord, tmp2);
#else
   (fctn->keyword_double)(num, zcoord, tmp2);
#endif   
   delete [] tmp2;

   // now find the atoms in flgs which are within the distance of other
   double dist, dx;
   double r2 = node->dval;
   r2 *= r2;
//   printf("radius^2 is %lf\n", r2); fflush(stdout);
   for (i=num-1; i>=0; i--) {
      if (flgs[i]) {
	 flgs[i] = 0;
	 for (int j=num-1; j>=0; j--) {
	    if (others[j]) {
	       // now need to do a distance search
	       dist = xcoord[i] - xcoord[j]; dist *= dist;
	       if (dist > r2) continue;
	       dx = ycoord[i] - ycoord[j]; dist += dx * dx;
	       if (dist > r2) continue;
	       dx = zcoord[i] - zcoord[j]; dist += dx * dx;
	       if (dist > r2) continue;
	       flgs[i] = 1;
	       break;
	    }
	 } // inner loop (j)
      }
   } // outer loop (i)
   delete [] others;
   delete [] xcoord;
   delete [] ycoord;
   delete [] zcoord;
//   printf("Done within\n"); fflush(stdout);
   
}

// a node of the tree merges symbol_datas
// a leaf of the tree produces symbol_datas
PARSETREE_TEMP_HDR
symbol_data *ParseTree PARSETREE_TEMP_REF::eval(atomparser_node *node, int num, int *flgs)
{
   int i;
   int *flg1, *flg2;
   symbol_data *tmp;
   fflush(stdout);
   switch(node->node_type) {
    case AND:
      eval(node->left, num, flgs);  // implicit 'and'
      eval(node->right, num, flgs);
      return NULL;
    case NOT:
      {
	 flg1 = new int[num];
	 memcpy(flg1, flgs, num*sizeof(int));
	 eval(node->left, num, flg1);
	 // this gives: A and B
	 // I want A and (not B)
	 for (i=num-1; i>=0; i--) {
	    if (flgs[i]) flgs[i] = !flg1[i];
	 }
	 delete [] flg1;
	 break;
      }
    case OR:
      flg1 = new int[num];
      memcpy(flg1, flgs, num*sizeof(int));
      eval(node->left, num, flg1);
      flg2 = new int[num];
      memcpy(flg2, flgs, num*sizeof(int));
      eval(node->right, num, flg2);
      for (i=num-1; i>=0; i--) {
	 flgs[i] = flgs[i] && (flg1[i] || flg2[i]);
      }
      delete [] flg1;
      delete [] flg2;
      break;
    case FLOAT:
      tmp = new symbol_data(SymbolTableName::IS_FLOAT, 1);
      tmp->dval[0] = node->dval;
      return tmp;
    case INT:
      tmp = new symbol_data(SymbolTableName::IS_INT, 1);
      tmp->ival[0] = node->ival;
      return tmp;
    case WORD:
      tmp = new symbol_data(SymbolTableName::IS_STRING, 1);
      tmp->sval[0] = node->sele.s;
      return tmp;
    case KEY: return eval_key( node, num, flgs);
    case STRFCTN: eval_stringfctn( node, num, flgs); break;
    case FUNC:{
       tmp = eval_function( node, num, flgs);
//       for (int i=0; i<num; i++) {
//	  printf("%lf\n", tmp->dval[i]);
//       }
       return tmp;
   }
    case ADD:
    case SUB:
    case MULT:
    case MOD:
    case EXP:
    case DIV: return eval_mathop(node, num, flgs);
    case UMINUS: {
      tmp = eval(node->left, num, flgs);
      tmp->convert(SymbolTableName::IS_FLOAT);
      for (int i=0; i<tmp->num; i++) {
	 tmp->dval[i] = -tmp->dval[i];
      }
      return tmp;
    }
    case COMPARE:
      eval_compare(node, num, flgs);
      break;
    case WITHIN:  // this gets the coordinates from 'x', 'y', and 'z'
      eval_within(node, num, flgs);
      break;
    case SAME:
      eval_same(node, num, flgs);
      break;
    case SINGLE:
      eval_single(node, num, flgs);
      break;
    default: printf("WHAT!! I should know that parse tree element.%d\n",
		    node->node_type);
      break;
   }
   return NULL;
}

// this will create a list of flags, then call eval
// returns NULL if things went bad
// or the array either set to 1 (if selected) or 0.
// this array is static, so don't deallocate it
PARSETREE_TEMP_HDR
int *ParseTree PARSETREE_TEMP_REF::evaluate(
#if PARSETREE_TEMP == 1
/*const*/ T &new_obj
#else
int num_atoms
#endif
                                             )
{
#if PARSETREE_TEMP == 1   
   obj = new_obj;        // save the pointer so I call the correct class
   int num = new_obj->num_atoms();
#else
   int num = num_atoms;
#endif
   if (!tree || num < 0 ) {  // yes, I allow 0 atoms
      return NULL;
   }

   if (num_selected < num) {
      if (num == 0) num_selected = 1; else num_selected = num;
      if (selected_array) delete [] selected_array;
      selected_array = new int[num_selected];
   }
   if (num == 0) return selected_array;  // but I don't do anything with it
   for (int i=0; i<num; i++) {
      selected_array[i] = 1;
   }
   // things should never return data so complain if that happens
   symbol_data *retdat = eval(tree, num, selected_array);
   if (retdat) {
      printf("Atom selection returned data when it shouldn't\n");
      fflush(stdout);
      delete retdat;
   }
   return selected_array;
}


////////////////
// this will set a list of flags, then call eval on that array
// it returns 0 if things went bad, 1 otherwise
// the array either set to 1 (if selected) or 0.
PARSETREE_TEMP_HDR
int ParseTree PARSETREE_TEMP_REF::evaluate(
#if PARSETREE_TEMP == 1
/*const*/ T &new_obj,
#else
int num_atoms,
#endif
                int *flgs)
{
#if PARSETREE_TEMP == 1   
   obj = new_obj;        // save the pointer so I call the correct class
   int num = new_obj->num_atoms();
#else
   int num = num_atoms;
#endif
   if (!tree || num < 0 ) {  // yes, I allow 0 atoms
      return NULL;
   }

   for (int i=0; i<num; i++) {
      flgs[i] = 1;
   }
   // things should never return data so complain if that happens
   symbol_data *retdat = eval(tree, num, flgs);
   if (retdat) {
      printf("Atom selection returned data when it shouldn't\n");
      fflush(stdout);
      delete retdat;
   }
   return 1;
}


///////////////// the symbol_data
symbol_data::symbol_data(void) {
   num = 0;
   dval = NULL;
   ival = NULL;
   sval = NULL;
}

void symbol_data::make_space(void) {  // delete and recreate the data space
   free_space();
   switch(type) {
    case SymbolTableName::IS_FLOAT:
      dval = new double[num];
      if (!dval) {
 msgErr << "Out of memory in ParseTree.C: symbol_data::make_space float"
		<< sendmsg;
      }
      break;
    case SymbolTableName::IS_INT:
      ival = new int[num];
      if (!ival) {
 msgErr << "Out of memory in ParseTree.C: symbol_data::make_space int"
		<< sendmsg;
      }
      break;
    case SymbolTableName::IS_STRING:
      sval = new GString[num];
      if (!sval) {
 msgErr << "Out of memory in ParseTree.C: symbol_data::make_space GString"
		<< sendmsg;
      }
      break;
   }
}
void symbol_data::free_space(void) {  // just delete the space
   switch(type) {
    case SymbolTableName::IS_FLOAT: {
       if (dval) delete [] dval;
       dval = NULL;
       break;
    }
    case SymbolTableName::IS_INT: {
       if (ival) delete [] ival;
       ival = NULL;
       break;
    }
    case SymbolTableName::IS_STRING: {
       if (sval) delete [] sval;
       sval = NULL;
       break;
    }
    default: {
       msgErr << "Unknown data type " << type
	      << " in symbol_data::free_space"
	      << sendmsg;
    }
    }
}
// given the new type and the number of elements, create space
symbol_data::symbol_data(SymbolTableName::symtype new_type, int new_num) {
   type = new_type;
   num = new_num;
   dval = NULL;
   ival = NULL;
   sval = NULL;
   make_space();
}
symbol_data::~symbol_data(void) {
   free_space();
}
void symbol_data::extend(int newnum) {
   // copy the first element until there are newnum elements
   if (num != 1) {
      if (num != newnum) {
	 printf("Trying to extend from %d to %d isn't allowed\n",
		num, newnum);
      }
      return;
   }
   switch (type) {
    case SymbolTableName::IS_STRING: {
       GString tmp = sval[0];
       free_space();
       num = newnum;
       make_space();
       for (int i=num-1; i>=0; i--) {
	  sval[i] = tmp;
       }
       break;
    }
    case SymbolTableName::IS_INT: {
       int tmp = ival[0];
       free_space();
       num = newnum;
       make_space();
       for (int i=num-1; i>=0; i--) {
	  ival[i] = tmp;
       }
       break;
    }	  
    case SymbolTableName::IS_FLOAT: {
       double tmp = dval[0];
       free_space();
       num = newnum;
       make_space();
       for (int i=num-1; i>=0; i--) {
	  dval[i] = tmp;
       }
       break;
    }	  
    } // switch
} // extend


// convert from a node with an intrinsic value (it is an in, or whatever)
// to a symbol_data array of length 1
symbol_data::symbol_data(atomparser_node *node) { 
   switch(node->node_type) {
    case INT: type = SymbolTableName::IS_INT; break;
    case FLOAT: type = SymbolTableName::IS_FLOAT; break;
    case WORD: type = SymbolTableName::IS_STRING; break;
    default:
      printf("Major wierdness in parser: unknown data type %d: "
	     "assuming string\n",
	     node->node_type);
      type = SymbolTableName::IS_STRING;
      break;
   }
   num = 1; 
   dval = NULL;
   ival = NULL;
   sval = NULL;
   make_space();
   switch(type) {
    case SymbolTableName::IS_FLOAT: { dval[0] = node->dval; break; }
    case SymbolTableName::IS_INT: { ival[0] = node->ival; break; }
    case SymbolTableName::IS_STRING: { sval[0] = node->sele.s; break; }
   }
}

void symbol_data::convert(SymbolTableName::symtype totype) {
//   msgInfo << "fromtype == " << type << " totype == " << totype << sendmsg;
   if (totype == type) return;
   if (totype == SymbolTableName::IS_FLOAT) {
      double *tmp = new double[num];
      if ( type == SymbolTableName::IS_INT) {
	 for (int i=num-1; i>=0; i--) {
	    tmp[i] = (double) ival[i];
	 }
      } else { // SymbolTableName::IS_STRING
	 for (int i=num-1; i>=0; i--) {
	    tmp[i] = atof(sval[i]);
	 }
      }
      free_space();
      type = totype;
      dval = tmp;
      return;
   }
   if (totype == SymbolTableName::IS_STRING) {
      GString *tmp = new GString[num];
      char s[100];
      if (type == SymbolTableName::IS_INT) {
	 for (int i=num-1; i>=0; i--) {
	    sprintf(s, "%ld", (long) ival[i]);
	    tmp[i] = s;
	 }
      } else { // SymbolTableName::IS_FLOAT
	 for (int i=num-1; i>=0; i--) {
	    sprintf(s, "%lf", (double) dval[i]);
	    tmp[i] = s;
	 }
      }
      free_space();
      type = totype;
      sval = tmp;
      return;
   }
   if (totype == SymbolTableName::IS_INT) {
      int *tmp = new int[num];
      if (type == SymbolTableName::IS_FLOAT) {
	 for (int i=num-1; i>=0; i--) {
	    tmp[i] = (int) dval[i];
	 }
      } else{
	 for (int i=num-1; i>=0; i--) {
	    tmp[i] = atoi(sval[i]);
	 }
      }
      free_space();
      type = totype;
      ival = tmp;
      return;
   }
}

