/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: OpenGLRenderer.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 96/02/03 02:33:51 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Subclass of DisplayDevice, this object has routines used by all the
 * different display devices that use OpenGL for rendering.
 * Will render drawing commands into a window.
 * This is not the complete definition,
 * however, of a DisplayDevice; something must provide routines to open
 * windows, reshape, clear, set perspective, etc.  This object contains the
 * code to render a display command list.
 ***************************************************************************/

#include "OpenGLRenderer.h"
#include "DispCmds.h"
#include "Inform.h"
#include "utilities.h"
#include <math.h>


/////////////////////////  constructor and destructor  

// constructor ... initialize some variables
OpenGLRenderer::OpenGLRenderer(char *nm) : DisplayDevice(nm) {

  MSGDEBUG(1,"Creating OpenGLRenderer ..." << sendmsg);

  // initialize data
  need_aaon = need_cueon = FALSE;
  dataBlock = NULL;
  screenX = screenY = 0;
  haveFontData = FALSE;

  // create quadric objects for drawing spheres, cylinders, and disks
  // also make sure we are at start drawing filled objects ... may be changed
  // by setting the detail level
  objQuadric = gluNewQuadric();
  pointsQuadric = gluNewQuadric();
  sphereQuadric = objQuadric;
  gluQuadricDrawStyle(objQuadric, (GLenum)GLU_FILL);
  gluQuadricDrawStyle(pointsQuadric, (GLenum)GLU_POINT);
}

// destructor
OpenGLRenderer::~OpenGLRenderer(void) {
 // delete the quadrics
 gluDeleteQuadric(objQuadric);
 gluDeleteQuadric(pointsQuadric);
}

/////////////////////////  protected nonvirtual routines  

// change current line width
void OpenGLRenderer::set_line_width(int w) {
  if(w > 0) {
   glLineWidth((GLfloat)w);
   lineWidth = w;
  }
}


// change current line style ... styles defined in constructor
void OpenGLRenderer::set_line_style(int s) {
  if(s == ::DASHEDLINE) {
    lineStyle = s;
    glEnable(GL_LINE_STIPPLE);
  } else {
    lineStyle = ::SOLIDLINE;
    glDisable(GL_LINE_STIPPLE);
  }
}


// change current sphere resolution
void OpenGLRenderer::set_sphere_res(int r) {
  sphereRes = r;
}

// change current sphere type
void OpenGLRenderer::set_sphere_mode(int m) {
  sphereMode = m;

  // set which quadric to use for drawing spheres ... the one for the
  // current detail level, or the one for points (which will always be
  // used regardless of the current detail level)
  if(m == ::SOLIDSPHERE) {
    sphereQuadric = objQuadric;
  } else if(m == ::POINTSPHERE) {
    sphereQuadric = pointsQuadric;
  } else {
    sphereQuadric = objQuadric;
    sphereMode = ::SOLIDSPHERE;
  }
}

// this routine draws a triangle between three points 
void OpenGLRenderer::triangle( float *p1, float *p2, float *p3,
			   float *n1, float *n2, float *n3) {
  GLenum drawmode = GL_TRIANGLES;
  if(my_detail_level == DETAIL_POINTS)
    drawmode = GL_POINTS;
  else if(my_detail_level == DETAIL_WIREFRAME)
    drawmode = GL_LINE_LOOP;

  glBegin(drawmode);
  glNormal3fv((GLfloat *)n1);
  glVertex3fv((GLfloat *)p1);
  glNormal3fv((GLfloat *)n2);
  glVertex3fv((GLfloat *)p2);
  glNormal3fv((GLfloat *)n3);
  glVertex3fv((GLfloat *)p3);
  glEnd();
}

void OpenGLRenderer::square(float *perp, float *p1, float *p2, float *p3,
                              float *p4) {
  GLenum drawmode = GL_QUADS;
  if(my_detail_level == DETAIL_POINTS)
    drawmode = GL_POINTS;
  else if(my_detail_level == DETAIL_WIREFRAME)
    drawmode = GL_LINE_LOOP;

  glBegin(drawmode);
  glNormal3fv((GLfloat *)perp);
  glVertex3fv((GLfloat *)p1);
  glVertex3fv((GLfloat *)p2);
  glVertex3fv((GLfloat *)p3);
  glVertex3fv((GLfloat *)p4);
  glEnd();
}


// this routine draws a cylinder from start to end, using rod_res panels,
//      and of radius rod_radius
void OpenGLRenderer::cylinder(float *end, float *start, int rod_res,
   float rod_radius, float rod_top_radius) {
  float R, RXY, phi, theta, lenaxis[3];

  // need to do some preprocessing ... find length of vector
  lenaxis[0] = end[0] - start[0];
  lenaxis[1] = end[1] - start[1];
  lenaxis[2] = end[2] - start[2];
  R = sqrtf(lenaxis[0]*lenaxis[0]+lenaxis[1]*lenaxis[1]+lenaxis[2]*lenaxis[2]);
  if (R <= 0.0)
    return;

  // determine phi rotation angle, amount to rotate about y
  phi = acosf(lenaxis[2] / R);

  // determine theta rotation, amount to rotate about z
  RXY = sqrtf(lenaxis[0]*lenaxis[0]+lenaxis[1]*lenaxis[1]);
  if(RXY <= 0.0) {
    theta = 0.0;
  } else {
    theta = acosf(lenaxis[0] / RXY);
    if(lenaxis[1] < 0.0)
      theta = (2.0 * PI) - theta;
  }

  // now set up proper transformations to move cylinder from Z-axis to position
  glPushMatrix();
  glTranslatef((GLfloat)(start[0]), (GLfloat)(start[1]),
	       (GLfloat)(start[2]));
  if(theta != 0.0)
    glRotatef(((GLfloat)theta / PI) * 180.0, 0.0, 0.0, 1.0);
  if(phi != 0.0)
    glRotatef(((GLfloat)phi / PI) * 180.0, 0.0, 1.0, 0.0);

  // call utility routine to draw spheres
  gluCylinder(objQuadric, (GLdouble)rod_radius, (GLdouble)rod_top_radius,
	      (GLdouble)R, (GLint)rod_res, 1);

  // if this is a cone, we also draw a disk at the bottom
  gluQuadricOrientation(objQuadric, (GLenum)GLU_INSIDE);
  gluDisk(objQuadric, (GLdouble)rod_top_radius, (GLdouble)rod_radius,
	  (GLint)rod_res, 1);
  gluQuadricOrientation(objQuadric, (GLenum)GLU_OUTSIDE);

  // restore the matrix
  glPopMatrix();
}


/////////////////////////  protected virtual routines  

// define a new light source ... return success of operation
int OpenGLRenderer::do_define_light(int n, float *color, float *position) {
  GLfloat light_position[4];
  GLfloat light_color[4];
  
  for(int i=0; i < 3; i++)  {
    light_color[i] = color[i];
    light_position[i] = position[i];
  }
  light_position[3] = 0.0;
  light_color[3] = 1.0;

  /*
  msgWarn << "Defining light " << n << ": pos = " << light_position[0];
  msgWarn << ", " << light_position[1] << ", " << light_position[2];
  msgWarn << ", " << light_position[3] << "\n";
  msgWarn << "               " << n << ": col = " << light_color[0];
  msgWarn << ", " << light_color[1] << ", " << light_color[2];
  msgWarn << ", " << light_color[3] << sendmsg;
  */

  glLightfv((GLenum)(GL_LIGHT1 + n), GL_POSITION, light_position);
  glLightfv((GLenum)(GL_LIGHT1 + n), GL_DIFFUSE, light_color);
  glLightfv((GLenum)(GL_LIGHT1 + n), GL_SPECULAR, light_color);

  return TRUE;
}

// activate a given light source ... return success of operation
int OpenGLRenderer::do_activate_light(int n, int turnon) {

  if(turnon) {
    glEnable((GLenum)(GL_LIGHT1 + n));
  } else {
    glDisable((GLenum)(GL_LIGHT1 + n));
  }

  return TRUE;
}

// define a new material ... return success of operation
int OpenGLRenderer::do_define_material(int, float *) {
  // for now, this does nothing ... we have OpenGL use the current color
  // to set the material diffuse setting, and the other material
  // components are never changed.
  return TRUE;
}

// activate the use of material characteristics ... return success
// for OpenGL, this just enables/disables the light model calculations
int OpenGLRenderer::do_activate_material(int, int turnon) {

  if(turnon) {
    if(aa_enabled()) {
      need_aaon = TRUE;
      aa_off();
    }
    if(cueing_enabled()) {
      need_cueon = TRUE;
      cueing_off();
    }

    glEnable(GL_LIGHTING);
  } else {
 
    if(need_aaon && aa_available())
      aa_on();
    if(need_cueon && cueing_available())
      cueing_on();
    need_aaon = need_cueon = FALSE;

    glDisable(GL_LIGHTING);
  }

  return TRUE;
}

/////////////////////////  public virtual routines  

// size of physical display object
void OpenGLRenderer::screen_size_mm(long& x, long& y) {
  x = screenX;
  y = screenY;
}

void OpenGLRenderer::screen_size_pixel(long& x, long& y) {
  x = screenX;
  y = screenY;
}


//
// virtual routines to affect the device's transformation matrix
// These do not use the local transMat matrix, but just use the GL viewing
// matrix directly.
//

void OpenGLRenderer::push(void) {
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
}

void OpenGLRenderer::pop(void) {
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
}

void OpenGLRenderer::loadmatrix(Matrix4 &m) {
  GLfloat *oldmat = (GLfloat *)(m.mat[0]);
  GLfloat tmpmat[16];
  int k = 0;
  for(int i=0; i < 4; i++)
    for(int j=0; j < 4; j++)
      tmpmat[k++] = oldmat[4*j + i];
  glMatrixMode(GL_MODELVIEW);
  glLoadMatrixf(tmpmat);
}

void OpenGLRenderer::multmatrix(Matrix4 &m) {
  GLfloat tmpmat[16];
  int i, j, k=0;
  for(i=0; i < 4; i++)
    for(j=0; j < 4; j++)
      tmpmat[k++] = (GLfloat)(m.mat[j][i]);
  glMatrixMode(GL_MODELVIEW);
  glMultMatrixf(tmpmat);
}

//
// virtual routines to return 2D screen coordinates, given 2D or 3D world
// coordinates.  These assume the proper GL window has focus, etc.
// The coordinates returned are absolute screen coords, relative to the lower
// left corner of the display monitor
//
void OpenGLRenderer::abs_screen_loc_3D(float *loc, long *spos) {
  GLdouble modelMatrix[16], projMatrix[16];
  GLint vp[4];
  GLdouble pos[3];

  // get current matrices and viewport for project call
  glGetDoublev(GL_MODELVIEW_MATRIX, modelMatrix);
  glGetDoublev(GL_PROJECTION_MATRIX, projMatrix);
  glGetIntegerv(GL_VIEWPORT, vp);

  // call the GLU routine to project the object coord to world coords
  if(!gluProject((GLdouble)(loc[0]), (GLdouble)(loc[1]), (GLdouble)(loc[2]),
     modelMatrix, projMatrix, vp, pos, pos + 1, pos + 2)) {
    msgErr << "Cannot determine window position of world coordinate.";
    msgErr << sendmsg;
  } else {
    spos[0] = (long)(pos[0]) + xOrig;
    spos[1] = (long)(pos[1]) + yOrig;
  }
}

void OpenGLRenderer::abs_screen_loc_2D(float *loc, long *spos) {
  float newloc[3];
  newloc[0] = loc[0];
  newloc[1] = loc[1];
  newloc[2] = 0.0;
  abs_screen_loc_3D(newloc, spos);
}


// Given a 3D point (pos A),
// and a 2D rel screen pos point (for pos B), computes the 3D point
// which goes with the second 2D point at pos B.  Result returned in B3D.
// NOTE: currently, this algorithm only supports the simple case where the
// eye look direction is along the Z-axis.  A more sophisticated version
// requires finding the plane to which the look direction is normal, which is
// assumed here to be the Z-axis (for simplicity in coding).
void OpenGLRenderer::find_3D_from_2D(float *A3D, float *B2D, float *B3D) {
  GLdouble modelMatrix[16], projMatrix[16], w1[3], w2[3];
  GLint vp[4];
  float lsx, lsy;		// used to convert rel screen -> abs

  // get current matrices and viewport for unproject call
  glGetDoublev(GL_MODELVIEW_MATRIX, modelMatrix);
  glGetDoublev(GL_PROJECTION_MATRIX, projMatrix);
  glGetIntegerv(GL_VIEWPORT, vp);

  // get window coordinates of 2D point
  lsx = B2D[0];
  lsy = B2D[1];
  abs_screen_pos(lsx, lsy);

  // call the GLU routine to unproject the window coords to world coords
  if(!gluUnProject((GLdouble)lsx, (GLdouble)lsy, 0,
     modelMatrix, projMatrix, vp, w1, w1 + 1, w1 + 2)) {
    msgErr << "Cannot determine world coords of window position 1.";
    msgErr << sendmsg;
    return;
  }
  if(!gluUnProject((GLdouble)lsx, (GLdouble)lsy, 1.0,
     modelMatrix, projMatrix, vp, w2, w2 + 1, w2 + 2)) {
    msgErr << "Cannot determine world coords of window position2.";
    msgErr << sendmsg;
    return;
  }

  // finally, find the point where line returned as w1..w2 intersects the 
  // given 3D point's plane (this plane is assumed to be parallel to the X-Y
  // plane, i.e., with a normal along the Z-axis.  A more general algorithm
  // would need to find the plane which is normal to the eye look direction,
  // and which contains the given 3D point.)
  
  // special case: w1z = w2z ==> just return given 3D point, since there
  //		is either no solution, or the line is in the given plane
  if(w1[2] == w2[2]) {
    memcpy((void *)B3D, (void *)A3D, 3*sizeof(float));
  } else {
    float relchange = (A3D[2] - w1[2]) / (w2[2] - w1[2]);
    B3D[0] = (w2[0] - w1[0]) * relchange + w1[0];
    B3D[1] = (w2[1] - w1[1]) * relchange + w1[1];
    B3D[2] = A3D[2];
  }
}


//
// antialiasing and depth-cueing
//

// turn on antialiasing effect
void OpenGLRenderer::aa_on(void) {
  if(aaAvailable && !aaEnabled) {
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnable(GL_BLEND);
    glEnable(GL_POINT_SMOOTH);
    glEnable(GL_LINE_SMOOTH);
    glEnable(GL_POLYGON_SMOOTH);
    aaEnabled = TRUE;
  }
}

// turn off antialiasing effect
void OpenGLRenderer::aa_off(void) {
  if(aaAvailable && aaEnabled) {
    glDisable(GL_BLEND);
    glDisable(GL_POINT_SMOOTH);
    glDisable(GL_LINE_SMOOTH);
    glDisable(GL_POLYGON_SMOOTH);
    aaEnabled = FALSE;
  }
}

// turn on hardware depth-cueing
void OpenGLRenderer::cueing_on(void) {
  if(cueingAvailable && !cueingEnabled) {
    glEnable(GL_FOG);
    cueingEnabled = TRUE;
  }
}

// turn off hardware depth-cueing
void OpenGLRenderer::cueing_off(void) {
  if(cueingAvailable && cueingEnabled) {
    glDisable(GL_FOG);
    cueingEnabled = FALSE;
  }
}


//
//*******************  the rendering routine  *********************
//
// This scans the given command list until the end, doing the commands
// in the order they appear
//
void OpenGLRenderer::render(void *cmdList) {
  char *cmdptr = (char *)cmdList;
  int tok, cmdsize;

  MSGDEBUG(3, "OpenGLRenderer: rendering command list." << sendmsg);
  
  if(!cmdList)
    return;

  // set the quadric style based on the detail level
  switch(my_detail_level) {
  case DETAIL_NONE:
    return;
  case DETAIL_POINTS:
    gluQuadricDrawStyle(objQuadric, (GLenum)GLU_POINT);
    deactivate_materials();
    break;
  case DETAIL_WIREFRAME:
    gluQuadricDrawStyle(objQuadric, (GLenum)GLU_LINE);
    deactivate_materials();
    break;
  case DETAIL_FLAT:
    gluQuadricDrawStyle(objQuadric, (GLenum)GLU_FILL);
    deactivate_materials();
    break;
  case DETAIL_FULL:
  default:
    gluQuadricDrawStyle(objQuadric, (GLenum)GLU_FILL);
    if(materialsActive)
      activate_material();
    break;
  }

  gluQuadricDrawStyle(objQuadric, (GLenum)GLU_FILL);
  
  // save transformation matrix
  push();

  // scan through the list, getting each command and executing it, until
  // the end of commands token is found
  dataBlock = NULL;
  while((tok = ((int *)cmdptr)[0]) != DLASTCOMMAND) {
    if (tok == DLINKLIST) {
       cmdptr += sizeof(int);       // step forward one (no length here)
       cmdptr = *((char **) cmdptr);// and go to the next link
       tok = ((int *)cmdptr)[0];    // this is guarenteed to be neither
    }                               // a DLINKLIST nor DLASTCOMMAND
       
    cmdsize = ((int *)cmdptr)[1];
    cmdptr += 2*sizeof(int);

    MSGDEBUG(4, "OpenGLRenderer: doing command " << tok);
    MSGDEBUG(4, " of size " << cmdsize << sendmsg);

    if(tok == DPOINT) {
      // plot a point at the given position
      glBegin(GL_POINTS);
      glVertex3fv((GLfloat *)cmdptr);
      glEnd();

    } else if(tok == DPOINT_I) {
      // plot a point at the given position, using indices into dataBlock
      glBegin(GL_POINTS);
      glVertex3fv((GLfloat *)dataBlock + ((int *)cmdptr)[0]);
      glEnd();

    } else if(tok == DLINE) {
      // plot a line
      if(my_detail_level == DETAIL_POINTS) {
	glBegin(GL_POINTS);
      } else {
	glBegin(GL_LINES);
      }
      glVertex3fv((GLfloat *)cmdptr);
      glVertex3fv((GLfloat *)cmdptr + 3);
      glEnd();
      
    } else if(tok == DLINE_I) {
      // plot a line, using indices into dataBlock
      if(my_detail_level == DETAIL_POINTS) {
	glBegin(GL_POINTS);
      } else {
	glBegin(GL_LINES);
      }
      glVertex3fv((GLfloat *)dataBlock + ((int *)cmdptr)[0]);
      glVertex3fv((GLfloat *)dataBlock + ((int *)cmdptr)[1]);
      glEnd();
      
    }  else if(tok == DSPHERE) {
      glPushMatrix();
      glTranslatef((GLfloat)*((float *)cmdptr),
		   (GLfloat)*((float *)cmdptr + 1),
		   (GLfloat)*((float *)cmdptr + 2));
      gluSphere(sphereQuadric, (GLdouble)*((float *)cmdptr + 3),
		(GLint)sphereRes, (GLint)sphereRes);
      glPopMatrix();

    }  else if(tok == DSPHERE_I) {
      int n1 = (int)(((float *)cmdptr)[0]);
      float *cptr = ((float *)dataBlock) + n1;
      glPushMatrix();
      glTranslatef((GLfloat)*(cptr), (GLfloat)*(cptr+1), (GLfloat)*(cptr+2));
      gluSphere(sphereQuadric, (GLdouble)(((float *)cmdptr)[1]),
		(GLint)sphereRes, (GLint)sphereRes);
      glPopMatrix();
     
    }  else if(tok == DTRIANGLE) {
      // draw a triangle between the three points
      triangle(  (float *) cmdptr      , ((float *) cmdptr) + 3,
		 ((float *) cmdptr) +  6, ((float *) cmdptr) + 9,
		 ((float *) cmdptr) + 12, ((float *) cmdptr) + 15  );

    }  else if(tok == DTRIANGLE_I) {
      // draw a triangle between the three points, using indices into dataBlock
      triangle(dataBlock + ((int *) cmdptr)[1],
	       dataBlock + ((int *) cmdptr)[2],
	       dataBlock + ((int *) cmdptr)[3],
	       dataBlock + ((int *) cmdptr)[0],
	       dataBlock + ((int *) cmdptr)[0],
	       dataBlock + ((int *) cmdptr)[0]  );

    }  else if(tok == DSQUARE) {
      // draw a square, given the four points
      square( (float *) cmdptr     , ((float *) cmdptr) + 3, 
             ((float *) cmdptr) + 6, ((float *) cmdptr) + 9,
             ((float *) cmdptr) +12 );

    }  else if(tok == DSQUARE_I) {
      // draw a square, given the four points, using indices into dataBlock
      square(dataBlock + ((int *)cmdptr)[0], dataBlock + ((int *)cmdptr)[1],
             dataBlock + ((int *)cmdptr)[2], dataBlock + ((int *)cmdptr)[3],
             dataBlock + ((int *)cmdptr)[4]);

    } else if(tok == DCYLINDER) {
      // draw a cylinder of given radius and resolution
      cylinder((float *)cmdptr, ((float *)cmdptr) + 3,
	       (int)(((float *)cmdptr)[7]), ((float *)cmdptr)[6],
	       ((float *)cmdptr)[6]);

    }  else if(tok == DCYLINDER_I) {
      // draw a cylinder of given radius and resolution, using indices
      cylinder(dataBlock +  (int)(((float *)cmdptr)[0]),
	       dataBlock +  (int)(((float *)cmdptr)[1]),
	       (int)(((float *)cmdptr)[3]), 
	       ((float *)cmdptr)[2], ((float *)cmdptr)[2]);

    } else if(tok == DCONE) {
      // draw a cone of given radius and resolution
      cylinder(((float *)cmdptr) + 3, ((float *)cmdptr),
	   (int)(((float *)cmdptr)[7]), ((float *)cmdptr)[6], 0.0);

    }  else if(tok == DCONE_I) {
      // draw a cone of given radius and resolution, using indices
      cylinder(dataBlock +  (int)(((float *)cmdptr)[1]),
	   dataBlock +  (int)(((float *)cmdptr)[0]),
	   (int)(((float *)cmdptr)[3]), 
	   ((float *)cmdptr)[2], 0.0);

    } else if(tok == DTEXTPOS) {
      // move the current character position to the point given
      glRasterPos3f((GLfloat)(((float *)cmdptr)[0]),
		    (GLfloat)(((float *)cmdptr)[1]),
		    (GLfloat)(((float *)cmdptr)[2]));

    } else if(tok == DTEXTPOS_I) {
      // move the current character position to the point given, using indices
      int n1 = ((int *)cmdptr)[0];
      glRasterPos3f((GLfloat)dataBlock[n1],
		    (GLfloat)dataBlock[n1 + 1],
		    (GLfloat)dataBlock[n1 + 2]);
      
    } else if(tok == DTEXT) {
      // display text at current position
      if(haveFontData) {
	glPushAttrib(GL_LIST_BIT);
	glListBase(fontListBase);
	glCallLists(strlen(cmdptr), GL_UNSIGNED_BYTE, (GLubyte *)cmdptr);
	glPopAttrib();
      }

    } else if(tok == DCOLORINDEX) {
      // set the current color to the given color index ... assumes the
      // color has already been defined
      int n = ((int *)cmdptr)[0];

      if(!matDefined[n]) {
	msgErr << "OpenGLRenderer: Error, color " << n << " not defined.";
	msgErr << sendmsg;
      } else {
	// bind the material as the current one
	materialOn = n;

	// set the color
	glColor4fv((GLfloat *)(&(matData[n][COLOR_INDEX])));
      }

    } else if(tok == DCOLORRGB) {
      glColor3fv((GLfloat *)cmdptr);

    } else if(tok == DPICKPOINT || tok == DPICKPOINT_I
		|| tok == DPICKLINE || tok == DPICKLINE_I
		|| tok == DPICKBOX || tok == DPICKBOX_I) {
      ; 		// do nothing, these are picking tokens

    } else if(tok == DLIGHTONOFF) {
      // turn on or off a light source
      activate_light(((int *)cmdptr)[0], ((int *)cmdptr)[1]);

    } else if(tok == DMATERIALS) {
      // enable/disable material characteristics
      /*
      msgWarn << "OGLR: toggling materials: " << ((int *)cmdptr)[0] << "\n";
      msgWarn << "      materials on now: " << materialsActive << "(";
      msgWarn << materialOn << ")\n";
      msgWarn << "      current material color: ";
      msgWarn << matData[materialOn][COLOR_INDEX] << ", ";
      msgWarn << matData[materialOn][COLOR_INDEX + 1] << ", ";
      msgWarn << matData[materialOn][COLOR_INDEX + 2] << ", ";
      msgWarn << matData[materialOn][COLOR_INDEX + 3] << sendmsg;
      */
      if(my_detail_level == DETAIL_FULL && ((int *)cmdptr)[0])
        activate_material();
      else
        deactivate_materials();

    } else if(tok == DSPHERERES) {
      // set the current sphere resolution
      set_sphere_res(((int *)cmdptr)[0]);

    } else if(tok == DSPHERETYPE) {
      // set the current sphere type
      set_sphere_mode(((int *)cmdptr)[0]);
      
    } else if(tok == DLINESTYLE) {
      // set the current line style
      set_line_style(((int *)cmdptr)[0]);
      
    } else if(tok == DLINEWIDTH) {
      // set the current line width
      set_line_width(((int *)cmdptr)[0]);

    } else if(tok == DPUSH) {
      // push the current trans matrix
      push();

    } else if(tok == DPOP) {
      // pop the current trans matrix
      pop();
      
    } else if(tok == DLOAD) {
      // load the given trans matrix onto the stack
      Matrix4 tmpmat((float *)cmdptr);
      loadmatrix(tmpmat);

    } else if(tok == DMULT) {
      // multiply the given trans matrix
      Matrix4 tmpmat((float *)cmdptr);
      multmatrix(tmpmat);

    } else if(tok == DTRANS) {
      // do a translation
      glTranslatef((GLfloat)(((float *)cmdptr)[0]),
		   (GLfloat)(((float *)cmdptr)[1]),
		   (GLfloat)(((float *)cmdptr)[2]));
      
    } else if(tok == DSCALE) {
      // do a scale
      glScalef((GLfloat)(((float *)cmdptr)[0]),
	       (GLfloat)(((float *)cmdptr)[1]),
	       (GLfloat)(((float *)cmdptr)[2]));

    } else if(tok == DROT) {
      // do a rotation about a given axis
      GLfloat axis[3];
      axis[0] = axis[1] = axis[2] = 0.0;
      axis[(int)(((float *)cmdptr)[1])] = 1.0;
      glRotatef((GLfloat)(((float *)cmdptr)[0]), axis[0], axis[1], axis[2]);

    } else if(tok == DCOLORDEF) {
      // define a new color
      int n = (int)(((float *)cmdptr)[0]);
      define_material(n, ((float *)cmdptr) + 1);

    } else if(tok == DLIGHTDEF) {
      // define a new light source
      int n = (int)(((float *)cmdptr)[0]);
      define_light(n, ((float *)cmdptr) + 1, ((float *)cmdptr) + 4);
      
    } else if(tok == DCLEAR) {
      // clear the display
      clear();

    } else if(tok == DMOREDATA || tok == DNEWDATA) {
      // set the current drawing data block
#ifdef VMDCAVE
      dataBlock = (float *)cmdptr;	// point to current position in list
#else
      dataBlock = ((float **)cmdptr)[0];	// point to given memory loc
#endif

    } else {
      // command not found, help!
      msgErr << "OpenGLRenderer: Unknown drawing token " << tok
             << " encountered ... Skipping this command." << sendmsg;
    }
    
    // update position in array
    cmdptr += cmdsize;
  }
  
  // restore transformation matrix
  pop();
}

/***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	OpenGLRenderer.C,v $
 * Revision 1.3  96/02/03  02:33:51  02:33:51  billh (Bill Humphrey)
 * Fixes to compile on HP's
 * 
 * Revision 1.2  1996/02/01 01:34:02  billh
 * Fixed problem with finding the origin of the window;
 * added character text display and cursor shape changing capabilities.
 *
 * Revision 1.1  1995/10/19 16:08:17  billh
 * Initial revision
 *
 * Revision 1.19  1995/08/15  21:33:20  dalke
 * added support for AIX3
 *
 * Revision 1.18  1995/06/07  02:28:23  billh
 * Moved cursor and button virtual functions to GLDisplayDevice
 * (from GLRenderer).
 *
 * Revision 1.17  1995/05/11  22:49:04  billh
 * Added virtual functions to queue and check events, and to get/modify
 * the cursor position, shape, etc.  All graphics/windowing-specific
 * functions have now been moved to the DisplayDevice.
 *
 * Revision 1.16  95/04/12  20:36:26  billh
 * Added two-sided lighting; made sure polygons drawn counter-clockwise.
 * 
 * Revision 1.15  1995/04/11  09:46:51  dalke
 * new formate for the DTRIANGLE that allows different normals
 * for the triangle verticies.
 *
 * Revision 1.14  1995/04/05  11:27:08  billh
 * Fixed bug with mapping 2D -> 3D coordinates, for special case w1z==w2z
 *
 * Revision 1.13  95/03/24  18:49:36  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.12  1995/03/10  07:49:33  dalke
 * added fast cylinders by precomputing in the DispCmd
 * This makes things about 2.5 times faster for pure cylinder
 * drawing.
 *
 * Revision 1.11  1995/02/22  04:07:05  billh
 * Added virtual routine find_3D_from_2D, which takes a 3D point at pos A,
 * and 2D rel screen position at B, and returns the 3D point corresponding
 * to the 2D point at B in the same plane as point A.
 *
 * Revision 1.10  1995/02/12  07:15:50  dalke
 * now reads the new "DLINKLIST" and continues rendering with
 * the new memory
 *
 * Revision 1.9  1995/02/03  23:18:53  billh
 * Improved cylinder drawing; normals now calculated for each vertex.
 *
 * Revision 1.8  1994/11/03  01:35:56  billh
 * Fixed problem with drawing indexed spheres with NPGL library.
 *
 * Revision 1.7  1994/10/28  18:31:10  billh
 * Added definitions for line styles; disabled near and far clipping for
 * HP's using NPGL library.
 *
 * Revision 1.6  1994/10/05  05:33:01  billh
 * arguments changed to allow to compile on HP's.
 *
 * Revision 1.5  1994/10/05  04:37:15  billh
 * Took out double backslash from text, even in comments.
 *
 * Revision 1.4  1994/10/04  22:22:17  billh
 * fixed syntax error with include
 *
 * Revision 1.3  1994/10/04  22:07:55  billh
 * Took out sphere library usage if __NPGL__ is defined, since the npgl lib
 * does not have a version of the sphere drawing routines.  Spheres are
 * just drawn as points for npgl rendering.
 *
 * Revision 1.2  94/09/28  22:06:52  billh
 * changed fsqrt to sqrtf
 * 
 * Revision 1.1  1994/09/23  06:01:39  billh
 * Initial revision
 *
 ***************************************************************************/
