/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Mouse.h,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.17 $	$Date: 96/03/23 05:05:27 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Mouse UI object, which maintains the current state of the 
 * mouse, including what it is currently used for, and how much it has moved
 * from one measurement to the next.  This also deals with any pop-up or
 * pull-down menus available by using the mouse, as well as picking objects.
 *
 * A three-button mouse is assumed here, with the following usage:
 *	1) Buttons 1 and 2 : manipulation and picking.
 *	2) Button 3 (right): pop-up menu
 *
 * This is the general base class definition; specific versions for each
 * windowing/graphics system may be supplied.  This class can function as
 * it is, however, and will just be a 'zero' mouse - the buttons will never
 * be pressed, and the cursor will always be at 0,0.
 *
 ***************************************************************************/
#ifndef MOUSE_OBJ_H
#define MOUSE_OBJ_H

#include "UIObject.h"
#include "Command.h"
#include "NameList.h"
class PopupMenu;
class DisplayDevice;
class Pickable;
class Inform;
class MouseEvent;

#define TRANS_CURSOR 1
#define SCALE_CURSOR 2
#define PICK_CURSOR 3
#define WAIT_CURSOR 4


//////////////////////  Windowing System Event Object 
class WinEvent : public Command {

public:
  // the type of event
  long eventCode;
  
  // the return value for the event
  long val;

  // constructor; must specify what the event type code and value are
  WinEvent(long d, long v, int newUIid=(-1));

};


//////////////////////  Mouse controller class definition 
class Mouse : public UIObject {

public:
  // different available mouse movement modes
  enum MoveMode { ROTATION, TRANSLATION, SCALING, LIGHT, PICKING, USER};

private:
  // display device we need to use to get info about the graphics display
  // and cursor
  DisplayDevice *dispDev;

  // the current move mode
  MoveMode moveMode;
  
  // are we currently trying to pick something?
  // If so, set this to picking mode in use.
  int activePickMode;

  // flag for whether an active pick is in progress.
  // if 0, no pick.  if > 0, pickInProgress = pick-button + 1
  int pickInProgress;

  // which particular object the mouse is affecting (if mode == LIGHT)
  int moveObj;

  // current & last position of the mouse, and last angular velocity 
  int currX, currY, oldX, oldY;
  
  // whether a pick/drag is in progress (button down)
  int isTracking;

  // which button was pressed to initiate tracking
  int activeButton;

  // whether the mouse has moved since the button went down
  int mouseMoved;
  // if the mouse is moving, the user has the option of specifying an
  // alternate detail level.  This lowers the complexity of the drawn
  // object and makes things faster.  When the move starts, the Mouse
  // stores the current detail level into 'old_detail_level' and changes
  // the display's detail level to its alt_detail_level setting.  When
  // the move is finished, the old_detail_level is restored
  int old_detail_level;

  // increment for mouse translation, rotation
  float transInc, rotInc, scaleInc;

  // rotational velocities, and scaling factor, and flag for whether rotating
  float xRotVel, yRotVel, zRotVel, scaling, RotVelScale;

  // do action when the mouse is moved
  // arg: which buttons are currently being pressed
  int mouse_moved(int b1Down, int b2Down);

  //
  // pop-up menu data
  //

  // routine which creates the initial mouse pop-up menu.
  // The 1st argument is an optional Pickable object which may have a
  // popup menu available for it; in that case  it is added to the main
  // menu displayed.  If 1st the argument is null, this just creates the
  // regular popup menu.  The 2nd arg is the tag of the picked item.
  PopupMenu *create_popup_menu(Pickable *, int = 0);

  // list of user-defined menus
  NameList<NameListStringPtr> userMenuList;

  // the particular list of user-defined single items ... this is one of the
  // submenus in the 'userMenu' member, it is just a special one because it
  // is automatically created when the Mouse is started.
  NameListStringPtr mainUserMenu;

  // routine which actually adds a new menu separator
  int do_add_user_menu_separator(NameListStringPtr);

  // routine which actually adds a new menu item
  int do_add_user_menu_item(NameListStringPtr, char *, char *);

  //
  // keystroke command data
  //

  // routine which creates the initial keyboard shortcut commands
  void create_key_commands(void);

  // namelist of keystroke-vs-text command
  NameList<char *> userKeys;

public:
  // constructor/destructor
  // args: UIList to register with, CommandQueue to use, display which
  // does graphics drawing, and display with pointer info (x,y, buttons)
  // If the last arg is not given, the cursor display is set the same as
  // the graphics display device.
  Mouse(UIList *, CommandQueue *, DisplayDevice *);
  virtual ~Mouse(void);
  
  //
  // general mouse actions
  //

  // stop rotation of object
  void stop_rotation(void);

  // routines for Tk (or other non-polling toolkit) mouse interaction
  int is_tracking(void) const;
  void handle_mouse_event(const MouseEvent &);

  //
  // routines to query about the status of the mouse
  //

  // return the current x or y position of the mouse, or the state of a button
  int x(void);
  int y(void);
  int button_down(int);

  //
  // routines used to deal with the current mouse mode
  //

  // set the current move mode
  int move_mode(MoveMode, int = 0);
  
  // return the current move mode
  MoveMode curr_move_mode(void) { return moveMode; }

  // string name for the given move mode
  char *move_mode_name(MoveMode);

  //
  // routines used to deal with the current picking mode
  //

  // return the current active pick mode (-1 if none)
  int curr_pick_mode(void) { return activePickMode; }

  // return whether we are currently picking something
  int picking(void) { return pickInProgress; }

  //
  // routines to set the user-defined menu items.  All return success.
  //
  
  // create a new submenu in the main user-defined list (it will be initially
  // empty, but can be added to later)
  int create_user_menu(char *label);

  // add a new item to the user-defined list, at the main level (as opposed
  // to adding a new submenu)
  int add_user_menu_item(char *text, char *txtcmd);

  // add a separator to the end of the main user menu
  int add_user_menu_separator(void);

  // add a new item to the submenu with the given name
  int add_user_submenu_item(char *submenu, char *text, char *txtcmd);

  // add a separator to the end of the specified user submenu
  int add_user_submenu_separator(char *submenu);

  // print out the user menu commands to the given Inform object.  Ret success
  int print_user_menu(Inform *);

  //
  // routines to set the user-defined single-key shortcuts commands.
  // All return success.
  //
  
  // bind a new text command to a given key ... replaces old command if nec.
  int add_user_key_command(char, const char *);

  // print out the user keyboard shortcut commands.  Return success.
  int print_user_keys(Inform *);

  //
  // virtual routines for UI init/display
  //
   
  // reset the user interface (force update of all info displays)
  virtual void reset(void);
  
  // update the display due to a command being executed.  Return whether
  // any action was taken on this command.
  // Arguments are the command type, command object, and the 
  // success of the command (T or F).
  virtual int act_on_command(int, Command *, int );
  
  // check for an event, and queue it if found.  Return TRUE if an event
  // was generated.
  virtual int check_event(void);
 
};

#endif

