/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: MoleculeFileEDM.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 95/05/12 00:39:47 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *  read in the electron density map and construct isosurfaces
 *   the format is described in the X-PLOR 3.1 manual, p211
 *
 ***************************************************************************/

#include "MoleculeFileEDM.h"
#include "ReadEDM.h"
#include "DispCmds.h"
#include "Inform.h"


MoleculeFileEDM::MoleculeFileEDM(char *filename, Scene *sc)
: MoleculeFile(filename, sc)
{
   edm = NULL;
}
MoleculeFileEDM::MoleculeFileEDM(char *filename, Displayable *disp)
: MoleculeFile(filename, disp)
{
   edm = NULL;
}
MoleculeFileEDM::~MoleculeFileEDM(void)
{
   if (edm) {
      delete edm;
   }
}


int MoleculeFileEDM::create(void)
{
   edm = read_edm(strFile);
   if (!edm) {
      return FALSE;
   }

   return(MoleculeFile::create());  // This shouldn't do anything bad
}


// this will be changed to make things more useful
void MoleculeFileEDM::create_cmdlist(void) {
   find_isosurface(0.90);
}

// return position of the center-of-volume
void MoleculeFileEDM::cov(float &x, float &y, float &z) {
   if (edm) {
      x = (edm->minx + edm->maxx)/2.0;
      y = (edm->miny + edm->maxy)/2.0;
      z = (edm->minz + edm->maxz)/2.0;
   } else {
      x = y = z = 0.0;
   }
}

// scaling factor required to make the grid fit within (-1 ... 1)
float MoleculeFileEDM::scale_factor(void) { 
   float wx = edm->maxx - edm->minx;
   float wy = edm->maxy - edm->miny;
   float wz = edm->maxz - edm->minz;
   if (wx > wy) {
      if (wx > wz) {
	 return 2.0/wx;
      } else {
	 return 2.0/wz;
      }
   } else {
      if (wy > wz) {
	 return 2.0/wy;
      } else {
	 return 2.0/wz;
      }
   }
}


// I find first which points are on the boundary using
// marching cubes, I think.  Actually, I'm bastardizing
// the 2nd half of the Wyvill, McPheeters, and Wyvill
// paper on "Soft Objects", in _Advanced_Computer_Graphics_
// Ed. by Tosiyasu Kunii  (Proceesdings of Computer Graphics
// Tokyo, 1986)

// Find the isoline(s), assuming linear interpolation, of a square
//  Input is scalar values of the corners of a square of unit size
//  The corners of the square are named:
//         side3
//       x12   x22            
//    side4      side2
//       x11   x21 
//         side1

//  The output is either 0, 1, or 2 sets of coordinate pairs
// The number is put into num
// The values are:
// for edge 1 : (edges[0], edges[1]) - (edges[2], edges[3])
// for edge 2 : (edges[4], edges[5]) - (edges[6], edges[7])
#define to
#define side1 {edges[i++] = (iso-x11)/(x21-x11); edges[i++] = 0;}
#define side2 {edges[i++] = 1; edges[i++] = (iso-x21)/(x22-x21); }
#define side3 {edges[i++] = (iso-x12)/(x22-x12); edges[i++] = 1;}
#define side4 {edges[i++] = 0; edges[i++] = (iso-x11)/(x12-x11); }
#define Return {*num = i/4; return;}
static void find_edges( float iso, float x11, float x21, float x22, float x12,
		 int *num, float *edges)
{
   // If iso>X then X is "cold".  If iso < X then X is "hot"
   // There are 2^4 possibilities, of which 7 are unique
   // there is one degeneracy, when one pair of opposite corners
   // are hot and the other pair are cold.  This is settled by:
   // 1) if avg. of the corners is hot, hot wins  (ie. hots are connected)
   // 2) if avg. of the corners is cold, cold wins
   // 3) if avg. of the corners == iso, hot wins ('cause I said so)

   // It is simplest to enumerate all the cases:
   float sum;
   int i=0;
   if (x11 < iso) {
      if (x21 < iso) {
	 if (x22 < iso) {
	    if (x12 < iso) {  // all cold
	       Return;
	    } else {          // only x21 hot
	       side3 to side4;
	       Return;
	    }
	 } else { // x22 hot
	    if (x12 < iso) { // only x22 hot
	       side2 to side3;
	       Return;
	    } else {   // both x22 and x12 hot
	       side2 to side4;
	       Return;
	    }
	 }
      } else { // x21 is hot
	 if (x22 < iso) {
	    if (x12 < iso) { // only x21 is hot
	       side1 to side2;
	       Return;
	    } else {     // opposites are hot -- x12 and x21
	       sum = x11+x12+x21+x22;
	       if (sum < iso*4) {  // cold wins
		  side1 to side2;
		  side3 to side4;
		  Return;
	       } else {
		  side1 to side4;
		  side2 to side3;
		  Return;
	       }
	    }
	 } else {  // x22 is hot (as well as x21)
	    if (x12 < iso) {
	       side1 to side3;
	       Return;
	    } else {   // and so is x12 -- is, only x11 is cold
	       side1 to side4;
	       Return;
	    } // end of "else" for x12
	 } // end of "else" for x22
      } // end of "else" for x21
   } else { // end of "then" clause for x11 -- halfway there! -- now x11 hot
      if (x21 < iso) {
	 if (x22 < iso) {
	    if (x12 < iso) {  // only x11 hot
	       side1 to side4;
	       Return;
	    } else {  // x11 and x12 hot
	       side1 to side3;
	       Return;
	    }
	 } else { // x11 and x22 hot
	    if (x12 < iso) {
	       sum = x11+x12+x22+x21;
	       if (sum < iso*4) { // cold wins
		  side1 to side4;
		  side2 to side3;
	       } else {
		  side1 to side2;
		  side3 to side4;
	       }
	       Return;
	    } else {          // only x21 cold
	       side1 to side2;
	       Return;
	    }
	 }
      } else { // x11 and x21 hot
	 if (x22 < iso) {
	    if (x12 < iso) {  // top is cold, bottom is hot
	       side2 to side4;
	       Return;
	    } else {          // only x22 is cold
	       side2 to side3;
	       Return;
	    }
	 } else { // x11, x21, and x22 are all hot
	    if (x12 < iso) { //
	       side3 to side4;
	       Return;
	    } else {  // everyone is hot
	       Return;
	    } // end of x12
	 } // end of x22
      } // end of x21
   } // end of x11
} // end of function (And I'm glad my editor auto indents parens!)

#define draw_line( x1,  y1,  z1,  x2,  y2,  z2) \
{                                               \
   pos1[0] = x1; pos1[1] = y1; pos1[2] = z1;    \
   pos2[0] = x2; pos2[1] = y2; pos2[2] = z2;    \
   myline.putdata(pos1, pos2, this);            \
}
   
void MoleculeFileEDM::find_isosurface(float val)
{
   int nx, ny, nz;
   int i, j, k;
   reset_disp_list();
   edm->grid->numrange(&nx, &ny, &nz);

   int numedges;
   float edgepoints[8];  // either 0, 1, or 2 2D edges will be returned
   float x0 = edm->minx;
   float y0 = edm->miny;
   float z0 = edm->minz;
   float dx = (edm->maxx - x0) / float(nx);
   float dy = (edm->maxy - y0) / float(ny);
   float dz = (edm->maxz - z0) / float(nz);

   DispCmdLine myline;
   float pos1[3];  // for storing the info for DispCmdLine
   float pos2[3];
   
   // go through the surfaces perpendicular to k
   for (k=0; k<nz-1; k++) {
      for (i=0; i<nx-2; i++) {
	 for (j=0; j<ny-2; j++) {
	    find_edges(val,
		       edm->grid->item(i,j,k),
		       edm->grid->item(i+1,j,k),
		       edm->grid->item(i+1,j+1,k),
		       edm->grid->item(i,j+1,k),
		       &numedges,
		       edgepoints);
	    switch(numedges) {
	     default: break;
	     case 0: break;
	     case 2: draw_line(x0 + (float(i)+edgepoints[4])*dx,
			       y0 + (float(j)+edgepoints[5])*dy,
			       z0 +  float(k)               * dz,
			       x0 + (float(i)+edgepoints[6])*dx,
			       y0 + (float(j)+edgepoints[7])*dy,
			       z0 +  float(k)               * dz);
			       // delibrate fall through!
	     case 1: draw_line(x0 + (float(i)+edgepoints[0])*dx,
			       y0 + (float(j)+edgepoints[1])*dy,
			       z0 +  float(k)               * dz,
			       x0 + (float(i)+edgepoints[2])*dx,
			       y0 + (float(j)+edgepoints[3])*dy,
			       z0 +  float(k)               * dz);
			       break;
	    } // switch
	 }  // j
      } // i 
   } // k

   // go through the surfaces perpendicular to j
   for (j=0; k<ny-1; j++) {
      for (i=0; i<nx-2; i++) {
	 for (k=0; j<nz-2; k++) {
	    find_edges(val,
		       edm->grid->item(i  ,j ,k  ),
		       edm->grid->item(i+1,j ,k  ),
		       edm->grid->item(i+1,j ,k+1),
		       edm->grid->item(i  ,j ,k+1),
		       &numedges,
		       edgepoints);
	    switch(numedges) {
	     default: break;
	     case 0: break;
	     case 2: draw_line(x0 + (float(i)+edgepoints[4])*dx,
			       y0 +  float(j)               *dy,
			       z0 + (float(k)+edgepoints[5])*dz,
			       x0 + (float(i)+edgepoints[6])*dx,
			       y0 +  float(j)               *dy,
			       z0 + (float(k)+edgepoints[7])*dz);
			       // deliberate fall through!
	     case 1: draw_line(x0 + (float(i)+edgepoints[4])*dx,
			       y0 +  float(j)               *dy,
			       z0 + (float(k)+edgepoints[5])*dz,
			       x0 + (float(i)+edgepoints[6])*dx,
			       y0 +  float(j)               *dy,
			       z0 + (float(k)+edgepoints[7])*dz);
			       break;
	    } // switch
	 }  // k
      } // i 
   } // j

   // go through the surfaces perpendicular to i
   for (i=0; i<nx-2; i++) {
      for (j=0; j<ny-2; j++) {
	 for (k=0; k<nz-1; k++) {
	    find_edges(val,
		       edm->grid->item(i,j  ,k),
		       edm->grid->item(i,j+1,k),
		       edm->grid->item(i,j+1,k+1),
		       edm->grid->item(i,j  ,k+1),
		       &numedges,
		       edgepoints);
	    switch(numedges) {
	     default: break;
	     case 0: break;
	     case 2: draw_line(x0 + (float(i)              )*dx,
			       y0 + (float(j)+edgepoints[4])*dy,
			       z0 + (float(k)+edgepoints[5])*dz,
			       x0 + (float(i)              )*dx,
			       y0 + (float(j)+edgepoints[6])*dy,
			       z0 + (float(k)+edgepoints[7])*dz);
			       // delibrate fall through!
	     case 1: draw_line(x0 + (float(i)              )*dx,
			       y0 + (float(j)+edgepoints[0])*dy,
			       z0 + (float(k)+edgepoints[1])*dz,
			       x0 + (float(i)              )*dx,
			       y0 + (float(j)+edgepoints[2])*dy,
			       z0 + (float(k)+edgepoints[3])*dz);
			       break;
	    } // switch
	 }  // j
      } // i 
   } // k
}

int MoleculeFileEDM::add_rep(AtomColor *ac, AtomRep *ar, AtomSel *as) {
   return MoleculeFile::add_rep(ac, ar, as);
}

int MoleculeFileEDM::change_rep(int n, AtomColor *ac,
				AtomRep *ar, AtomSel *as) {
   return MoleculeFile::change_rep(n, ac, ar, as);
}

