/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Molecule.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.8 $	$Date: 96/03/24 14:01:23 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Main Molecule objects, which contains all the capabilities necessary to
 * store, draw, and manipulate a molecule.  This adds to the functions of
 * DrawMolecule and BaseMolecule by adding routines to read and write various
 * coordinate files.
 *
 * Note: Other capabilities, such as writing structure files, mutations, etc.
 * should go into this level.
 *
 ***************************************************************************/

#include "Molecule.h"
#include "Inform.h"
#include "DisplayDevice.h"


///////////////////////////  constructor  
Molecule::Molecule(char *src, Scene *sc)
  : DrawMolecule(sc), coorIOFiles(8) {
  source = stringdup(src);
  source2 = NULL;
}

Molecule::Molecule(char *src, Displayable *dp) 
  : DrawMolecule(dp), coorIOFiles(8) {
  source = stringdup(src);
  source2 = NULL;
}

///////////////////////////  destructor  
Molecule::~Molecule(void) {
  if(source)  delete [] source;
  if(source2) delete [] source2;

  while(coorIOFiles.num() > 0) {
    delete coorIOFiles[0];
    coorIOFiles.remove(0);
  }
}

///////////////////////////  public routines  

// set the info about the file type (PSF, PDB, XYZ, ...)
void Molecule::set_primary_type(int typ)
{
  filetype = typ;
}
// if an addition coordinate file was used, set the info about it
void Molecule::set_secondary_info(char *name, int typ)
{
  if (source2) delete [] source2;
  source2 = stringdup(name);
  filetype2 = typ;
}

// ask to read in a coordinate file of the specified type
// arguments: filename, type, beginning frame, ending frame, frame skip
// (if any of the last three are < 0, default values for that arg used).
// Return success.
int Molecule::read_coor_file(char *fn, int t, int begFrame, int endFrame,
	int frameSkip) {

  if(t < 0 || t >= CoorFileData::COORTYPES || !fn)
    return FALSE;

  // create new file, make sure it was opened correctly
  CoorFileData *cfdata = 
  	new CoorFileData(this, fn, t, TRUE, begFrame, endFrame, frameSkip);

  if(cfdata->opened) {
    MSGDEBUG(2,"Added request to read file '" << fn << "', of type " << t);
    MSGDEBUG(2,"\nReading from frame " << cfdata->begFrame << " to ");
    if(cfdata->endFrame < 0) {
      MSGDEBUG(2,"end");
    } else {
      MSGDEBUG(2,cfdata->endFrame);
    }
    MSGDEBUG(2,", skipping frames in steps of " << cfdata->frameSkip);
    MSGDEBUG(2,sendmsg);

    // add to list
    coorIOFiles.append(cfdata);
    return TRUE;
  } else {
    delete cfdata;
    return FALSE;
  }
}


// ask to write out a coordinate file of the specified type
// arguments: filename, type, beginning frame, ending frame, frame skip
// (if any of the last three are < 0, default values for that arg used).
// Return success.
int Molecule::write_coor_file(char *fn, int t, int begFrame, int endFrame,
	int frameSkip) {

  if(t < 0 || t >= CoorFileData::COORTYPES || !fn)
    return FALSE;

  // create new file, make sure it was opened correctly
  CoorFileData *cfdata = 
  	new CoorFileData(this, fn, t, FALSE, begFrame, endFrame, frameSkip);

  if(cfdata->opened) {
    MSGDEBUG(2,"Added request to write file '" << fn << "', of type " << t);
    MSGDEBUG(2,"\nWriting frame " << cfdata->begFrame << " to ");
    if(cfdata->endFrame < 0) {
      MSGDEBUG(2,"end");
    } else {
      MSGDEBUG(2,cfdata->endFrame);
    }
    MSGDEBUG(2,", skipping frames in steps of " << cfdata->frameSkip);
    MSGDEBUG(2,sendmsg);

    // add to list
    coorIOFiles.append(cfdata);
    return TRUE;
  } else {
    delete cfdata;
    return FALSE;
  }
}


// prepare for drawing ... can do one or more of the following:
//	- open a new file and start reading
//	- continue reading an already open file
//	- finish reading a file and close it
// when done, this then 'prepares' the parent class
void Molecule::prepare(DisplayDevice *d) {
  
  // are we already reading/writing?
  if(coorIOFiles.num() > 0) {
    // r/w'ing file ... get new frame, until NULL is returned (then close)
    Timestep *newstep = coorIOFiles[0]->next();

    if(!newstep) {
      // no new frame found; remove from list
      msgInfo << "Finished with coordinate file " << coorIOFiles[0]->name;
      msgInfo << "." << sendmsg;
      delete coorIOFiles[0];
      coorIOFiles.remove(0);
    }
  }

  // do prepare for parent class
  DrawMolecule::prepare(d);
}

