/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: DrawMolecule.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.10 $	$Date: 96/03/20 06:58:26 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Displayable version of a DrawMolecule, derived from BaseMolecule and
 * Displayable3D.  This contains all the info for rendering
 * the molecule.
 *
 ***************************************************************************/

#include "DrawMolecule.h"
#include "AtomColor.h"
#include "AtomRep.h"
#include "AtomSel.h"
#include "PopupMenu.h"
#include "utilities.h"
#include "Global.h"
#include "MoleculeList.h"
// storage for frame text string
static char *frameTextBuf = "Time: +000.00000";

// data for drawing frame position indicator
#define FRPF	0.02
#define FRPL	0.1
static float framePosPoints[6][3] = { { FRPF, FRPF + FRPL, 0.0 },
	{ FRPF, FRPF, 0.0 }, {1.0 - FRPF, FRPF, 0.0},
	{1.0 - FRPF, FRPF + FRPL, 0.0}, {0.0, 0.0, 0.0}, {0.0, 0.0, 0.0} };



///////////////////////  constructor and destructor
DrawMolecule::DrawMolecule(Scene *sc)
	: Displayable3D(MULT, "", sc, 4), repList(8),
	  dataCmd(6, framePosPoints[0]), cmdTxt(frameTextBuf) {
  needRedraw = TRUE;
  active = TRUE;
}

DrawMolecule::DrawMolecule(Displayable *par)
	: Displayable3D(MULT,"",par,4), repList(8),
	  dataCmd(6, framePosPoints[0]), cmdTxt(frameTextBuf) {
  needRedraw = TRUE;
  active = TRUE;
}
  
// destructor ... free up any extra allocated space (the child Displayables
// will be deleted by the Displayable3D destructor)
DrawMolecule::~DrawMolecule(void) {
  // delete all molecule representations
  for(int i=0; i < components(); i++)
    delete component(i);  
}


///////////////////////  private routines

// regenerate the command list
void DrawMolecule::create_cmdlist(void) {

  MSGDEBUG(2,"DrawMolecule: creating command list ..." << sendmsg);

  reset_disp_list();

/*
  cmdMaterials.putdata(FALSE,this);
  cmdLineWidth.putdata(1,this);
  cmdCol.putdata(REGWHITE, this);
  dataCmd.put(this);
  cmdLine.putdata(0, 1, this);
  cmdLine.putdata(1, 2, this);
  cmdLine.putdata(2, 3, this);
  cmdLine.putdata(4, 5, this);
*/

}
  

///////////////////////  public routines

// routine to set the name of the molecule
void DrawMolecule::set_name(char *newname) {
  if(name)  delete [] name;
  name = new char[strlen(newname) + 8];
  sprintf(name,"%s:%03d",newname,id());
}


// create the molecule.  Successive classes should call their parent 'create'
// after they are done.
// Returns success .... if not successful, should be 'delete'ed
int DrawMolecule::create(void) {
  int retval;

  // do base class creation first
  if((retval = BaseMolecule::create()) == TRUE) {
    // update coordinates in display list
    create_cmdlist();
  }
  
  return retval;
}


// prepare for drawing ... do any updates needed right before draw.
//  ... this updates the animation
void DrawMolecule::prepare(DisplayDevice *) {

  // update the animation
  anim_update();
  
}


// Create a popup menu, with the given name; return NULL if none available
// subclasses should have the parent first create a menu, and then add items
// to the end of it.  The argument is the tag of the selected item; here, it is
// the selected atom.
PopupMenu *DrawMolecule::create_popup_menu(int atomIndex) {

  // if we have a specialized menu from an earlier class, get it first
  PopupMenu *pm = Displayable::create_popup_menu(atomIndex);

  // create or add menus as necessary
  if(!pm) {
    pm = new PopupMenu(name);
  }

  //
  // add items to the menu to affect this molecule
  //
  char cmdbuf[128];
  int molIndex = moleculeList -> mol_index_from_id(id());
  if (molIndex < 0) {
    msgErr << "This molecule isn't on the molecule list!" << sendmsg;
  }
  // basic pop-up commands related to the molecule

  // center on this atom
  float *framepos = current()->pos;
  sprintf(cmdbuf, "vmd_set_center {%f %f %f}", 
	  framepos[3*atomIndex], framepos[3*atomIndex+1], 
	  framepos[3*atomIndex+2]);
  pm->add_item("Center", cmdbuf, TRUE, FALSE);

  // pick this atom
  sprintf(cmdbuf, "label add Atoms %d/%d", id(), atomIndex);
  pm->add_item("Pick", cmdbuf, TRUE, FALSE);
  
  // return some info to the user
  PopupMenu *sm1 = new PopupMenu("Info");
  MolAtom *atm = atom(atomIndex);
  sprintf(cmdbuf, "vmd_print_atom_info %d %d", id(), atomIndex);
  sm1->add_item("Print", cmdbuf, TRUE, FALSE);
  sm1->add_separator();
  sprintf(cmdbuf, "name: %s", atm->namestr    );
  sm1->add_item(cmdbuf, "", TRUE, FALSE);
  sprintf(cmdbuf, "type: %s", atm->typestr    );
  sm1->add_item(cmdbuf, "", TRUE, FALSE);
  sprintf(cmdbuf, "index: %d", atomIndex       );
  sm1->add_item(cmdbuf, "", TRUE, FALSE);
  sprintf(cmdbuf, "resname: %s", atm->resnamestr );
  sm1->add_item(cmdbuf, "", TRUE, FALSE);
  sprintf(cmdbuf, "resid: %s", atm->residstr   );
  sm1->add_item(cmdbuf, "", TRUE, FALSE);
  sprintf(cmdbuf, "chain: %c", atm->chainstr[0]);
  sm1->add_item(cmdbuf, "", TRUE, FALSE);
  sprintf(cmdbuf, "segname: %s", atm->segnamestr );
  sm1->add_item(cmdbuf, "", TRUE, FALSE);
  sprintf(cmdbuf, "x: %8.3f", framepos[3*atomIndex + 0]);
  sm1->add_item(cmdbuf, "", TRUE, FALSE);
  sprintf(cmdbuf, "y: %8.3f", framepos[3*atomIndex + 1]);
  sm1->add_item(cmdbuf, "", TRUE, FALSE);
  sprintf(cmdbuf, "z: %8.3f", framepos[3*atomIndex + 2]);
  sm1->add_item(cmdbuf, "", TRUE, FALSE);

  pm->add_submenu(sm1, TRUE, FALSE, FALSE);

  // status information
  sm1 = new PopupMenu("Mol. Status");
  sprintf(cmdbuf, "mol top %d", id());
  sm1->add_item("Top", cmdbuf, TRUE, TRUE, 
	       moleculeList->is_top(molIndex));

  if (moleculeList->active(molIndex)) {
    sprintf(cmdbuf, "mol inactive %d", id());
    sm1->add_item("Make Inactive", cmdbuf, TRUE, FALSE);
  } else {
    sprintf(cmdbuf, "mol active %d", id());
    sm1->add_item("Make Active", cmdbuf, TRUE, FALSE);
  }

  if (moleculeList->fixed(molIndex)) {
    sprintf(cmdbuf, "mol free %d", id());
    sm1->add_item("Free", cmdbuf, TRUE, FALSE);
  } else {
    sprintf(cmdbuf, "mol fix %d", id());
    sm1->add_item("Fix", cmdbuf, TRUE, FALSE);
  }

  sprintf(cmdbuf, "mol off %d", id());
  sm1->add_item("Hide", cmdbuf, TRUE, FALSE);

  pm->add_submenu(sm1, TRUE, FALSE, FALSE);

  // return the final menu
  return pm;
}


// return Nth component ... change to proper return type
DrawMolItem *DrawMolecule::component(int n) {
  if(n >= 0 && n < components())
    return repList[n];
  else
    return NULL;
}


// return whether the Nth atom is displayed.  This is true if ANY
// representation is displaying the given atom
int DrawMolecule::atom_displayed(int n) {
  if(displayed() && n >= 0 && n < nAtoms) {
    for(int i=(components() - 1); i >= 0; i--) {
      if((repList[i])->atom_displayed(n))
        return TRUE;
    }
  }
  
  // if here, atom is not shown
  return FALSE;
}


// delete the Nth representation ... return success
int DrawMolecule::del_rep(int n) {
  DrawMolItem *rep = component(n);
  if(rep) {
    delete rep;		// delete the object
    repList.remove(n);	// and it's slot in the representation list
  }

  return (rep != NULL);
}


// add a new representation (component) ... return success
int DrawMolecule::add_rep(AtomColor *ac, AtomRep *ar, AtomSel *as) {
    AtomColor *newac;
    AtomRep *newar;
    AtomSel *newas;

    // create new representation objects
    if(!ac)
      newac = new AtomColor(NULL, NULL);
    else
      newac = new AtomColor(*ac);
    
    if(!ar)
      newar = new AtomRep();
    else
      newar = new AtomRep(*ar);
      
    if(!as) {
       newas = new AtomSel(NULL);
    } else {
       newas = new AtomSel(*as);
    }

    // find the selected atoms to display, and the color for each atom
    newas->find(this);
    newac->find(this);

    // make sure the selections are now all ok
    if (newas->selected == AtomSel::NO_PARSE) {
      msgErr << "No atoms selected ... new rep not added." << sendmsg;
    } else {
      // create new rep; it will add itself as a child to this object
      repList.append(new DrawMolItem(this, newac, newar, newas));
      return TRUE;
    }
    
    // if we're here, there was an error
    delete newas;
    delete newar;
    delete newac;
    return FALSE;
}


// change the Nth representation ... return success.
// if any object is NULL, that characteristic is not changed.
int DrawMolecule::change_rep(int n, AtomColor *ac, AtomRep *ar, AtomSel *as) {
  DrawMolItem *rep = component(n);
  if(rep) {
    rep->change_color(ac);
    rep->change_rep(ar);
    rep->change_sel(as);
    return TRUE;
  } else
    return FALSE;
}

//  force a redraw of the Nth representation ... return success
int DrawMolecule::force_recalc(int n) {
  DrawMolItem *rep = component(n);
  if (rep) {
    rep->force_recalc();
    return TRUE;
  }
  return FALSE;
}

// redraw all the representations
void DrawMolecule::force_recalc(void) {
  for (int i=components()-1; i>=0; i--) {
    component(i)->force_recalc();
  }
}

