/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CoorFileData.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 96/03/24 07:16:54 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Contains data needed to read and write a coordinate file from a Molecule
 * class.  Will open the file upon creation, and signal if file is OK to
 * read or write.
 *
 ***************************************************************************/

#include "CoorFileData.h"
#include "CoorPDB.h"
#include "CoorDCD.h"
#include "Inform.h"
#include "Molecule.h"
#include "utilities.h"


// static data; last element should be ""
char *CoorFileSuffix[CoorFileData::COORTYPES + 1] = {
	"pdb", "dcd", "" };

char *CoorFileDesc[CoorFileData::COORTYPES + 1] = {
	"Brookhaven Protein Data Bank (pdb)",
	"CHARMM Binary Trajectory File (dcd)", "" };


/////////////////////////  constructor  /////////////////////////////////
CoorFileData::CoorFileData(Molecule *m, char *n, int t, int inp,
		int bf, int ef, int fs) {
  // save data about file
  name = stringdup(n);
  type = t;
  is_input = inp;
  begFrame = bf;
  endFrame = ef;
  frameSkip = fs;
  mol = m;
  opened = FALSE;

  // make sure frame data is correct
  if(begFrame < 0)
    begFrame = 0;
  if(endFrame < begFrame)
    endFrame = (-1);
  if(frameSkip <= 0)
    frameSkip = 1;
  recentFrame = begFrame - 1;

  inf = NULL;
  outf = NULL;
  // make sure frame data is valid
  if(!m || (!is_input && (m->num() < begFrame || endFrame >= m->num())) ) {
    msgErr << "Illegal frames requested for coordinate file I/O" << sendmsg;

  } else {
    // open the requested file and initialize it, to make sure it's valid

    if(is_input) {		// see if READING a new file
      if(type == PDB) {
        // open PDB file for reading
        inf = new ICoorPDB(mol);
      } else if(type == DCD) {
        // open DCD file for reading
        inf = new ICoorDCD(mol);
      }
    } else {			// see if WRITING a new file
      if(type == PDB) {
        // open PDB file for writing
        outf = new OCoorPDB(mol);
      } else if(type == DCD) {
        // open DCD file for writing
        outf = new OCoorDCD(mol, OCoorDCD::BINARY);
      }
    }

    // make sure a file object was created
    if(!inf && !outf) {
      // error
      msgErr << "Unknown coordinate file format specified." << sendmsg;

    } else {
      // now initialize the file, i.e. open and read or write the header data
      MSGDEBUG(1,"Opening new coordinate file: " << name << sendmsg);

      if(is_input)
        opened = inf->init(name);
      else
        opened = outf->init(name);
    }
  }
}


///////////////////////////  destructor  /////////////////////////////
CoorFileData::~CoorFileData(void) {
  if(inf)  delete inf;
  if(outf) delete outf;
  delete [] name;
}


///////////////////////////  public routines  /////////////////////////////

// perform next operation, either read or write.  Return pointer to
// timestep read/written, or NULL if done.
Timestep *CoorFileData::next(void) {
  Timestep *newstep = NULL;
  if(opened) {
    if(is_input && inf) {
      // read next frame
      Timestep *tmpstep = NULL;
      do {
	if (tmpstep) delete tmpstep;  // delete the data if I don't need it
        newstep = inf->read();
	tmpstep = newstep;
        recentFrame++;
      } while(newstep && ((recentFrame - begFrame) % frameSkip)
			&& (endFrame < 0 || recentFrame <= endFrame));

      // if we have a keeper, append it to molecule
      if(newstep && !((recentFrame - begFrame) % frameSkip)
      		&& (endFrame < 0 || recentFrame <= endFrame))
        mol->append_frame(newstep);
	
      // check if quitting time
      if(!newstep || (endFrame >= 0 && recentFrame >= endFrame)) {
        delete inf;
        inf = NULL;
      }

    } else if(!is_input && outf) {
      // write next frame
      if(recentFrame < 0)
        recentFrame = begFrame;
      else
        recentFrame += frameSkip;

      // get next frame, and write to file
      if((endFrame<0 || recentFrame<=endFrame) && mol->num() >= recentFrame) {
        newstep = mol->item(recentFrame);
        outf->write(newstep);
      } else {
        // we're done; close file and stop writing
	delete outf;
	outf = NULL;
      }
    }
  }
  
  return newstep;
}

