/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Command.h,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.31 $	$Date: 95/12/19 17:03:51 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *     This file contains the base definition of the Command object.
 *  The idea is that all actions should be instantiated as
 *  a derivitive of a Command object and placed in a command queue.
 *  At every event loop, the commands are read off the current command
 *  queue and told to "execute".  When each command
 *  is finished, all of the UIs are notified that a command of some 
 *  "cmdtype" executed, so that the UI can update its information 
 *  if so desired.
 *
 *	Collections of related commands (i.e. command which all start with
 *  the same word, but are different variations) should be placed in files
 *  with the names 'Cmd*.C' and 'Cmd*.h'.  These files should define all the
 *  specific Command subclasses, as well as provide global 'text callback'
 *  functions which are called to process text commands which start with
 *  a specific word.
 *
 ***************************************************************************/
#ifndef COMMAND_H
#define COMMAND_H

#include <strstream.h>
#include <iostream.h>
#include "Inform.h"
class CommandQueue;


// typedef for the callback function to be used for new text commands
// this function is called when a command starting with a specific word
// is found; the arguments are the argc, argv pair for the command (which
// includes the initial word as argv[0])
// the function returns whether an error occurred (TRUE) or not (FALSE)
// arguments: number of args, arg char pointers, command queue to use, UI id
typedef int TextCallback(int, char **, CommandQueue *, int);



///////////////////////////// The Command object
// Derived classes must provide a unique constructor, destructor, and
// 'do_execute' routine, as well as optionally provide a 'create_text' routine
// if the command has a text equivalent which should be echoed to a log file.
class Command {

  public:
    // ALL the commands types must be in this enumeration.  The last element
    // must be "TOTAL" and the sequence must start at zero.  This is because
    // we reference Command::TOTAL to figure out how many commands there are.
    enum Cmdtype { NOTHING,
        WIN_EVENT, GUI_EVENT, TEXT_EVENT,
	HELP, RESET, QUIT, DEBUG, ECHO, WAIT, LOGWRITE, LOGREAD,
	ROTMAT, ROTATE, TRANSLATE, SCALE, ROCKON, ROCKOFF, STOPROT,
	ANIM_DIRECTION, ANIM_JUMP, ANIM_SKIP, ANIM_STYLE, ANIM_SPEED,
	ANIM_READ, ANIM_WRITE, ANIM_DELETE, ANIM_READ_DELETE_FILE,
	COLOR_NAME, COLOR_CHANGE, COLOR_SCALE_METHOD, COLOR_SCALE_MIDPOINT,
	COLOR_SCALE_MIN, COLOR_SCALE_MAX,
	DISP_RESHAPE, DISP_RESETVIEW, DISP_STEREO, DISP_PROJ, DISP_EYESEP, 
        DISP_FOCALLEN, DISP_LIGHT_ON, DISP_LIGHT_HL, DISP_LIGHT_ROT,
	DISP_CLIP, DISP_DEPTHCUE, DISP_ANTIALIAS, CMD_AXES, CMD_STAGE,
	DISP_SCRHEIGHT, DISP_SCRDIST, DISP_DETAIL, DISP_ALTDETAIL, DISP_UPDATE,
	EXTERNAL_START, EXTERNAL_END,
	LABEL_ADD, LABEL_LIST, LABEL_SHOW, LABEL_DELETE, LABEL_GRAPH,
	MENU_SHOW, MENU_MOVE, MENU_LOC,
	MOL_NEW, MOL_LIST, MOL_DEL, MOL_ACTIVE, MOL_FIX, MOL_ON, MOL_TOP,
	MOL_SELECT, MOL_REP, MOL_COLOR, MOL_ADDREP, MOL_MODREP, MOL_DELREP,
	MOL_MODREPITEM,
	MOUSE_MODE,
	PICK_START, PICK_MOVE, PICK_END,
	REMOTE_INIT, REMOTE_CANCEL, REMOTE_LIST, REMOTE_NEW, REMOTE_ATTACH,
	REMOTE_RUN, REMOTE_SETOPT, REMOTE_READOPT, REMOTE_WRITEOPT,
	REMOTE_CHANGE_SETTING,
	RENDER, RENDERLIST,
	SIGMA_EVENT,
	SIM_SETPARAM, SIM_PATCHDISP, SIM_TOP, SIM_DETACH, SIM_KILL, SIM_LIST,
	TOOL_INFO, TOOL_SIZE, TOOL_OFFSET, TOOL_PUSH, TOOL_POP,
	TOOL_REPLACE, TOOL_BUTTON, TOOL_COMMAND,
	TRACKER_START, TRACKER_PAUSE, TRACKER_LIST, TRACKER_LIST_AVAIL,
	TRACKER_LOC, TRACKER_UNPAUSE, TRACKER_SETPOS, TRACKER_MODPOS,
	TRACKER_SETORIENT, TRACKER_MODORIENT,
	USER_MENU_ADD, USER_MENU_LIST,
	TOTAL
    };

  private:
    Cmdtype mytype;	//  This is used to distinguish between "Commands"
    int UIid;		// the id of the UI that made me (or -1 for none)
              		// This is used if a UI wants a quick way to figure
			// out if the command came from itself

    int textExists;	// flag indicating if the text equiv has been created.
    int hasTextCmd;	// flag whether the command even HAS a text equiv
    char cmdtextbuf[256];	// storage for text rep of this command

    void check_and_create_text(void);	// creates text equiv, if necessary

  protected:
    // stream-based object to format text
    ostrstream *cmdText;

    // virtual function to allow each object to do a different action.
    // called by 'execute'.
    // By default, does nothing.
    virtual int do_execute(void);

    // virtual function which is called when a text version of the command
    // must be created to be printed to the console or a log file.  If a
    // special version is not provided, this just creates an empty string.
    virtual void create_text(void);

  public:
    // constructor ... command type, max length of text command, UI id
    // constructor for derived class should print text rep of this command
    // to the strstream 'cmdText'
    Command(Cmdtype, int);

    // destructor; free string spaced used for text rep of this command
    virtual ~Command(void);

    // Do the action specified for this command.  Return success.
    int execute(void);

    //
    // info about this Command
    //
    
    // return whether this command has a text equivalent
    int has_text(void);
    
    // return string with command text
    char *text(void);

    // return unique ID code of this command
    Cmdtype gettype( void) { return mytype; }

    // the id of the user interface which created this command.  If (-1),
    // unknown (or none).
    int getUIid(void) { return UIid; }        // get the UI id field

    //
    // routines to print out data
    //
    
    // print string with text rep of this command to the given ostream
    friend ostream& operator<<(ostream &o, Command &cmd) {
      if(cmd.hasTextCmd) {
        // need to construct the string if it has not been done so yet
        cmd.check_and_create_text();
        
        // now we must make sure the command has a text equiv
        if(cmd.hasTextCmd)
          o << cmd.cmdtextbuf;
      }
      return o;
    }
    
    // print string with text rep of this command to the given Inform obj
    friend Inform& operator<<(Inform &o, Command &cmd) {
      if(cmd.hasTextCmd) {
        // need to construct the string if it has not been done so yet
        cmd.check_and_create_text();
        
        // now we must make sure the command has a text equiv
        if(cmd.hasTextCmd)
          o << cmd.cmdtextbuf;
      }
      return o;
    }

};

#endif

