/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: CmdLabel.C,v $
 *      $Author: dalke $        $Locker:  $                $State: Exp $
 *      $Revision: 1.9 $      $Date: 95/09/23 18:38:58 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Command objects used to create, list, delete, or graph labels for measuring
 * geometries.
 *
 ***************************************************************************/

#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include "CmdLabel.h"
#include "GeometryList.h"
#include "GeometryAtom.h"
#include "GeometryBond.h"
#include "GeometryAngle.h"
#include "GeometryDihedral.h"
#include "GeometryTug.h"
#include "MoleculeList.h"
#include "Molecule.h"
#include "CommandQueue.h"
#include "Global.h"
#include "utilities.h"
#include "Inform.h"

// The following uses the Cmdtypes LABEL_ADD, LABEL_SHOW, LABEL_LIST,
// LABEL_GRAPH and LABEL_DELETE from the Command class


// return codes for atom name string
enum AtomNameReturnCodes { ANAME_ILLEGAL, ANAME_BADMOL, ANAME_BADSEG,
	ANAME_BADRES, ANAME_BADATOM, ANAME_OK };


////////////////////////////////////////////////////////////////////
///////////////////////  text processors
////////////////////////////////////////////////////////////////////

// text callback routine for 'label'; return TRUE if an error occurs.
int text_cmd_label(int argc, char **argv, CommandQueue *cmdQueue, int id) {

  if(!strupncmp(argv[1], "add", CMDLEN)) {
    if(argc > 3)
      cmdQueue->append(new CmdLabelAdd(argv[2], argc-3, argv+3, id));
    else
      return TRUE;

  } else if(!strupncmp(argv[1], "list", CMDLEN)) {
    if(argc == 3)
      cmdQueue->append(new CmdLabelList(argv[2], id));
    else
      return TRUE;

  } else if(!strupncmp(argv[1], "show", CMDLEN) ||
	    !strupncmp(argv[1], "hide", CMDLEN)) {
    int item;
    if(argc == 3 || (argc == 4 && !strupncmp(argv[3], "all", CMDLEN)))
      item = (-1);
    else if(argc == 4)
      item = atoi(argv[3]);
    else
      return TRUE;
    
    cmdQueue->append(new CmdLabelShow(argv[2],
		 !strupncmp(argv[1], "show", CMDLEN), item, id));

  } else if(!strupncmp(argv[1], "delete", CMDLEN)) {
    int item;
    if(argc == 3 || (argc == 4 && !strupncmp(argv[3], "all", CMDLEN)))
      item = (-1);
    else if(argc == 4)
      item = atoi(argv[3]);
    else
      return TRUE;

    cmdQueue->append(new CmdLabelDelete(argv[2], item, id));

  } else if(!strupncmp(argv[1], "graph", CMDLEN) && argc > 3) {
    int item = atoi(argv[3]);
    char graphcmd[128];
    *graphcmd = '\0';
    for(int i=4; i < argc; i++) {
      strcat(graphcmd, argv[i]);
      strcat(graphcmd, " ");
    }
    cmdQueue->append(new CmdLabelGraph(argv[2], item, graphcmd, id));

  } else
    return TRUE;

  return FALSE;
}


// utility routine: return mol ID and atom index given a string
// representing an atom.
// If returns == ANAME_OK, name was OK and atom index found.
// If returns == ANAME_ILLEGAL, illegal specification;
// If returns == ANAME_BADMOL, unknown molecule ID
// If returns == ANAME_BADSEG, unknown segment specification
// If returns == ANAME_BADRES, unknown residue specification
// If returns == ANAME_BADATOM, unknown atom name
int find_atom_from_name(char *checkstr, int *mid, int *aid) {
  register int i;
  int mol_id, colons = 0, dashes = 0;
  Molecule *mol;
  char *segname, *resid, *namestart;
  char astr[128];
  int strformat = (-1);

  // make sure the string is in one of two formats:
  //	1) there are 1..2 colons, and 1 dash
  //	2) there is 1 forward slash, with a number before and after the slash
  if(checkstr) {

    // make a copy of the string, since we'll modify things
    strcpy(astr, checkstr);

    // first check for a slash
    char *tmpstr = astr;
    while(*tmpstr) {
      if(*tmpstr == '/') {
        strformat = 0;
        namestart = tmpstr + 1;
	break;
      }
      tmpstr++;
    }
    
    // if a slash was found, make sure there are two numbers
    if(strformat == 0 && (!isdigit(*astr) || !isdigit(*namestart)))
      strformat = (-1);		// not numeric; unknown format

    // check for other format if necessary
    if(strformat < 0) {
      tmpstr = astr;
      while(*tmpstr) {
        if(*tmpstr == ':') {
          colons++;
  	  *tmpstr = '\0';
  	  if(colons == 1)
  	    segname = tmpstr + 1;
  	  else if(colons == 2)
  	    namestart = tmpstr + 1;
        } else if(*tmpstr == '-') {
          dashes++;
  	  *tmpstr = '\0';
  	  resid = tmpstr + 1;
        }
        tmpstr++;
      }
    }
    
    // make sure the 2nd format was found
    if(dashes == 1 && colons >= 1 && colons <= 2)
      strformat = 1;
  }

  // return error if unknown format
  if(strformat < 0)
    return ANAME_ILLEGAL;

  // find the molecule ID, if present, and the atom index
  if(strformat == 0) {
    // check molecule id
    mol_id = atoi(astr);
    mol = moleculeList->molecule(moleculeList->mol_index_from_id(mol_id));
    if(!mol)
      return ANAME_BADMOL;

    // check atom index
    int atomindex = atoi(namestart);
    if(atomindex < 0 || atomindex >= mol->nAtoms)
      return ANAME_BADATOM;

    // everythings OK, return values
    *mid = mol_id;
    *aid = atomindex;
    return ANAME_OK;

  } else {
    if(colons == 2 && isdigit(*astr)) {
      mol_id = atoi(astr);
      mol = moleculeList->molecule(moleculeList->mol_index_from_id(mol_id));
      if(!mol)
        return ANAME_BADMOL;
    } else if(colons == 1 && moleculeList->top()) {
      mol = moleculeList->top();
      mol_id = mol->id();
      namestart = segname;
      segname = astr;
    } else 
      return ANAME_BADMOL;

    // now find the indices for the atom names
    int segnameindex = (mol->segNames).typecode(segname);
    if(segnameindex < 0)
      return ANAME_BADSEG;
    int residindex = (mol->resIds).typecode(resid);
    if(residindex < 0)
      return ANAME_BADRES;
    int atomindex = (mol->atomNames).typecode(namestart);
    if(atomindex < 0)
      return ANAME_BADATOM;

    // finally search through the molecule for the proper atom
    for(i=(mol->nAtoms - 1); i >= 0; i--) {
      MolAtom *atom = mol->atom(i);
      if(atom->segnameindex==segnameindex && atom->residindex==residindex &&
      	atom->nameindex==atomindex) {
        *mid = mol_id;
        *aid = i;
        return ANAME_OK;
      }
    }
  }
  
  // if here, no atom matched.  return bad atom error
  return ANAME_BADATOM;
}


// check the given category name, and return its index if OK, -1 otherwise
int check_geom_cat_name(char *geomcat) {
  int retval = geometryList->geom_list_index(geomcat);
  if(retval < 0)
    msgErr << "Unknown geometry category '" << geomcat << "'." << sendmsg;
  return retval;
}


// print out name and value in common format
void display_label_name(int n, Geometry *g) {
  msgInfo << " ";
  if(n >= 0)
    msgInfo << n << ": ";
  msgInfo << g->name();
  if(g->has_value())
    msgInfo << " = " << g->calculate();
}


////////////////////// add a new label 
CmdLabelAdd::CmdLabelAdd(char *geomcat, int n, char **items, int UIid) :
	Command(LABEL_ADD, UIid) {
  // NOTE: this creates the proper Geometry object; if it cannot do this,
  //	error messages are printed, and no Geometry obj is created.

  cat = (-1);
  geom = NULL;

  if(n < 1) {
    msgErr << "No objects specified to be labeled." << sendmsg;
  } else {  
    // find geometry category, if known
    cat = check_geom_cat_name(geomcat);
    if(cat >= 0 && cat < 4) {
      // find molecule id's and atom names
      int *middata = new int[n];
      int *atmdata = new int[n];
      int i;
      for(i=0; i < n; i++) {
        int errcode = find_atom_from_name(items[i], middata+i, atmdata+i);
        char *errmsg = NULL;
        switch(errcode) {
          case ANAME_ILLEGAL: errmsg="Illegal atom name syntax"; break;
	  case ANAME_BADMOL:  errmsg="Unknown molecule in atom name"; break;
	  case ANAME_BADRES:  errmsg="Unknown residue name in atom name"; break;
	  case ANAME_BADATOM: errmsg = "Unknown atom name"; break;
        }
        if(errmsg) {
          msgErr << errmsg << ": '" << items[i] << "'" << sendmsg;
	  break;		// from for loop
        }
      }
    
      if(i == n) {
        if(cat == geometryList->geom_list_index("Atoms") && n == 1)
          geom = new GeometryAtom(middata[0], atmdata[0], moleculeList);
        else if(cat == geometryList->geom_list_index("Bonds") && n == 2)
          geom = new GeometryBond(middata, atmdata, moleculeList);
        else if(cat == geometryList->geom_list_index("Angles") && n == 3)
          geom = new GeometryAngle(middata, atmdata, moleculeList);
        else if(cat == geometryList->geom_list_index("Dihedrals") && n == 4)
          geom = new GeometryDihedral(middata, atmdata, moleculeList);
      }

      delete [] middata;
      delete [] atmdata;
    } else if (cat < 6) {
      // Tugs or Restraints
      // For now, parameters are 'label add Tugs mol-id atom-# x y z'
      msgInfo << "CmdLabelAdd: args are:";
      for(int i=0; i < n; i++)
        msgInfo << " " << items[i];
      msgInfo << sendmsg;

      if (n >= 5) {
        int molid = atoi(items[0]),
            atomid = atoi(items[1]);
        float pos[3];
        pos[0] = atof(items[2]);
        pos[1] = atof(items[3]);
        pos[2] = atof(items[4]);

        geom = new GeometryTug(molid, atomid, pos, moleculeList);
      }
    }
  }

  *cmdText << "label add " << geomcat;
  for(int j=0; j < n; j++)
    *cmdText << " " << items[j];
  *cmdText << ends;
}

void CmdLabelAdd::create_text(void) {
  // does nothing, string created in constructor since it's easier here
}

int CmdLabelAdd::do_execute(void) {

  if(cat >= 0 && geom != NULL) {
    int retval = geometryList->add_geometry(cat, geom);
    if(retval <= 0) {
      if(retval == 0)
        msgErr << "Could not add new label " << geom->name() << sendmsg;
      delete geom;
    } else {
      msgInfo << "Added new " << geometryList->geom_list_name(cat) << " label";
      display_label_name((-1), geom);
      msgInfo << sendmsg;
    }
    return retval; 
  } else {
    msgErr << "Could not add new label." << sendmsg;
  }
  
  // if here, command failed
  return FALSE;
}


////////////////////// list the labels in a category
CmdLabelList::CmdLabelList(char *geomcat, int UIid) :
	Command(LABEL_LIST, UIid) {

  // find geometry category, if known
  cat = check_geom_cat_name(geomcat);

  *cmdText << "label list " << geomcat << ends;
}

void CmdLabelList::create_text(void) {
  // does nothing, string created in constructor since it's easier here
}

int CmdLabelList::do_execute(void) {

  if(cat >= 0) {
    GeomListPtr glist = geometryList->geom_list(cat);
    int gnum = glist->num();
    msgInfo << "Labels in category '" << geometryList->geom_list_name(cat);
    msgInfo << "': " << gnum << sendmsg;
    for(int i=0; i < gnum; i++) {
      display_label_name(i, (*glist)[i]);
      msgInfo << sendmsg;
    }
    
    return TRUE;
  }
  
  // if here, command failed
  return FALSE;
}


////////////////////// toggle a geometry category on/off
CmdLabelShow::CmdLabelShow(char *geomcat, int s, int n, int UIid) :
	Command(LABEL_SHOW, UIid) {

  // find geometry category, if known
  cat = check_geom_cat_name(geomcat);
  show = s;
  item = n;

  *cmdText << "label " << (show ? "show" : "hide") << " " << geomcat;
  if(n >= 0)
    *cmdText << " " << n;
  *cmdText << ends;
}

void CmdLabelShow::create_text(void) {
  // does nothing, string created in constructor since it's easier here
}

int CmdLabelShow::do_execute(void) {

  if(cat >= 0) {
    geometryList->show_geometry(cat, item, show);
    return TRUE;
  }
  
  // if here, command failed
  return FALSE;
}

//////////////////////// delete a label
CmdLabelDelete::CmdLabelDelete(char *geomcat, int n, int UIid) :
	Command(LABEL_DELETE, UIid) {

  // find geometry category, if known
  cat = check_geom_cat_name(geomcat);
  item = n;

  *cmdText << "label delete " << geomcat;
  if(n >= 0)
    *cmdText << " " << n;
  *cmdText << ends;
}

void CmdLabelDelete::create_text(void) {
  // does nothing, string created in constructor since it's easier here
}

int CmdLabelDelete::do_execute(void) {

  if(cat >= 0) {
    return geometryList->del_geometry(cat, item);
  }

  // if here, command failed
  return FALSE;
}


///////////////// graph the values for the Nthe label in the given category
CmdLabelGraph::CmdLabelGraph(char *geomcat, int n, char *gcmd, int UIid) :
	Command(LABEL_GRAPH, UIid) {

  // find geometry category, if known
  cat = check_geom_cat_name(geomcat);
  item = n;

  // save graphing command
  graphCmd = stringdup(gcmd);
  
  *cmdText << "label graph " << geomcat << " " << n << " " << graphCmd << ends;
}

void CmdLabelGraph::create_text(void) {
  // does nothing, string created in constructor since it's easier here
}

int CmdLabelGraph::do_execute(void) {

  if(cat >= 0) {
    // get geometry list
    GeomListPtr glist = geometryList->geom_list(cat);
    int gnum = glist->num();
    if(item < 0 || item >= gnum) {
      msgErr << "Illegal label number " << item << sendmsg;
      return FALSE;
    }
    
    // get geometry object; return error if label has no values
    Geometry *g = (*glist)[item];
    if(g->has_value()) {

      // fill array with list of geom values; return error if cannot calc all
      ResizeArray<float> gValues(1024);
      if(g->calculate_all(gValues)) {

        // print or graph data
        if(! strlen(graphCmd)) {
          // if go graph command, write data to console
          msgInfo << "Values for label";
          display_label_name(item, g);
          msgInfo << sendmsg;
          for(int i=0; i < gValues.num(); i++) {
	    msgInfo << "  "  << gValues[2*i] << "  " << gValues[2*i + 1];
	    msgInfo << sendmsg;
          }
        } else {
          // if a graphing command is given, write data to temp file and run
          // the command via 'system'
      
          // first create a temporary filename
          char filename[128];
          tmpnam(filename);
      
          // now open the file for writing
          FILE *outfile = fopen(filename, "w");
          if (!outfile) {
            msgErr << "Cannot open file for graph output." << sendmsg;
            return FALSE;
          }
  start_wait(); // turn the WAIT cursor on
          // and write the values into it
          int values = (gValues.num() / 2);
          for(int i=0; i < values; i++)
            fprintf(outfile,"%f  %f\n", gValues[2*i], gValues[2*i + 1]);
          fclose(outfile);

          // exec the graphing command, giving the filename as an optionally
          // used argument
          char execstr[128];
          sprintf(execstr, graphCmd, filename, filename, filename, filename);
	  double beforeCmd = time_of_day();
          system(execstr);

          // make sure there is at least three seconds before continuing
	  double afterCmd = time_of_day();
	  if(afterCmd - beforeCmd < 2.0)
            sleep(3 - (int)(afterCmd - beforeCmd));
      
          // finally delete the file
          unlink(filename);
  stop_wait();
        }
      
        // we were successful
        return TRUE;
      }
    }

    // if here, could not graph properly
    msgErr << "Unable to graph data." << sendmsg;
  }
  
  // but if we're here, we were not successful
  return FALSE;
}


