/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: CmdAnimate.C,v $
 *      $Author: billh $        $Locker:  $                $State: Exp $
 *      $Revision: 1.5 $      $Date: 95/05/11 22:01:21 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Command objects for doing animation.
 *
 ***************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "CmdAnimate.h"
#include "MolAction.h"
#include "CommandQueue.h"
#include "Global.h"
#include "utilities.h"

// The following uses the Cmdtypes:
//	ANIM_DIRECTION, ANIM_JUMP, ANIM_SKIP, ANIM_STYLE, ANIM_SPEED,
//	ANIM_READ, ANIM_WRITE, ANIM_DELETE, ANIM_READ_DELETE_FILE

////////////////////////////////////////////////////////////////////
///////////////////////  text processors
////////////////////////////////////////////////////////////////////

// text callback routine for 'animate'; return TRUE if an error occurs.
int text_cmd_animate(int argc, char **argv, CommandQueue *cmdQueue, int id) {
  if(argc == 2) {
    Animation::AnimDir newDir;
    if(!strupncmp(argv[1], "reverse", CMDLEN) ||
    	!strupncmp(argv[1], "rev", CMDLEN))
      newDir = Animation::REVERSE;
    else if(!strupncmp(argv[1], "forward", CMDLEN) ||
    	!strupncmp(argv[1], "for", CMDLEN))
      newDir = Animation::FORWARD;
    else if(!strupncmp(argv[1], "prev", CMDLEN))
      newDir = Animation::REVERSE1;
    else if(!strupncmp(argv[1], "next", CMDLEN))
      newDir = Animation::FORWARD1;
    else if(!strupncmp(argv[1], "pause", CMDLEN))
      newDir = Animation::FORWARD1;
    else
      return TRUE;		// error
    cmdQueue->append(new CmdAnimDir(newDir, id));
  } else if(argc == 3) {
    if(!strupncmp(argv[1], "skip", CMDLEN))
      cmdQueue->append(new CmdAnimSkip(atoi(argv[2]),id));
    else if(!strupncmp(argv[1], "delete", CMDLEN)) {
      if(!strupncmp(argv[2], "all", CMDLEN))
        cmdQueue->append(new CmdAnimDelete(-1, -1, -1, -1, id));
      else
        return TRUE;
    } else if(!strupncmp(argv[1], "speed", CMDLEN))
      cmdQueue->append(new CmdAnimSpeed(atof(argv[2]),id));
    else if(!strupncmp(argv[1], "style", CMDLEN)) {
      int newStyle = Animation::ONCE;
      Animation::AnimStyle enumVal;
      while(newStyle < Animation::TOTAL_STYLES) {
        if(!strupncmp(argv[2], animationStyleName[newStyle], CMDLEN))
	  break;
	newStyle++;
      }
      if(newStyle == Animation::ONCE)
        enumVal = Animation::ONCE;
      else if(newStyle == Animation::ROCK)
        enumVal = Animation::ROCK;
      else if(newStyle == Animation::LOOP)
        enumVal = Animation::LOOP;
      else
        return TRUE;		// error, unknown style
      cmdQueue->append(new CmdAnimStyle(enumVal, id));
    } else if(!strupncmp(argv[1], "goto", CMDLEN)) {
      int newframe;
      if(!strupncmp(argv[2], "start", CMDLEN))
        newframe = SET_FRAME_START;
      else if(!strupncmp(argv[2], "end", CMDLEN))
        newframe = SET_FRAME_END;
      else if(isdigit(argv[2][0]))
        newframe = atoi(argv[2]);
      else
        return TRUE;		// error, bad frame goto command
      cmdQueue->append(new CmdAnimJump(newframe, id));
    } else
      return TRUE;
  } else if(argc >= 4) {
    int bf = (-1), ef = (-1), fs = (-1), mid = (-1);
    int fileType = CoorFileData::UNKNOWN;
    char *fileName = NULL;
    int do_action = (-1);
    int currarg = 1;

    // find out what to do first
    if(!strupncmp(argv[currarg], "read", CMDLEN)) {
      do_action = 0;
    } else if(!strupncmp(argv[currarg], "write", CMDLEN)) {
      do_action = 1;
    } else if(!strupncmp(argv[currarg], "delete", CMDLEN)) {
      do_action = 2;
    } else if (!strupncmp(argv[currarg], "readdel", CMDLEN)) {
      do_action = 3;
    } else
      return TRUE;
    currarg++;

    // if reading or writing, get file type and name
    if(do_action == 0 || do_action == 1 || do_action == 3) {
      for(int i=0; i < CoorFileData::COORTYPES; i++)
        if(!strupncmp(argv[currarg], CoorFileSuffix[i], CMDLEN))
	  fileType = i;
      if(fileType == CoorFileData::UNKNOWN) {
        msgErr << "Unknown coordinate file type " << argv[currarg];
	msgErr << sendmsg;
	return TRUE;
      }
      currarg++;
      fileName = argv[currarg++];
    }
    
    // find if any beg, end, or skip specifiers
    while(currarg < argc) {
      if(currarg < (argc - 1)) {
        if(!strupncmp(argv[currarg], "beg", CMDLEN)) {
	  bf = atoi(argv[currarg+1]);
	  currarg += 2;
	} else if(!strupncmp(argv[currarg], "end", CMDLEN)) {
	  ef = atoi(argv[currarg+1]);
	  currarg += 2;
	} else if(!strupncmp(argv[currarg], "skip", CMDLEN)) {
	  fs = atoi(argv[currarg+1]);
	  currarg += 2;
	} else if(do_action == 2 && argc == 4 && currarg == 2 &&
			!strupncmp(argv[currarg], "all", CMDLEN)) {
	  mid = atoi(argv[currarg+1]);
	  bf = ef = fs = (-1);
	  currarg += 2;
	} else
	  return TRUE;
      } else {
        // only one item left; it must be the molecule id
	mid = atoi(argv[currarg++]);
      }
    }
    
    // do action now
    if(do_action == 0)
      cmdQueue->append(new CmdAnimReadFile(mid,fileName,fileType,
				bf,ef,fs, id));
    else if(do_action == 1)
      cmdQueue->append(new CmdAnimWriteFile(mid,fileName,fileType,
				bf,ef,fs, id));
    else if(do_action == 2)
      cmdQueue->append(new CmdAnimDelete(mid,bf,ef,fs, id));
    else if(do_action == 3)
      cmdQueue->append(new CmdAnimReadFileDelete(mid,fileName,fileType,
				bf, ef,fs, id));
    else
      return TRUE;
  } else
    return TRUE;
    
  // if here, everything worked out ok
  return FALSE;
}


//////////////////// set direction of animation
int CmdAnimDir::do_execute(void) {
  int retval;
  if(retval = (moleculeList != NULL)) {
    if(newDir == Animation::REVERSE) {
      MolReverse action;
      moleculeList->act(action, whichMol);
    } else if(newDir == Animation::REVERSE1) {
      MolReverse1 action;
      moleculeList->act(action, whichMol);
    } else if(newDir == Animation::FORWARD) {
      MolForward action;
      moleculeList->act(action, whichMol);
    } else if(newDir == Animation::FORWARD1) {
      MolForward1 action;
      moleculeList->act(action, whichMol);
    } else if(newDir == Animation::PAUSE) {
      MolPause action;
      moleculeList->act(action, whichMol);
    }
  }
  return retval;
}

void CmdAnimDir::create_text(void) {
  *cmdText << "animate ";
  if(newDir == Animation::REVERSE)
    *cmdText << "reverse";
  else if(newDir == Animation::REVERSE1)
    *cmdText << "prev";
  else if(newDir == Animation::FORWARD)
    *cmdText << "forward";
  else if(newDir == Animation::FORWARD1)
    *cmdText << "next";
  else if(newDir == Animation::PAUSE)
    *cmdText << "pause";
  *cmdText << ends;
}

CmdAnimDir::CmdAnimDir(Animation::AnimDir ad, int newUIid)
  : Command(Command::ANIM_DIRECTION, newUIid) {
  newDir = ad;
  whichMol = MoleculeList::ACTIVE;
}


//////////////////// set style of animation
int CmdAnimStyle::do_execute(void) {
  int retval;
  if(retval = (moleculeList != NULL)) {
    MolAnimStyle action(newStyle);
    moleculeList->act(action, whichMol);
  }
  return retval;
}

void CmdAnimStyle::create_text(void) {
  *cmdText << "animate style " << animationStyleName[newStyle] << ends;
}

CmdAnimStyle::CmdAnimStyle(Animation::AnimStyle as, int newUIid)
  : Command(Command::ANIM_STYLE, newUIid) {
  newStyle = as;
  whichMol = MoleculeList::ACTIVE;
}


//////////////////// jump to a new frame
int CmdAnimJump::do_execute(void) {
  int retval;
  if(retval = (moleculeList != NULL)) {
    MolSetFrame action(newFrame);
    moleculeList->act(action, whichMol);
  }
  return retval;
}

void CmdAnimJump::create_text(void) {
  *cmdText << "animate goto " << newFrame << ends;
}

CmdAnimJump::CmdAnimJump(int newval, int newUIid)
  : Command(Command::ANIM_JUMP, newUIid) {
  newFrame = newval;
  whichMol = MoleculeList::ACTIVE;
}


//////////////////// set frame skip value
int CmdAnimSkip::do_execute(void) {
  int retval;
  if(retval = (moleculeList != NULL)) {
    MolSetSkip action(newSkip);
    moleculeList->act(action, whichMol);
  }
  return retval;
}

void CmdAnimSkip::create_text(void) {
  *cmdText << "animate skip " << newSkip << ends;
}

CmdAnimSkip::CmdAnimSkip(int newval, int newUIid)
  : Command(Command::ANIM_SKIP, newUIid) {
  newSkip = newval;
  whichMol = MoleculeList::ACTIVE;
}


//////////////////// set animation speed
int CmdAnimSpeed::do_execute(void) {
  int retval;
  if(retval = (moleculeList != NULL)) {
    MolSetSpeed action(newSpeed);
    moleculeList->act(action, whichMol);
  }
  return retval;
}

void CmdAnimSpeed::create_text(void) {
  *cmdText << "animate speed " << newSpeed << ends;
}

CmdAnimSpeed::CmdAnimSpeed(float newval, int newUIid)
  : Command(Command::ANIM_SPEED, newUIid) {
  newSpeed = newval;
  whichMol = MoleculeList::ACTIVE;
}


//////////////////// append new frames from a file
int CmdAnimReadFile::do_execute(void) {
  int retval;
  Molecule *m;
  if(retval = (moleculeList != NULL)) {
    if(whichMol >= 0)
      m = moleculeList->molecule(moleculeList->mol_index_from_id(whichMol));
    else
      m = moleculeList->top();
    if(!m) {
      msgErr << "Illegal molecule ID " << whichMol << " specified." <<sendmsg;
      retval = FALSE;
    } else {
      retval=m->read_coor_file(fileName,fileType,begFrame,endFrame,frameSkip);
      if(!retval) {
        msgErr << "Unable to open coordinate file " << fileName;
        msgErr << " for reading." << sendmsg;
      } else {
        msgInfo << "Opened coordinate file " << fileName << " for reading.";
        msgInfo << sendmsg;
      }
    }
  }
  return retval;
}

void CmdAnimReadFile::create_text(void) {
  *cmdText << "animate read " << CoorFileSuffix[fileType] << " " << fileName;
  *cmdText << " beg " << begFrame;
  *cmdText << " end " << endFrame;
  *cmdText << " skip " << frameSkip;
  if(whichMol >= 0)
    *cmdText << " " << whichMol;
  *cmdText << ends;
}

CmdAnimReadFile::CmdAnimReadFile(int m,char *n,int t,int bf,int ef,int fs,
  			int newUIid) : Command(Command::ANIM_READ, newUIid) {
  whichMol = m;
  fileType = t;
  begFrame = bf;
  endFrame = ef;
  frameSkip = fs;
  fileName = stringdup(n);
}

CmdAnimReadFile::~CmdAnimReadFile(void) {
  delete [] fileName;
}


/////// read frames from a file, then delete the file
/////// This is used by the Babel conversion and is an undocumented
/// feature in the UIText

int CmdAnimReadFileDelete::do_execute(void) {
  int retval;
  Molecule *m;
  if(retval = (moleculeList != NULL)) {
    if(whichMol >= 0)
      m = moleculeList->molecule(moleculeList->mol_index_from_id(whichMol));
    else
      m = moleculeList->top();
    if(!m) {
      msgErr << "Illegal molecule ID " << whichMol << " specified." <<sendmsg;
      retval = FALSE;
    } else {
      retval=m->read_coor_file(fileName,fileType,begFrame,endFrame,frameSkip);
      unlink(fileName);
      if(!retval) {
        msgErr << "Unable to open coordinate file " << fileName;
        msgErr << " for reading." << sendmsg;
      }
    }
  }
  return retval;
}

CmdAnimReadFileDelete::CmdAnimReadFileDelete(int m, char *n, int t,
int bf, int ef, int fs, int newUIid) : Command(Command::ANIM_READ, newUIid) {
  whichMol = m;
  fileType = t;
  begFrame = bf;
  endFrame = ef;
  frameSkip = fs;
  fileName = stringdup(n);
}

CmdAnimReadFileDelete::~CmdAnimReadFileDelete(void) {
  delete [] fileName;
}



//////////////////// write frames to a file
int CmdAnimWriteFile::do_execute(void) {
  int retval;
  Molecule *m;
  if(retval = (moleculeList != NULL)) {
    if(whichMol >= 0)
      m = moleculeList->molecule(moleculeList->mol_index_from_id(whichMol));
    else
      m = moleculeList->top();
    if(!m) {
      msgErr << "Illegal molecule ID " << whichMol << " specified." <<sendmsg;
      retval = FALSE;
    } else {
      retval =
	m->write_coor_file(fileName,fileType,begFrame,endFrame,frameSkip);
      if(!retval) {
        msgErr << "Unable to open coordinate file " << fileName;
        msgErr << " for writing." << sendmsg;
      } else {
        msgInfo << "Opened coordinate file " << fileName << " for writing.";
        msgInfo << sendmsg;
      }
    }
  }
  return retval;
}

void CmdAnimWriteFile::create_text(void) {
  *cmdText << "animate write " << CoorFileSuffix[fileType] << " " << fileName;
  *cmdText << " beg " << begFrame;
  *cmdText << " end " << endFrame;
  *cmdText << " skip " << frameSkip;
  if(whichMol >= 0)
    *cmdText << " " << whichMol;
  *cmdText << ends;
}

CmdAnimWriteFile::CmdAnimWriteFile(int m,char *n,int t,int bf,int ef,int fs,
  			int newUIid) : Command(Command::ANIM_WRITE, newUIid) {
  whichMol = m;
  fileType = t;
  begFrame = bf;
  endFrame = ef;
  frameSkip = fs;
  fileName = stringdup(n);
}

CmdAnimWriteFile::~CmdAnimWriteFile(void) {
  delete [] fileName;
}


//////////////////// delete frames
int CmdAnimDelete::do_execute(void) {
  int retval;
  Molecule *m;
  if(retval = (moleculeList != NULL)) {
    if(whichMol >= 0)
      m = moleculeList->molecule(moleculeList->mol_index_from_id(whichMol));
    else
      m = moleculeList->top();
    if(!m) {
      msgErr << "Illegal molecule ID " << whichMol << " specified." << sendmsg;
      retval = FALSE;
    } else {
      // delete specified frames from this molecule
      int eframe = (endFrame >= 0 ? endFrame : m->num() - 1);
      int bframe = (begFrame >= 0 ? begFrame : 0);
      int fskip = (frameSkip >= 1 ? frameSkip : 1);
      int deleted = 0;
      while(bframe <= eframe) {
        m->delete_frame(bframe);
	eframe--;
	bframe += (fskip - 1);
	deleted++;
      }
      msgInfo << "Finished deleting " << deleted << " frames from molecule ";
      msgInfo << m->name << sendmsg;
    }
  }
  return retval;
}

void CmdAnimDelete::create_text(void) {
  *cmdText << "animate delete ";
  if(begFrame < 0 && endFrame < 0 && frameSkip < 0) {
    *cmdText << "all";
  } else {
    *cmdText << " beg " << begFrame;
    *cmdText << " end " << endFrame;
    *cmdText << " skip " << frameSkip;
  }
  if(whichMol >= 0)
    *cmdText << " " << whichMol;
  *cmdText << ends;
}

CmdAnimDelete::CmdAnimDelete(int m, int bf, int ef, int fs, int newUIid)
	: Command(Command::ANIM_DELETE, newUIid) {
  whichMol = m;
  begFrame = bf;
  endFrame = ef;
  frameSkip = fs;
}

