/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: AtomRep.h,v $
 *      $Author: dalke $        $Locker:  $                $State: Exp $
 *      $Revision: 1.16 $      $Date: 96/02/01 23:25:14 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Parse and maintain the data for how a molecule should be represented.
 *
 ***************************************************************************/
#ifndef ATOMREP_H
#define ATOMREP_H

#include "utilities.h"

// default atom representation, and max cmd string size
#define DEFAULT_ATOMREP		AtomRep::LINES
#define MAX_ATOMREP_CMD		255
#define MAX_ATOMREP_DATA        5

#ifdef VMDALPHA_SHAPE
// This is not yet available outside of the Theoretical Biophysics
// Group and is still under development.
class AlphaShape;
#endif
#ifdef VMDSURF
// The SURF executable can be retreived from ftp.cs.unc.edu in /pub/GRIP
class Surf;
#endif


class AtomRep {

public:
  // different methods for representing atoms
  enum RepMethod { LINES, BONDS, CPK, POINTS, VDW, DOTTED, LICORICE, RIBBONS, 
		   TUBE, STRUCTURE, HBONDS, 
#ifdef VMDALPHA_SHAPE
		   ALPHASHAPE,
#endif
#ifdef VMDSURF
		   SURF,
#endif
		   REPOFF, TOTAL };

  // these are the same as above, but for historical reasons
  enum RepDataNames { SPHERERAD, BONDRAD, SPHERERES, BONDRES, LINETHICKNESS };

  // initial string with representation command
  char cmdStr[MAX_ATOMREP_CMD + 1];

private:
  // results of command ... how to represent atoms, and some parameters
  int repMethod;
  float repDataStorage[TOTAL][MAX_ATOMREP_DATA];
  float *repData;

  // parse the given command, and store results.  Return success.
  int parse_cmd(char *);

public:
  AtomRep(void);
  AtomRep(AtomRep &ar) { copy(ar); }
  ~AtomRep(void);
  
  // equal operator, to change the current settings.
  AtomRep& copy(AtomRep &);
  AtomRep& operator=(AtomRep &ar) { return copy(ar); }

  // provide new settings
  int change(char *newcmd) { return parse_cmd(newcmd); }

  // set the value of a specific parameter
  void set_data(int n, float newval) {
    if(repData[n] != newval) {
      repData[n] = newval;
      option_changed = TRUE;
    }
  }

  //
  // info about current settings
  //

  // return representation method
  int method(void) { return repMethod; }

  // return value of Nth data item
  float get_data(int n) { return repData[n]; }

  //
  // old-style data retrieval routines (should be removed someday)
  //
  
  // return sphere radius (may not be applicable)
  float sphere_rad(void) { return repData[SPHERERAD]; }
  
  // return bond radius (may not be applicable)
  float bond_rad(void) { return repData[BONDRAD]; }

  // return sphere res (may not be applicable)
  int sphere_res(void) { return (int)(repData[SPHERERES] + 0.01); }
  
  // return bond res (may not be applicable)
  int bond_res(void) { return (int)(repData[BONDRES] + 0.01); }
  
  // return line thickness (may not be applicable)
  int line_thickness(void) { return (int)(repData[LINETHICKNESS] + 0.01); }

  int option_changed; // set TRUE if the data element changed, so that 
     // representations (like ALPHA, if enabled) dont always recompute
     // the shape when the probe radius value change

#ifdef VMDALPHA_SHAPE
  AlphaShape *alpha;   // the alpha shape

     // Hmm, after a few hours of trying I realized that the following is
     // a very hard thing to do in the current scheme.  The <Add to
     // Selected> doesnt really know what the current selection is
     // about, so there isnt really a way to pass the pointer about.
     // I leave alpha_count here so we can use it during the next/
     // rewrite.
  int *alpha_count;    // During an "Add to [selected|alpha]" of an
                       // alpha shape I let the copy point to the same
		       // file space.  Therefore, I need to keep a reference
		       // count so I do not delete the alpha when something 
		       // else needs it
  void delete_alpha(void);
  void set_alpha(AlphaShape *newalpha);
#endif  
#ifdef VMDSURF
  Surf *surf;
  int *surf_count;
  void set_surf(Surf *newsurf);
  void delete_surf(void);
#endif
};


//
// global structures and variables defining the data each rep requires
//

// The following structure is used to define how each individual data
// value is used by each rep style.  Each style uses a number of different
// data items to define that rep; this structure defines the type and range
// of values allowed for those items.
// Each AtomRep has storage for 5 optional data values.
// The order they are specified is defined by each rep style.
typedef struct {
  char *name;                // name of this data value
  int index;                 // which one of the 6 storage slots to use
  float defValue;            // default value for the item
  int sigDigits;             // # signif. digits; if 0, integer; if < 0, undef
  int hasmin, hasmax;        // flags for whether this item has limited range
  float min, max;            // minimum and maximum values
  float mininc, maxinc;      // small and large increment amounts for GUI
} AtomRepDataInfo;

// structure definition for structures used to define what data each rep needs
typedef struct {
  int rep;                   // integer code for this rep style
  char *name;                // name of this rep
  int numdata;               // number of data values required for style
  AtomRepDataInfo repdata[MAX_ATOMREP_DATA];  // info about each data value
} AtomRepParamStruct;


// array with definition of data for each rep
extern AtomRepParamStruct AtomRepInfo[AtomRep::TOTAL];


#endif

