/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: AtomRep.C,v $
 *      $Author: dalke $        $Locker:  $                $State: Exp $
 *      $Revision: 1.21 $      $Date: 96/02/07 05:22:28 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Parse and maintain the data for how a molecule should be represented.
 *
 ***************************************************************************/

#include <string.h>
#include <stdlib.h>
#include "AtomRep.h"
#include "Inform.h"
#include "utilities.h"
#include "config.h"
#ifdef VMDALPHA_SHAPE
#include "AlphaShape.h"
#endif
#ifdef VMDSURF
#include "Surf.h"
#endif

/***********************  NOTES on ADDING NEW REPRESENTATIONS ************
 So, you wanna add a new representation ... here's what you have to do:
 1. Choose a a name for the rep, and decide how many parameters it needs.
    This object has support for up to MAX_ATOMREP_DATA parameters.
 2. Choose the order in which these will be specified in the text command.
 3. Add a new identifier in the main RepMethod enumeration for this new
    rep.  The order is important; where you put the new item is where it
    will appear in lists of the different representations.
 4. Add a new element to the AtomRepInfo structure array (below).  This element
    defines the name of the new rep, and gives info about all the parameters
    needed for that rep.  You must put the item into the array in the same
    position where the new enum value was inserted into RepMethod.
 5. Once this structure is properly defined, the GUI and text interface will
    properly process and display the info for the new rep.  At this point,
    you must add the code to display the new rep in DrawMolItem.
 6. Sit back and enjoy ...

 FORMAT for AtomRepInfo STRUCTURE:
 ---------------------------------
 Easiest thing to do is to copy an item with the same number of parameters,
 and change the necessary values.  The items in each structure are:
	a. Integer enum code for the rep (same as was put in RepMethod enum)
	b. Name of rep (should be a single small word; this is the name
	   used in the GUI list, and what the user calls the rep when typing
	   in a command to change to this rep)
	c. Number of parameters required.
	d. An array of Structures which define how each parameter is used
	   for the representation.  The format for these items are:
		1. Label for parameter, as it will appear on the GUI control
		2. Index of the parameter in the AtomRep internal data
		   storage array.  This should be used when retrieving the
		   data; call 'get_data' with this index value to retrieve
		   the value for this parameter.  Each parameter for the
		   representation should have a different index, and they
		   should be in the range 0 ... MAX_ATOMREP_DATA-1.
		   There is an enumeration giving names for these indices,
		   which is used right now primarily for historical reasons.
		   Older parts of VMD used explicit calls to routines to get
		   values for things such as the line thickness and sphere
		   resolution, and these routines are still present.  But
		   since new reps may not necessarily have characteristics
		   which map to these items, this should not be used in the
		   future.
		3. Default value
		4. Number of signif. digits past the decimal point.  This is
		   used when displaying the number in the GUI; if it is 0,
		   the value will be displayed as an integer.
		5. Flag for whether the item has a minimum value. (1 or 0)
		6. Flag for whether the item has a maximum value. (1 or 0)
		7. Minimum value; if not applicable, just use 0
		8. Maximum value; if not applicable, just use 0
		9. Small increment size, used by GUI.
	       10. Large increment size, used by GUI.
 ***************************************************************************/


// define structures which indicate what data is needed by each rep
AtomRepParamStruct AtomRepInfo[AtomRep::TOTAL] = {

  // Lines
  { AtomRep::LINES, "Lines", 1, {
    { "Thickness", AtomRep::LINETHICKNESS, 2.0, 0, 1, 0, 1, 0, 1, 5 },
    0, 0, 0, 0, 0 }
  },

  // Bonds
  { AtomRep::BONDS, "Bonds", 2, {
    { "Bond Rad", AtomRep::BONDRAD, 0.3, 2, 1, 0, 0, 0, 0.1, 1.0 },
    { "Bond Res", AtomRep::BONDRES, 8.0, 0, 1, 0, 1, 0, 1, 5 },
    0, 0, 0, 0 }
  },

  // CPK
  { AtomRep::CPK, "CPK", 4, {
    { "S Rad", AtomRep::SPHERERAD, 1.0, 2, 1, 0, 0, 0, 0.1, 1.0 },
    { "B Rad", AtomRep::BONDRAD, 0.3, 2, 1, 0, 0, 0, 0.1, 1.0 },
    { "Sphere Res", AtomRep::SPHERERES, 6.0, 0, 1, 0, 1, 0, 1, 5 },
    { "Bond Res", AtomRep::BONDRES, 8.0, 0, 1, 0, 1, 0, 1, 5 },
    0, 0 }
  },

  // Points
  { AtomRep::POINTS, "Points", 0, { 0, 0, 0, 0, 0, 0 } },

  // VDW
  { AtomRep::VDW, "VDW", 2, {
    { "Sphere Rad", AtomRep::SPHERERAD, 1.0, 2, 1, 0, 0, 0, 0.1, 1.0 },
    { "Sphere Res", AtomRep::SPHERERES, 6.0, 0, 1, 0, 1, 0, 1, 5 },
    0, 0, 0, 0 }
  },

  // Dotted
  { AtomRep::DOTTED, "Dotted", 2, {
    { "Sphere Rad", AtomRep::SPHERERAD, 1.0, 2, 1, 0, 0, 0, 0.1, 1.0 },
    { "Sphere Res", AtomRep::SPHERERES, 6.0, 0, 1, 0, 1, 0, 1, 5 },
    0, 0, 0, 0 }
  },

  // Licorice
  { AtomRep::LICORICE, "Licorice", 3, {
    { "B Rad", AtomRep::BONDRAD, 0.3, 2, 1, 0, 0, 0, 0.1, 1.0 },
    { "Sphere Res", AtomRep::SPHERERES, 6.0, 0, 1, 0, 1, 0, 1, 5 },
    { "Bond Res", AtomRep::BONDRES, 8.0, 0, 1, 0, 1, 0, 1, 5 },
    0, 0, 0 }
  },

  // Ribbons
  { AtomRep::RIBBONS, "Ribbons", 3, {
    { "Rad", AtomRep::BONDRAD, 0.3, 2, 1, 0, 0, 0, 0.1, 1.0 },
    { "Res", AtomRep::BONDRES, 8.0, 0, 1, 0, 1, 0, 1, 5 },
    { "Thickness", AtomRep::LINETHICKNESS, 2.0, 0, 1, 0, 0, 0, 1, 5 },
    0, 0, 0 }
  },

  // Tube
  { AtomRep::TUBE, "Tube", 2, {
    { "Rad", AtomRep::BONDRAD, 0.3, 2, 1, 0, 0, 0, 0.1, 1.0 },
    { "Res", AtomRep::BONDRES, 8.0, 0, 1, 0, 0, 0, 1, 5 },
    0, 0, 0, 0 }
  },

  // Structure
  { AtomRep::STRUCTURE, "Structure", 3, {
    { "Radius", AtomRep::BONDRAD, 2.1, 2, 1, 0, 0, 0, 0.1, 1.0 },
    { "Res", AtomRep::BONDRES, 8.0, 0, 1, 0, 1, 0, 1, 5 },
    { "Thickness", AtomRep::LINETHICKNESS, 5.0, 0, 1, 0, 0, 0, 1, 5 },
    0, 0, 0 }
  },

  // HBonds
  { AtomRep::HBONDS, "HBonds", 3, {
    { "Dist", AtomRep::BONDRAD, 3.0, 2, 1, 0, 0, 0, 0.1, 1.0 },
    { "Angle", AtomRep::SPHERERAD, 20.0, 2, 1, 1, 0, 180.0, 1.0, 10.0 },
    { "Thickness", AtomRep::LINETHICKNESS, 1.0, 0, 1, 0, 1, 0, 1, 5 },
    0, 0, 0 }
  },

#ifdef VMDALPHA_SHAPE
  // Alpha
  { AtomRep::ALPHASHAPE, "Alpha", 1, {
    { "Alpha Rad", AtomRep::SPHERERAD, 1.0, 2, 1, 0, 0, 0, 0.1, 1.0 },
    0, 0, 0, 0, 0 }
  },
#endif

#ifdef VMDSURF
  // Surf
  { AtomRep::SURF, "Surf", 1, {
    { "Probe Rad", AtomRep::SPHERERAD, 1.4, 2, 1, 0, 0, 0, 0.1, 1.0 },
    0, 0, 0, 0, 0 }
  },
#endif

  // Off
  { AtomRep::REPOFF, "Off", 0, { 0, 0, 0, 0, 0, 0 } }
};


//////////////////////////  constructor and destructor
// constructor; parse string and see if OK
AtomRep::AtomRep(void) {
  
  MSGDEBUG(2,"Creating new AtomRep object ..." << sendmsg);

  // initialize variables
  repMethod  = DEFAULT_ATOMREP;
  repData = repDataStorage[repMethod];
  option_changed = FALSE;
  strcpy(cmdStr, AtomRepInfo[repMethod].name);
  for(int j=0; j < TOTAL; j++)
    for(int i=0; i < AtomRepInfo[j].numdata ; i++)
      repDataStorage[j][AtomRepInfo[j].repdata[i].index] =
	AtomRepInfo[j].repdata[i].defValue;

#ifdef VMDALPHA_SHAPE
  alpha = NULL;
  alpha_count = NULL;
#endif
#ifdef VMDSURF
  surf = NULL;
  surf_count = NULL;
#endif
  
}


#ifdef VMDALPHA_SHAPE
void AtomRep::delete_alpha(void)
{
   if (alpha) {                 // alpha can be used by several selections
      if (*alpha_count <= 1) {  // so I check the reference count to see if
	 delete alpha;          // anyone else is using this.
	 delete alpha_count;
      } else {
	 (*alpha_count)--;      // there is another selection with this shape
	                        // so just reduce the count
      }
      alpha = NULL;
      alpha_count = NULL;
   }
}
void AtomRep::set_alpha(AlphaShape *newalpha)
{
   if (alpha) {
      delete_alpha();
   }
   alpha = newalpha;
   alpha_count = new int;
   *alpha_count = 1;
}
#endif

#ifdef VMDSURF
void AtomRep::delete_surf(void)
{
   if (surf) {
      if ((*surf_count) <= 1) {
	 delete surf;
	 delete surf_count;
      } else {
	 (*surf_count)--;
      }
      surf = NULL;
      surf_count = NULL;
   }
}
void AtomRep::set_surf(Surf *newsurf)
{
   if (surf) {
      delete_surf();
   }
   surf = newsurf;
   surf_count = new int;
   *surf_count = 1;
}
#endif

AtomRep::~AtomRep(void) {
#ifdef VMDALPHA_SHAPE
   delete_alpha();
#endif
#ifdef VMDSURF
   delete_surf();
#endif
}

//////////////////////////  private routines

// parse the given command, and store results.  Return success.
int AtomRep::parse_cmd(char *newcmd) {
  int argc, i, j;
  char *argv[128], *cmdStrTok = NULL;

  // make sure the new command is not too long
  if(newcmd && strlen(newcmd) > MAX_ATOMREP_CMD) {
    msgErr << "Atom representation string is too long (over ";
    msgErr << MAX_ATOMREP_CMD << " characters)." << sendmsg;
    return FALSE;
  }

  // tokenize the command
  if(!newcmd || !(cmdStrTok = str_tokenize(newcmd, &argc, argv))) {
    // no command; keep current settings
    return TRUE;
  }

  // see if the command matches a representation we know about
  for(i=0; i < AtomRep::TOTAL; i++) {
    AtomRepParamStruct *ari = &(AtomRepInfo[i]);

    if(!strupncmp(argv[0], ari->name, CMDLEN)) {

      // the name matches; make sure we do not have too many arguments
      if((argc - 1) > ari->numdata) {
	msgErr << "Incorrect atom representation command '" << newcmd << "':";
	msgErr << "\n  '" << ari->name << "' only takes " << ari->numdata;
	msgErr << " arguments maximum." << sendmsg;
	delete [] cmdStrTok;
	return FALSE;
      }

      // indicate that we've changed to a new rep method
      repMethod = i;
      repData = repDataStorage[i];

      // copy out the data from the command and store it
      for(j=1; j < argc; j++) {
	// if a '-' is given, do not change that parameter
	if(!(argv[j][0] == '-' && argv[j][1] == '\0'))
	  set_data(ari->repdata[j-1].index, atof(argv[j]));
      }

      // found a match; break from for loop
      break;
    }
  }

  // if i is at the end of the list, we did not find a match
  if(i == AtomRep::TOTAL) {
    msgErr << "Unknown atom representation command '" << newcmd << "'.";
    msgErr << sendmsg;
    delete [] cmdStrTok;
    return FALSE;
  }

  // if we are here, everything went OK
  strcpy(cmdStr, newcmd);
  delete [] cmdStrTok;

  return TRUE;
}


//////////////////////////  public routines

// copy contents of another atomrep
AtomRep &AtomRep::copy(AtomRep &ar) {
  strcpy(cmdStr, ar.cmdStr);
  repMethod = ar.repMethod;
  repData = repDataStorage[repMethod];
  option_changed = FALSE;
  for(int j=0; j < TOTAL; j++)
    for(int i=0; i < MAX_ATOMREP_DATA; i++)
      repDataStorage[j][i] = ar.repDataStorage[j][i];

#ifdef VMDALPHA_SHAPE
  alpha = ar.alpha;
  alpha_count = ar.alpha_count;
  if (alpha_count) {   // increase the reference count
     (*alpha_count)++;
  }
#endif

#ifdef VMDSURF
  surf = ar.surf;
  surf_count = ar.surf_count;
  if (surf_count) {
     (*surf_count)++;
  }
#endif

  return *this;
}


