/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: ArtDisplayDevice.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.2 $	$Date: 96/03/24 07:07:20 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Writes to the ART raytracer.  This is available from gondwana.ecr.mu.oz.au
 * as part of the vort package.  To see the output I suggest:
 *   art plot.scn 1000 1000
 *   vort2ppm plot.pix > plot.ppm
 *   fromppm plot.ppm plot.rgb
 *   ipaste plot.rgb
 *
 ***************************************************************************/
#include <stdio.h>
#include <string.h>
#include <math.h>
#include "ArtDisplayDevice.h"
#include "Stack.h"
#include "Matrix4.h"
#include "DispCmds.h"
#include "Inform.h"
#include "utilities.h"

#define DEFAULT_RADIUS 0.002
#define DASH_LENGTH 0.02

// Be careful when you modify the coordinates.  To make things view the
// right way, I have to rotate everything around the (x,y,z) = (1,1,1)
// vector so that x->z, y->x, and z->y

#define ORDER(x,y,z) -z, -x, y
//#define ORDER(x,y,z) x,y,z

///////////////////////// constructor and destructor

// constructor ... initialize some variables
ArtDisplayDevice::ArtDisplayDevice(char *nm) : FileRenderer(nm) {
	
    MSGDEBUG(1,"Creating ArtDisplayDevice ..." << sendmsg);
    
    strcpy(renderCommand,"art %s 500 650");
    strcpy(fname,"plot.scn");
}
               
//destructor
ArtDisplayDevice::~ArtDisplayDevice(void) { }

///////////////////////// protected nonvirtual routines

// draw a point
void ArtDisplayDevice::point(float spdata[]) {
  float vec[3];

  // transform the world coordinates
  (transMat.top()).multpoint3d(spdata, vec);
   
  // draw the sphere
  fprintf(art_file, "sphere {\ncolour %f,%f,%f\n",
	  currentColor[0], currentColor[1], currentColor[2]);
  fprintf(art_file, "radius %f\n", float(lineWidth) * DEFAULT_RADIUS);
  fprintf(art_file, "center (%f,%f,%f)\n}\n", ORDER(vec[0], vec[1], vec[2]));
}

// draw a sphere
void ArtDisplayDevice::sphere(float spdata[]) {
  
  float vec[3];
  float radius;
    
  // transform the world coordinates
  (transMat.top()).multpoint3d(spdata, vec);
  radius = scale_radius(spdata[3]);
   
  // draw the sphere
  fprintf(art_file, "sphere {\ncolour %f,%f,%f\n",
	  currentColor[0], currentColor[1], currentColor[2]);
  fprintf(art_file, "radius %f\n", radius);
  fprintf(art_file, "center (%f,%f,%f)\n}\n", ORDER(vec[0], vec[1], vec[2]));
}

// draw a line (cylinder) from a to b
void ArtDisplayDevice::line(float *a, float*b) {
    int i, j, test;
    float dirvec[3], unitdirvec[3];
    float from[3], to[3], tmp1[3], tmp2[3];
    float len;
    
    if(lineStyle == ::SOLIDLINE ) {
  
        // transform the world coordinates
        (transMat.top()).multpoint3d(a, from);
        (transMat.top()).multpoint3d(b, to);
    
        // draw the cylinder
	fprintf(art_file, "cylinder {\n");
	fprintf(art_file, "colour %f,%f,%f\n", currentColor[0], 
		currentColor[1], currentColor[2]);
        fprintf(art_file, "center(%f,%f,%f)\n", 
		ORDER(from[0], from[1], from[2])); // first point
        fprintf(art_file, "center(%f,%f,%f)\n", 
		ORDER(to[0], to[1], to[2])); // second point
        fprintf(art_file, "radius %f\n}\n", 
		float(lineWidth)*DEFAULT_RADIUS); // radius
        
    } else if (lineStyle == ::DASHEDLINE ) {
        
         // transform the world coordinates
        (transMat.top()).multpoint3d(a, tmp1);
        (transMat.top()).multpoint3d(b, tmp2);

        // how to create a dashed line
        for(i=0;i<3;i++) {
            dirvec[i] = tmp2[i] - tmp1[i];  // vector from a to b
        }
        len = sqrtf( dirvec[0]*dirvec[0] + dirvec[1]*dirvec[1] + dirvec[2]*dirvec[2] );
        for(i=0;i<3;i++) {
            unitdirvec[i] = dirvec[i] / sqrtf(len);  // unit vector pointing from a to b
        }
           
        test = 1;
        i = 0;

        while( test == 1 ) {
            for(j=0;j<3;j++) {
                from[j] = tmp1[j] + (2*i)*DASH_LENGTH*unitdirvec[j];
                to[j] = tmp1[j] + (2*i + 1)*DASH_LENGTH*unitdirvec[j];
            }
            if( fabsf(tmp1[0] - to[0]) >= fabsf(dirvec[0]) ) {
                for(j=0;j<3;j++) {
                    to[j] = tmp2[j];
                }
                test = 0;
            }
    
            // draw the cylinder
	    fprintf(art_file, "cylinder {\n");
	    fprintf(art_file, "colour %f,%f,%f\n", currentColor[0], 
		    currentColor[1], currentColor[2]);
	    // first point
	    fprintf(art_file, "center(%f,%f,%f)\n", 
		    ORDER(from[0], from[1], from[2]));
	    // second point
	    fprintf(art_file, "center(%f,%f,%f)\n", 
		    ORDER(to[0], to[1], to[2])); 
	    // radius
	    fprintf(art_file, "radius %f\n}\n", 
		    float(lineWidth)*DEFAULT_RADIUS); 

            i++;
        }
    } else {
        msgErr << "ArtDisplayDevice: Unknown line style " << lineStyle << sendmsg;
    }

}

// draw a cylinder
void ArtDisplayDevice::cylinder(float *a, float *b, float r) {

  float vec1[3], vec2[3];
  
  // transform the world coordinates
  (transMat.top()).multpoint3d(a, vec1);
  (transMat.top()).multpoint3d(b, vec2);
    
  // draw the cylinder
  fprintf(art_file, "cylinder {\n");
  fprintf(art_file, "colour %f,%f,%f\n", currentColor[0], 
	  currentColor[1], currentColor[2]);
  // first point
  fprintf(art_file, "center(%f,%f,%f)\n", 
	  ORDER(vec1[0], vec1[1], vec1[2]));
  // second point
  fprintf(art_file, "center(%f,%f,%f)\n", 
	  ORDER(vec2[0], vec2[1], vec2[2])); 
  // radius
  fprintf(art_file, "radius %f\n}\n", 
	  scale_radius(r));
}

// draw a cone
void ArtDisplayDevice::cone(float *a, float *b, float r) {

  float vec1[3], vec2[3];
  
  // transform the world coordinates
  (transMat.top()).multpoint3d(a, vec1);
  (transMat.top()).multpoint3d(b, vec2);
    
  fprintf(art_file, "cone {\n");
  fprintf(art_file, "colour %f,%f,%f\n", currentColor[0], 
	  currentColor[1], currentColor[2]);
  // second point
  fprintf(art_file, "vertex(%f,%f,%f)\n", 
	  ORDER(vec2[0], vec2[1], vec2[2])); 
  // first point
  fprintf(art_file, "center(%f,%f,%f)\n", 
	  ORDER(vec1[0], vec1[1], vec1[2]));
  // radius
  fprintf(art_file, "radius %f\n}\n", scale_radius(r), 0.0);
}

// draw a triangle
void ArtDisplayDevice::triangle(float *a, float *b, float *c, float *n1, 
float *n2, float *n3) {

  float vec1[3], vec2[3], vec3[3];
  
  // transform the world coordinates
  (transMat.top()).multpoint3d(a, vec1);
  (transMat.top()).multpoint3d(b, vec2);
  (transMat.top()).multpoint3d(c, vec3);

  // draw the triangle
  fprintf(art_file, "polygon {\n");
  fprintf(art_file, "colour %f,%f,%f\n", currentColor[0], 
	  currentColor[1], currentColor[2]);
  
  fprintf(art_file, "vertex (%f,%f,%f),(%f,%f,%f)\n", 
	  ORDER(vec1[0], vec1[1], vec1[2]), // point one
	  ORDER(n1[0], n1[1], n1[2]));
  
  fprintf(art_file, "vertex (%f,%f,%f),(%f,%f,%f)\n", 
	  ORDER(vec2[0], vec2[1], vec2[2]), // point two
	  ORDER(n2[0], n2[1], n2[2]));
  fprintf(art_file, "vertex (%f,%f,%f),(%f,%f,%f)\n", 
	  ORDER(vec3[0], vec3[1], vec3[2]), // point three
	  ORDER(n3[0], n3[1], n3[2]));
  fprintf(art_file, "}\n");
}

// draw a square
void ArtDisplayDevice::square(float *, float *a, float *b, float *c, float *d) {
  
  float vec1[3], vec2[3], vec3[3], vec4[3];
  
  // transform the world coordinates
  (transMat.top()).multpoint3d(a, vec1);
  (transMat.top()).multpoint3d(b, vec2);
  (transMat.top()).multpoint3d(c, vec3);
  (transMat.top()).multpoint3d(d, vec4);

  // draw the square
  fprintf(art_file, "polygon {\n");
  fprintf(art_file, "colour %f,%f,%f\n", currentColor[0], 
	  currentColor[1], currentColor[2]);

  fprintf(art_file, "vertex (%f,%f,%f)\n", 
	  ORDER(vec1[0], vec1[1], vec1[2])); // point one
  fprintf(art_file, "vertex (%f,%f,%f)\n", 
	  ORDER(vec2[0], vec2[1], vec2[2])); // point two
  fprintf(art_file, "vertex (%f,%f,%f)\n", 
	  ORDER(vec3[0], vec3[1], vec3[2])); // point three
  fprintf(art_file, "vertex (%f,%f,%f)\n", 
	  ORDER(vec4[0], vec4[1], vec4[2])); // point four
  fprintf(art_file, "}\n");
  
}

///////////////////// public virtual routines

// initialize the file for output
int ArtDisplayDevice::prepare3D(int ) {

  if(!(art_file = fopen(fname,"w"))) {
    msgErr << "Cannot open file " << fname << " for Art output" << sendmsg;
    Initialized = FALSE;
  } else {
  
    Initialized = TRUE;
    art_filename = stringdup(fname);

    fprintf(art_file, "up(0, 0, 1) \n");
    fprintf(art_file, "lookat(-1, 0, 0, 0, 0, 0, 0)\n");
    fprintf(art_file, "fieldofview 45 \n");
    

    // write the light sources
    // The light code doesn't work because at this point I don't have the
    //  correct transformation matrix, so I'll leave only the one light source
    fprintf(art_file, "light {\n\tcolour 1, 1, 1\n"
	    "\tlocation (-10, 0, 0)\n}\n");

    // set the background
    fprintf(art_file, "background %f, %f, %f\n", backColor[0],
	    backColor[1], backColor[2]);

    // everything is plastic-like
    fprintf(art_file, "material 0.0, 0.75, 0.25, 20.0\n");
  }
  return Initialized;
}

    
// clean up after yourself
void ArtDisplayDevice::update(int) {

  if(Initialized) {
    fclose(art_file);
    if(art_filename) delete [] art_filename;
    msgInfo << "Art file generation finished" << sendmsg;
  }
}

