/**********************************************************************
 *         Copyright (C) 1995  The Board of Trustees of               *
 *                             the University of Illinois             *
 *                                                                    *
 *  This file is part of the RAPP software package, a library and     *
 *  associated programs for coordinating client/server applications.  *
 *                                                                    *
 **********************************************************************/

/* rapp_types.h: Internal data types for RAPP software.  Included
 *               by main header file, <rapp.h>.
 *
 * Rick Kufrin <r-kufrin@uiuc.edu>
 * National Center for Supercomputing Applications
 * May, 1995
 * 
 */

#ifndef _RAPP_TYPES_H
#define _RAPP_TYPES_H

#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netdb.h>
#include <rpc/rpc.h>

enum rapp_domain_type {UNIX, INET};
typedef enum rapp_domain_type rapp_domain_type_t;
enum rapp_proto_type {STREAM, DGRAM};
typedef enum rapp_proto_type rapp_proto_type_t;

struct rapp_passive_socket {
  union {
    struct sockaddr_un un;
    struct sockaddr_in in;
  } domain;
  rapp_domain_type_t type;
  int sock;
};
typedef struct rapp_passive_socket rapp_passive_socket_t;

struct rapp_active_socket {
  union {
    struct sockaddr_un un;
    struct sockaddr_in in;
  } domain;
  rapp_domain_type_t type;
  rapp_proto_type_t proto;
  int sock;
  int do_xdr;
  XDR **xdrs;
  char *data_buf;
  int bufsiz;
};
typedef struct rapp_active_socket rapp_active_socket_t;

#define RAPP_NDATATYPES 7

enum rapp_datatype {RAPP_BYTE, RAPP_INT, RAPP_UINT,
		      RAPP_LONG, RAPP_ULONG, RAPP_FLOAT, RAPP_DOUBLE};
typedef enum rapp_datatype rapp_datatype_t;

enum rapp_perm_type {RAPP_RDWR, RAPP_RDONLY};
typedef enum rapp_perm_type rapp_perm_t;

enum rapp_opt_type {RAPP_OPT_DATACONV, RAPP_OPT_EXECMODE};
typedef enum rapp_opt_type rapp_opt_type_t;

#if defined (sgi) || defined (SYSV)
typedef void * shmaddr_t;
#else
typedef char * shmaddr_t;
#endif

struct rapp_shmem_segment {
  int shmid;
  shmaddr_t shmaddr;
};
typedef struct rapp_shmem_segment rapp_shmem_t;

#if defined(__hpux)
union semun {
  int             val;    /* value for SETVAL */
  struct semid_ds *buf;   /* buffer for IPC_STAT & IPC_SET */
  ushort          *array; /* array for GETALL & SETALL */
};
#endif

enum rapp_option_type {RAPP_FLOAT_OPTION, RAPP_INT_OPTION, 
			 RAPP_STRING_OPTION};
typedef enum rapp_option_type rapp_option_type_t;

struct rapp_option {
  char               *keyword;
  rapp_option_type_t type;     /* The data type of the option's value */
  int                required; /* Non-zero == required                */
  char               *def;     /* Default value                       */
  char               *desc;    /* Textual description of the option   */
};
typedef struct rapp_option rapp_option_t;

struct rapp_optlist {
  int           count;        /* How many options are there?  */
  rapp_option_t **opt;        /* The option list              */
  char          *daemon;      /* The default daemon program   */
  char          *exe;         /* The application program      */
  char          *cwd;         /* The remote working directory */
};
typedef struct rapp_optlist rapp_optlist_t;

typedef struct rapp_proglist *rapp_proglist_t;
typedef struct rapp_joblist *rapp_joblist_t;

struct rapp_param {
  char               *keyword;
  char               *val;
};
typedef struct rapp_param rapp_param_t;

struct rapp_paramlist {
  int          count;     /* How many parameters are there?  */
  rapp_param_t **param;   /* The parameter list              */
  char         *exe;      /* The application program         */
  char         *cwd;      /* The remote working directory    */
};
typedef struct rapp_paramlist rapp_paramlist_t;

struct rapp_handler_list {
  int tag;
  int (*handler)();
  struct rapp_handler_list *next;
};
typedef struct rapp_handler_list rapp_handler_list_t;


/*
 * The following structures define the interfaces between the various
 * pieces of a RAPP-connected application.
 *
 * There are 4 communication channels present in a RAPP "connected"
 * application:
 *
 * 1. Application <-> Application daemon
 * 2. Client <-> Application daemon
 * 3. Client <-> Consumer
 * 4. Consumer <-> Application
 *
 * Note that, in the case of an "unconnected" application, only channel
 * number (1) above is in existence.
 */

/* rapp_app_handle contains the data structures needed for a RAPP
 * application to communicate with the other RAPP component programs.
 */

struct rapp_app_handle {
  int connected;
  int disconnecting;
  int xfer_interval;
  int xfer_count;
  rapp_active_socket_t *appd_sock;
  rapp_passive_socket_t *consumer_psock;
  rapp_active_socket_t *consumer_asock;
  void *static_data;
  void *dynamic_data;
  int (*sendStatic)(rapp_active_socket_t *, void *);
  int (*sendDynamic)(rapp_active_socket_t *, void *, void *);
  void (*killHandler)(void);
  rapp_handler_list_t *handlers;
};
typedef struct rapp_app_handle *rapp_app_handle_t;

/* rapp_appd_handle contains the data structures needed for a RAPP
 * application daemon to communicate with the other RAPP component
 * programs.
 */

struct rapp_appd_handle {
  rapp_passive_socket_t *psock_client;
  rapp_active_socket_t *asock_client;
  int app_port;
  rapp_active_socket_t *asock_udp;
  rapp_passive_socket_t *psock_app;
  rapp_active_socket_t *asock_app;
  rapp_paramlist_t *paramlist;
  int app_pid;
  int rappd_jid;
  int xfer_interval;
  int (*execFunc)(struct rapp_appd_handle *, rapp_paramlist_t *);
  int (*recvStatic)(rapp_active_socket_t *, void *);
  int (*sendStatic)(rapp_active_socket_t *, void *);
  rapp_handler_list_t *handlers;
  void *static_data;
  char *authpass;
  int connected;
  int app_active;
};
typedef struct rapp_appd_handle *rapp_appd_handle_t;

/* rapp_consumer_handle contains the data structures needed for a RAPP
 * consumer to communicate with the other RAPP component programs.
 */

struct rapp_consumer_handle {
  rapp_passive_socket_t *client_psock;
  rapp_active_socket_t *client_asock;
  rapp_active_socket_t *app_asock;
  int semid;
  rapp_shmem_t *shmem;
  void *static_data;
  void *dynamic_data;
  int (*recvDynamic)(rapp_active_socket_t *, void *, void *);
};
typedef struct rapp_consumer_handle *rapp_consumer_handle_t;

/* rapp_client_handle contains the data structures needed for a RAPP
 * client to communicate with the other RAPP component programs.
 */

struct rapp_client_handle {
  rapp_active_socket_t *appd_sock;
  rapp_active_socket_t *consumer_asock;
  int consumer_pid;
  int semid;
  rapp_shmem_t *shmem;
  void *static_data;
  void *dynamic_data;
  int (*recvStatic)(rapp_active_socket_t *, void *);
  int connected;
  int app_active;
};
typedef struct rapp_client_handle *rapp_client_handle_t;

/*
 * User-defined command handler utility routines.  Common to
 * both the application and application daemon, so convenient
 * to place here.
 */

int
rapp_install_handler(rapp_handler_list_t **handler_list, int tag, 
		     int (*handler)());

int
(*rapp_lookup_handler(rapp_handler_list_t *, int))();


#endif /* _RAPP_TYPES_H */
