/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Scene.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.10 $	$Date: 1995/05/11 23:57:04 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Scene object, which maintains a list of Displayable objects and
 * draws them to a DisplayDevice.
 *
 * Each Scene has a list of Displayable objects, and also a list of display
 * commands.  The command lists are used to draw the objects, the Displayable
 * objects to prepare and update objects for drawing.
 *
 * Each Scene has a list of possible picking modes; these are created by
 * the request of Pickable objects, which then get notified when an
 * object is picked.  Only Pickables which are interested in the specified
 * mode are told of the picking result.  This capability is inherited from
 * PickList.
 *
 ***************************************************************************/
#ifndef SCENE_H
#define SCENE_H

#include "PickList.h"
#include "Displayable.h"
class DisplayDevice;


class Scene : public PickList {

protected:
  // number of 2D and 3D Displayable objects this Scene is managing
  int numDisplayable2D, numDisplayable3D;

  // number of command lists we are managing
  int numCmdLists2D, numCmdLists3D;

  // are we rocking the objects back and forth?  If so, what, where, and how?
  int Rocking, rockSteps, currRockStep, rockOnce;
  char rockAxis;
  float rockAmount;

public:
  // constructor/destructor
  Scene(void);
  virtual ~Scene(void);

  //
  // query for info about Scene
  //
  
  // return number of Displayable objects this has, and number of command lists
  int num_displayable2D(void) { return numDisplayable2D; }
  int num_displayable3D(void) { return numDisplayable3D; }
  int num_disp_list2D(void) { return numCmdLists2D; }
  int num_disp_list3D(void) { return numCmdLists3D; }
  int rocking(void) { return Rocking; }
  char rock_axis(void) { return rockAxis; }
  float rock_amount(void) { return rockAmount; }
  
  //
  // virtual routines to access the specific displayable's and lists
  //
  
  // return the Nth Displayable, for the given dim (2 or 3)
  virtual Displayable *displayable2D(int) = 0;
  virtual Displayable *displayable3D(int) = 0;
 
  // return the Nth display command list, for the given dim (2 or 3)
  virtual char *disp_list2D(int n) = 0;
  virtual char *disp_list3D(int n) = 0;

  //
  // deal with command list storage, and registration of Displayables
  //
  
  // allocate/free display list storage space of specified size
  virtual char *get_disp_storage(int) = 0;
  virtual void free_disp_storage(char *) = 0;
  
  // (un)register the given Displayable & corresponding display list(of dim D)
  virtual void Register(Displayable *, char *, int) = 0;
  virtual void unRegister(Displayable *, char *, int) = 0;
  
  //
  // general operations for this Scene
  //
  
  // clear the entire Scene, i.e. remove all displayable objects
  void clear(void);

  // do action due to the fact that a color for the given ColorList for
  // the specified category has changed
  void color_changed(ColorList *, int);

  // prepare all registered Displayables
  virtual void prepare(DisplayDevice *display);

  // a displayable calls this when it is done preparing.  Arguments are
  // the pointer to the command list the displayable used, and the dimension
  // of the displayable.
  virtual void done_preparing(char *, int);

  // draw the scene to the given DisplayDevice, can change display states
  virtual void draw(DisplayDevice *);
  
  // draw the scene to a file in a given format, trying to match the
  // view of the given DisplayDevice as closely as possible
  // returns TRUE if successful, FALSE if not
  // There are no stereo output formats; if there are, then things will
  // become somewhat more difficult
  virtual int filedraw(char *, char *, DisplayDevice *);

  // 
  // routines to affect all Displayables in the Scene
  //
  
  // just change the transformation to the one given
  void load_transformation(Matrix4 &);
  
  // reset to identity the tranformation
  void reset_transformation(void);

  // scaling
  void set_scale(float s);
  void mult_scale(float s);
  void set_scale(Matrix4 &);
  void mult_scale(Matrix4 &);
  
  // rotating
  void add_rot(float x, char axis);
  void add_rot(Matrix4 &);
  void set_rot(Matrix4 &);

  // global translation
  void set_glob_trans(float, float, float);
  void set_glob_trans(Matrix4 &);
  void add_glob_trans(float, float, float);
  void add_glob_trans(Matrix4 &);

  // centering translation
  void set_cent_trans(float, float, float);
  void set_cent_trans(Matrix4 &);
  void add_cent_trans(float, float, float);
  void add_cent_trans(Matrix4 &);

  //
  // deal with rocking the objects in the Scene
  //

  void stop_rocking(void) { Rocking = FALSE; }
  void resume_rocking(void) { Rocking = TRUE; }
  void start_rocking(float a = 1.0, char ax = 'y', int steps = 180,
  			int doOnce = FALSE);

};

#endif

/* REVISION HISTORY:********************************************************
 *
 * $Log: Scene.h,v $
 * Revision 1.10  1995/05/11  23:57:04  billh
 * Moved log messages to end of file.
 *
 * Revision 1.9  95/03/31  16:01:44  billh
 * Moved start_rocking routine to .C file; added ability to rock just
 * once, so that a smooth rotation from a --> can be done.
 * 
 * Revision 1.8  95/03/24  18:51:45  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.7  1995/02/22  03:57:58  billh
 * Now supports picking objects with a 2D or 3D pointer.
 * Scene now derived from PickList.
 *
 * Revision 1.6  94/12/06  08:26:45  billh
 * Added routine color_changed, which simply calls this same routine
 * for all registered objects.  color_changed is called when a color in a
 * specific category is changed.
 * 
 * Revision 1.5  94/11/24  07:27:03  dalke
 * Added filedraw for writing the scene to some external graphics format
 * 
 * Revision 1.4  1994/10/01  03:06:53  billh
 * New format; Scene now a more general base class, from which are derived
 * NormalScene and CaveScene (for starters ...)
 *
 * Revision 1.3  94/09/29  19:27:11  dalke
 * Added mem. alloc/free functions so I can use shared mem.
 * Split draw routine up into prepare/draw/update
 * 
 * Revision 1.2  1994/09/14  03:55:10  billh
 * Changed due to restructuring of Displayable ... now calls
 * 'draw_prepare' and 'draw_update' for Displayable's.
 *
 * Revision 1.1  94/08/24  03:10:37  billh
 * Initial revision
 * 
 ***************************************************************************/
