/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: PopupMenu.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.1 $	$Date: 1995/05/11 23:20:03 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The pop-up (possibly pull-down) menu used by the Mouse and perhaps other
 * components.  The menu consists of elements with these characteristics:
 *	1) Name (as it appears in the menu)
 *	2) Text command to execute when the item is selected (may be null).
 *	3) Return code: this is an integer code which is returned by the
 *	   windowing system when the item is picked.   It is assigned by the
 *	   pop-up menu object, and for all regular menu items, it is > 0.
 *	   There are several "special" return codes, which indicate the menu
 *	   item is special.  These special return codes include:
 *	   	MENU_CODE_SUBMENU: the menu item contains a submenu
 *		MENU_CODE_SEP: the menu item is really a separator
 *		MENU_CODE_NONE: the menu item cannot be selected, it just
 *			exists to display text
 *	4) submenu object, if any
 *	5) flags to indicate if the menu item should be displayed, or if
 *	   it should be disabled (greyed out) if possible.
 *
 * A pop-up menu relys on a specific windowing system to provide the methods
 * for creating, posting, and activating a menu.
 *
 * A pop-up menu contains a list of pop-up items, each item containing the
 * characteristics listed above.  A menu is a list of these, and has a name
 * associated with the total list as well as with each item.
 *
 * These classes are used in the following way:
 *	1) A PopupMenu instance is created, that is at the start empty
 *	2) The user calls PopupMenu->add_item to add commands, until the
 *		menu is defined.  This will return integer "return codes"
 *		for the items added, which must be remembered by the caller.
 *	3) If there is a heirarchy of menus, create submenus as described,
 *		and use add_submenu to the parent menus.
 *	4) When the menu is defined, and it is to be used, the menu creator
 *		calls PopupMenu->activate(), which returns either a return
 *		code > 0, or MENU_CODE_NONE if nothing needs to be done by
 *		the menu creator.
 ***************************************************************************/

#include "PopupMenu.h"
#include "DisplayDevice.h"
#include "Inform.h"


/////////////////////////  pop-up menu item class definition

// constructor 1: specify return code, name, and text command
// (the text command may be NULL, in which case no action is done if this
//  item is selected)
PopupMenuItem::PopupMenuItem(int rc, char *nm, char *txt) {
  returnCode = (rc > 0 ? rc : MENU_CODE_NONE);
  itemName = stringdup(nm);
  textCmd = stringdup(txt);
  subMenu = NULL;
  showItem = TRUE;
  showCheck = checkItem = FALSE;
}


// constructor 2: submenu (which determines the name and return code)
PopupMenuItem::PopupMenuItem(PopupMenu *pm) {
  returnCode = MENU_CODE_SUBMENU;
  itemName = stringdup(pm->name());
  textCmd = NULL;
  subMenu = pm;
  showItem = TRUE;
  showCheck = checkItem = FALSE;
}


// constructor 3: nothing ... a separator
PopupMenuItem::PopupMenuItem(void) {
  returnCode = MENU_CODE_SEP;
  itemName = textCmd = NULL;
  subMenu = NULL;
  showItem = TRUE;
  showCheck = checkItem = FALSE;
}


// destructor: frees up allocated memory
PopupMenuItem::~PopupMenuItem(void) {

  if(itemName)
    delete [] itemName;

  if(textCmd)
    delete [] textCmd;

  if(subMenu)
    delete subMenu;
}

// print out the contents of the menu to the given Inform object
// print out each item as "label ... command"  or "label => submenu ''"
int PopupMenuItem::print(Inform *m, int leading) {

  for(int i=0; i < leading; i++)
    *m << ' ';

  if(is_submenu()) {
    *m << itemName << " ==> submenu " << subMenu->name() << sendmsg;
    subMenu->print(m, leading + 2);
  } else {
    if(is_separator())
      *m << "---- separator ----";
    else
      *m << itemName << " = " << textCmd;
    *m << sendmsg;
  }

  return TRUE;
}


/////////////////////////  pop-up menu class definition

// static data for PopupMenu - the next available return code
int PopupMenu::nextReturnCode = 1;

// constructor: name of menu, and initial guess on how many items it contain
PopupMenu::PopupMenu(char *nm, int initSize) : menuItems(initSize) {

  // save name of menu
  menuName = stringdup(nm);

  // init variables which windowing system may use if necessary
  menuID = 0;
  menuExists = FALSE;
}


// destructor: deletes all popup items, which may include submenus as well
PopupMenu::~PopupMenu(void) {

  // delete all the items in the list
  int n = items();
  for(int i=0; i < n; i++)
    delete menuItems[i];

  if(menuName)
    delete [] menuName;
}


// return the number of items in this list
int PopupMenu::items(void) {  return menuItems.num(); }


// return the Nth item, or NULL if illegal N given
PopupMenuItem *PopupMenu::menu_item(int n) {
  return (n >= 0 && n < items() ? menuItems[n] : NULL);
}


// return the menu item with the given label or NULL if not found
PopupMenuItem *PopupMenu::menu_item(char *nm) {
  if(!nm)
    return NULL;

  int n = items();
  for(int i=0; i < n; i++) {
    PopupMenuItem *mItem = menuItems[i];
    if(!strcmp(mItem->name(), nm))
      return mItem;
  }

  // if here, not found
  return NULL;
}


// print out the contents of the menu to the given Inform object
int PopupMenu::print(Inform *m, int leading) {

  // print out each item as "label ... command"  or "label => submenu ''
  int n = items();
  for(int i=0; i < n; i++)
    (menuItems[i])->print(m, leading);

  if(n < 1) {
    for(int j=0; j < leading; j++)
      *m << ' ';
    *m << "(empty)" << sendmsg;
  }

  return(n > 0);
}

// add a regular item; return the return code for the item
// arguments: item name, text command, whether to display it or not
int PopupMenu::add_item(char *nm, char *txt, int disp, int check, int con) {

  // create a new item
  PopupMenuItem *mItem = new PopupMenuItem(nextReturnCode++, nm, txt);
  mItem->showing(disp);
  mItem->show_check(check);
  mItem->checked(con);

  // reset the return code, if necessary
  if(nextReturnCode > 99999)
    nextReturnCode = 1;

  // add it to our list
  menuItems.append(mItem);

  // return the new return code
  return mItem->return_code();
}


// add a separator; return success
int PopupMenu::add_separator(void) {

  // create a new item
  PopupMenuItem *mItem = new PopupMenuItem();

  // add it to our list
  menuItems.append(mItem);

  // return the new return code
  return TRUE;
}


// add a submenu; return success
int PopupMenu::add_submenu(PopupMenu *pm, int disp, int check, int con) {

  // create a new item
  PopupMenuItem *mItem = new PopupMenuItem(pm);
  mItem->showing(disp);
  mItem->show_check(check);
  mItem->checked(con);

  // add it to our list
  menuItems.append(mItem);

  // return the new return code
  return TRUE;
}


// delete the Nth item.  Return success.
int PopupMenu::delete_item(int n) {

  // get the Nth item, if possible
  PopupMenuItem *mItem = menu_item(n);

  // only delete it if n is legal
  if(mItem) {
    delete mItem;
    menuItems.remove(n);
    return TRUE;
  }

  // if here, error
  return FALSE;
}


// given a return code, searches the menu until the first item with that
// code is found.  Returns the item pointer, or else NULL if none are found
// or the given return code is <= 0.
// The 2nd argument is used to determine where to start, if it is NULL, the
// current menu is used.  This is done so this routine can be recursive.
PopupMenuItem *PopupMenu::find_item(int rc, PopupMenu *menu) {
  if(rc <= 0)
    return NULL;

  if(!menu)
    menu = this;

  int n = menu->items();
  for(int i=0; i < n; i++) {
    PopupMenuItem *mItem = menu->menu_item(i);
    if(mItem->return_code() == rc)
      return mItem;
    else if(mItem->is_submenu()) {
      PopupMenuItem *testItem = find_item(rc, mItem->sub_menu());
      if(testItem)
        return testItem;
    }
  }

  // if here, not found
  return NULL;
}


// request the pop-up menu be activated by the windowing system; this
// will either return NULL, or the defined item.
PopupMenuItem *PopupMenu::activate(DisplayDevice *d) {

  // ask the display device to create the pop-up menu
  d->menu_create(this);

  // activate the menu, and return the result
  return find_item(d->menu_activate());
}


/***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: PopupMenu.C,v $
 * Revision 1.1  1995/05/11  23:20:03  billh
 * Initial revision
 *
 ***************************************************************************/
