/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: NameList.c,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 1995/05/11 23:19:47 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * NameList object, which stores a list of unique names indexed in the order
 * they are added.  For each name, which acts as a key, there is an associated
 * integer value.
 *
 ***************************************************************************/

#include <string.h>
#include "ResizeArray.h"
#include "Inform.h"
#include "utilities.h"


////////////////////  constructor
// starts with no names, which are then added
// via the 'add_name' routine.  When all names have been added, the
// routine 'init' is called, which creates indices for the names.
template<class T>
NameList<T>::NameList(void)
	: names(NLISTSIZE), Data(NLISTSIZE), sortedOrder(NLISTSIZE) {
  MSGDEBUG(1,"Created new NameList ..." << sendmsg);
  Num = 0;
}


////////////////////  destructor
// destructor for this class ... deallocates space if necessary
template<class T>
NameList<T>::~NameList(void) {
  MSGDEBUG(3,"Deleting NameList with " << num() << " names ..." << sendmsg);
  
  for(int i=0; i < num(); i++) {
    if(names[i])
      delete [] names[i];
  }
}

  
///////////////////
///////////////////  public routines
///////////////////


// add a new name to the list, with a given associated value.
// Return the index.  If already in the list, return the current index.
template<class T>
int NameList<T>::add_name(char *nm, const T &val) {
  // temporary storage with spaces stripped from beginning and end
  char tmpnm[128];
    
  // make sure the name is legal
  if(!nm)
    return (-1);      

  // strip leading spaces from the name
  char *s = tmpnm;
  while (*nm && *nm == ' ')		// skip past whitespace
    nm++;
  while (*nm)				// copy the string
    *(s++) = *(nm++);
  *s = 0; 				// terminate the copied string
  while(s != tmpnm && *(--s) == ' ')	// remove spaces at end of string
    *s = '\0';

  // scan through the list until a name is found, or the end is reached
  for(int npos=0; npos < num(); npos++) {
    if(!strcmp(tmpnm, names[npos])) {
      // string found, return index
      MSGDEBUG(3, "NameList: found existing name '" << tmpnm << "' at pos ");
      MSGDEBUG(3, npos << sendmsg);
      return npos;
    }
  }

  // if here, string not found; append new one, and return index = num() - 1
  names[Num] = stringdup(tmpnm);
  Data[Num] = val;
  
  // put name in sorted order ... just scan through list until a name
  // is found that is lexigraphically > nm, and put name just before it
  int j, k;
  for(j=0; j < Num; j++) {
    if(strcmp(names[sortedOrder[j]], names[Num]) > 0)
      break;		// found name > new name
  }
  for(k=Num; k > j; k--)
    sortedOrder[k] = sortedOrder[k-1];  // move everything past this point dn
  sortedOrder[j] = Num;		  // put in position of new name
  
  MSGDEBUG(3, "NameList: Added new name '" << tmpnm << "' at pos ");
  MSGDEBUG(3, Num << sendmsg);

  return Num++;
}


// return the name (null-terminated) for given typecode
template<class T>
char *NameList<T>::name(int a) {
  if(a >= 0 && a < num()) {
    return names[a];
  } else {
    msgErr << "NameList: No name with index " << a << " found in list with ";
    msgErr << num() << " names." << sendmsg;
    return NULL;	// not found
  }
}


// return the Nth name in sorted order (instead of order in which added)
template<class T>
char *NameList<T>::sort_name(int a) {
  if(a >= 0 && a < num()) {
    return names[sortedOrder[a]];
  } else {
    msgErr << "NameList: No sorted name with index " << a;
    msgErr << " found in list with " << num() << " names." << sendmsg;
    return NULL;	// not found
  }
}


// return the type index for the given name.  If the second argument is
// given and is > 0, it is used as the max length of the names to check
// for a match.  If is is <= 0, an exact match must be found.
//	returns (-1) if no match is found
template<class T>
int NameList<T>::typecode(char *nm, int maxlen) {
  if(!nm)
    return (-1);
  if(maxlen <= 0) {
    for(int i=0; i < num(); i++) {
      if(!strcmp(nm, names[i]))
        return i;
    }
  } else {
    for(int i=0; i < num(); i++) {
      if(!strncmp(nm, names[i], maxlen))
        return i;
    }
  }
  return (-1);
}


// returns actual index of name in Nth sorted position
template<class T>
int NameList<T>::sort_typecode(int a) {
  if(a >= 0 && a < num()) {
    return sortedOrder[a];
  } else {
    msgErr << "NameList: No index at sorted position " << a;
    msgErr << " found in list with " << num() << " names." << sendmsg;
    return (-1);	// not found
  }
}


// returns the data for the given name.  If the second argument is
// given and is > 0, it is used as the max length of the names to check
// for a match.  If is is <= 0, an exact match must be found.
template<class T>
T NameList<T>::data(char *nm, int maxlen) {
  if(!nm)
    return Data[0];
  if(maxlen > 0) {
    for(int i=0; i < num(); i++) {
      if(!strcmp(nm, names[i]))
        return Data[i];
    }
  } else {
    for(int i=0; i < num(); i++) {
      if(!strncmp(nm, names[i], maxlen))
        return Data[i];
    }
  }
  return Data[0];
}


// returns the data for the given index
template<class T>
T NameList<T>::data(int a) {
  if(a >= 0 && a < num())
    return Data[a];
  else {
//    msgErr << "NameList: Illegal index " << a << " while getting data.";
//    msgErr << sendmsg;
    return Data[0];
  }
}


// set the data value for the given index
template<class T>
void NameList<T>::set_data(int a, const T &val) {
  if(a >= 0 && a < num())
    Data[a] = val;
  else {
//    msgErr << "NameList: Illegal index " << a << " while setting data.";
//    msgErr << sendmsg;
  }
}

/* REVISION HISTORY:********************************************************
 *
 * $Log: NameList.c,v $
 * Revision 1.3  1995/05/11  23:19:47  billh
 * Moved log messages to end of file.
 *
 * Revision 1.2  95/03/24  18:50:59  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.1  1995/01/29  19:58:41  billh
 * Initial revision
 *
 * Revision 1.7  94/11/12  10:24:13  dalke
 * fixed up how spaces are deleted
 * 
 * Revision 1.6  1994/11/12  04:04:20  dalke
 * fixed problem with typecode
 *
 * Revision 1.5  1994/11/10  17:20:47  billh
 * Strips leading and trailing spaces from added names.
 *
 * Revision 1.4  94/11/02  01:37:23  billh
 * Added option to search for a name by just the first N characters, via
 * optional 2nd argument to 'name' and 'data'.
 * 
 * Revision 1.3  94/09/24  20:12:57  billh
 * Now a template; this is a basic associative array, with a character
 * string as the main lookup key, and a specified type for the data stored
 * for each name.
 * 
 * Revision 1.2  94/09/23  00:54:55  billh
 * names and Data now ResizeArray objects, instead of using a
 * linked list.  Much nicer, much shorter, and actually works.
 * 
 * Revision 1.1  94/09/17  09:11:36  billh
 * Initial revision
 * 
 ***************************************************************************/
