/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Global.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.46 $	$Date: 1995/05/13 01:35:58 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Global data structures, configuration settings, and macros for the 
 * vmd library.
 *
 * Where most 'optional components' are processed ... options which are not
 * included are ifdef'd out.
 ***************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

// project include files
#include "Global.h"
#include "Inform.h"
#include "VMDTitle.h"
#include "DisplayDevice.h"
#include "NormalScene.h"
#include "ColorList.h"
#include "LightList.h"
#include "CommandQueue.h"
#include "UIList.h"
#include "UIText.h"
#include "UIVR.h"
#include "Stage.h"
#include "Axes.h"
#include "TrackerList.h"
#include "MoleculeList.h"
#include "Mouse.h"
#include "ConfigList.h"
#include "GeometryList.h"
#include "FileRenderList.h"
#include "startup.h"

// CAVE-specific files
#ifdef VMDCAVE
#include "cave.h"
#include "CaveDisplayDevice.h"
#include "CaveRoutines.h"
#include "CaveScene.h"
#endif

// GL-specific files
#ifdef VMDGL
#include "GLDisplayDevice.h"
#endif

// use FORMS GUI?  If so, include GL and FORMS-specific objects
#ifdef VMDFORMS
#include "AnimateFormsObj.h"
#include "ColorFormsObj.h"
#include "DisplayFormsObj.h"
#include "EditFormsObj.h"
#include "FilesFormsObj.h"
#include "GeometryFormsObj.h"
#include "GraphicsFormsObj.h"
#include "MainFormsObj.h"
#include "MolFormsObj.h"
#include "RenderFormsObj.h"
#include "TrackerFormsObj.h"

#ifdef VMDREMOTE
#include "RemoteFormsObj.h"
#include "SimFormsObj.h"
#endif

#endif

//
// main global variables
//

char *myName=NULL;		// name of this program
char *myPath=NULL;		// directory location of this program
char *myTempDir=NULL;		// directory for temp file storage
char *dataPath=NULL;		// where VMD data files are found
char *vmdBabelBin=NULL;         // location of the Babel executable
char *vmdHTMLViewer=NULL;       // name of HTML viewer to use
DisplayDevice *display=NULL;	// where the images are rendered
Scene *scene=NULL;		// the list of all Displayable objects to draw
LightList *lights=NULL;		// the list of all Lights to be drawn
ColorList *colors=NULL;			// the list of all color objects
Axes *axes=NULL;			// set of axes shown on display
Stage *stage=NULL;			// stage displayed below objects
CommandQueue *commandQueue=NULL;	// the command processor
MoleculeList *moleculeList=NULL;	// list of all molecules
GeometryList *geometryList=NULL;	// list of all geometry monitors
VMDTitle *vmdTitle = NULL;		// the spinning title
ConfigList *configList = NULL;		// the configuration list
FileRenderList *fileRenderList = NULL;  // the ways to render scenes to file
UIList *uiList=NULL;			// collection of all UI objects
UIVR *uiVR=NULL;			// 3D user interface
UIText *uiText = NULL;			// text interface
Mouse *mouse = NULL;			// mouse interface

#ifdef VMDREMOTE
#include "RemoteList.h"
Remote *remote = NULL;		// current remote connection setup
RemoteList *remoteList = NULL;	// list of running simulations
#endif

#ifdef VMDEXTERNAL
#include "UIExternal.h"
UIExternal *uiExternal = NULL;          // for "external" interfaces
#endif

/*****************************************************************/

//
// global routines
//

// print out program title
void VMDtitle(void) {
  msgInfo << VERSION_MSG << "\n";
  msgInfo << "Authors: Andrew Dalke, Bill Humphrey, Rick Kufrin\n";
  msgInfo << "-------------------------------------------------\n";
  msgInfo << sendmsg;
}


// given a string, return a new one with the temp dir name prepended.
// This will use an internal static buffer, so it is NOT reentrant.
char *VMDtempFile(const char *s) {
  static char tmpfilebuf[1024];
  
  // copy in temp string
  strcpy(tmpfilebuf, myTempDir);
  strcat(tmpfilebuf, "/");
  strncat(tmpfilebuf, s, 1022 - strlen(myTempDir));
  tmpfilebuf[1023] = '\0';

  // return converted string
  return tmpfilebuf;
}


// initialization routine for the library globals
void VMDinit(int argc, char *argv[]) {

  // initialize Inform objects ... must be done FIRST
  msgInfo.need_newline(TRUE);
  msgWarn.need_newline(TRUE);
  msgErr.need_newline(TRUE);
  msgDebug.need_newline(TRUE);
#ifdef VMDDEBUG
  msgDebug.on(TRUE);
  msgDebug.output_level(2);
#else
  msgDebug.on(FALSE);
  msgDebug.output_level(1);
#endif

  MSGDEBUG(1, "Beginning main initialization; " << argc - 1);
  MSGDEBUG(1, " command-line parameter(s)." << sendmsg);

  // initialize name and path variables
  breakup_filename(argv[0], &myPath, &myName);
  
  //
  // process environment variables
  //
  VMDgetEnvironment();

  //
  // parse the command-line options
  //
  VMDparseCommandLine(argc, argv);

  // reads the definitions in the given init file at program start.
  VMDreadInit();

  //
  // print out title of program
  //
  if(showTitle == TITLE_ON)
    VMDtitle();

  //
  // create graphics context and global graphics objects
  //

  // initialize the display
  display = NULL;
  int *dloc = (displayLoc[0] > 0 ? displayLoc : NULL);
  int *dsiz = (displaySize[0] > 0 ? displaySize : NULL);
  if(dloc || dsiz);		// keep compiler happy if no GUI or graphics

  // check for a CAVE display
  if (which_display == DISPLAY_CAVE) {
#ifdef VMDCAVE
    CAVEConfigure(&argc, argv, NULL);
    grab_CAVE_memory(80); // 80 megs of memory, at least initially
    display = new CaveDisplayDevice;
#else
    msgErr << "Not compiled with the CAVE options set." << sendmsg;
    which_display = DISPLAY_WIN;
#endif
  }
  
  // check for a standard monitor display
  if(which_display == DISPLAY_WIN) {
#ifdef VMDGL
    display = new GLDisplayDevice(dsiz, dloc);
#else
    // here would go command to create non-GL window display, i.e. X
    // since not available yet, switch to using text
    which_display = DISPLAY_TEXT;
#endif
  }
  
  // still no display?  if not, create 'empty' display
  if(which_display == DISPLAY_TEXT || !display) {
    // neither the CAVE nor regular window is to be used ...
    // create a 'default' display, which really does nothing but can be used
    // to test the program
    which_display = DISPLAY_TEXT;
    display = new DisplayDevice("Default Display");
  }

  // set display screen config, if possible
  display->screen_height(displayHeight);
  display->distance_to_screen(displayDist);

  //
  // create the scene
  //

  scene = NULL;

#ifdef VMDCAVE
  if (which_display == DISPLAY_CAVE) {
    scene = new CaveScene;
    msgInfo << "Cave scene created." << sendmsg;
    CAVEInit();  // split into the various drawing processes
    msgInfo << "CAVE Initialized" << sendmsg;
    CAVEDisplay(cave_renderer, 0); // set the fctn ptr for the renders
    msgInfo << "There are " << CAVENumRenderProcesses() <<
      " children." <<  sendmsg;
  }
#endif

  // make sure there is a scene object.  If none yet, create default one.
  if(!scene)
    scene = new NormalScene;

  //
  // create other global objects for the program
  //

  // create color and light lists; turn on lights 0 and 1
  colors = new ColorList(scene);
  lights = new LightList(scene);
  lights->set(0);
  lights->current()->on();
  lights->set(1);
  lights->current()->on();

  // create other useful graphics objects, and tell them what colors to use
  axes = new Axes(scene);
  stage = new Stage(scene);
  stage->off();			// by default, stage is not displayed

  vmdTitle = new VMDTitle(scene);
  if(showTitle == TITLE_OFF)
    vmdTitle->off();

  // create the list of molecules (initially empty)
  moleculeList = new MoleculeList(scene);

  // create the list of geometry monitors (initially empty)
  geometryList = new GeometryList(scene);

  // make the classes which can render scenes to different file formats
  fileRenderList = new FileRenderList;

  // tell all these new objects to use the color list we created
  display->use_colors(colors);
  axes->use_colors(colors);
  stage->use_colors(colors);
  geometryList->use_colors(colors);
  moleculeList->use_colors(colors);		// MUST be called right after
						// this object is created!

#ifdef VMDREMOTE
  // create the list of running simulations (initially empty)
  remoteList = new RemoteList;
#endif

  //
  // initialize UI ... creates UIList and CommandQueue, trackers and tools
  //
  VMDinitUI();
  
  //
  // flush command queue, and make system ready to enter event loop
  //
  VMDupdate(FALSE);
  
  // set UIText to read in startup file, if possible, and execute any
  // initial commands.
  VMDreadStartup();
}


// initialize the user interface
void VMDinitUI(void) {

  MSGDEBUG(1, "Creating UI objects ..." << sendmsg);

  // create the user interface list ... all UI's register with this object
  uiList = new UIList;
  
  // create command queue
  commandQueue = new CommandQueue(uiList);

  // text user interface
  uiText = new UIText(uiList, commandQueue);  
  
  // mouse user interface
  mouse = new Mouse(uiList, commandQueue, display);

  // create the 3D (virtual reality-like) user interface
  uiVR = new UIVR(uiList, commandQueue);

  // create graphical user interfaces, if necessary
#ifdef VMDGUI
  if(which_display == DISPLAY_WIN) {

    // if we are using the FORMS library ...
#ifdef VMDFORMS
    new AnimateFormsObj(uiList, commandQueue, FALSE, TRUE);
    new ColorFormsObj(uiList, commandQueue, FALSE, TRUE);
    new DisplayFormsObj(uiList, commandQueue, FALSE, TRUE);
    new EditFormsObj(uiList, commandQueue, FALSE, TRUE);
    new FilesFormsObj(uiList, commandQueue, FALSE, TRUE);
    new GeometryFormsObj(uiList, commandQueue, FALSE, TRUE);
    new GraphicsFormsObj(uiList, commandQueue, FALSE, TRUE);
    new MainFormsObj(uiList, commandQueue, FALSE, TRUE);
    new MolFormsObj(uiList, commandQueue, FALSE, TRUE);
    new RenderFormsObj(uiList, commandQueue, FALSE, TRUE);
    new TrackerFormsObj(uiList, commandQueue, FALSE, TRUE);
#ifdef VMDREMOTE
    new RemoteFormsObj(uiList, commandQueue, FALSE, TRUE);
    new SimFormsObj(uiList, commandQueue, FALSE, TRUE);
#endif

#endif
  }
#endif

  // add in the External interface, only if VMDEXTERNAL is defined
#ifdef VMDEXTERNAL
  uiExternal = new UIExternal(uiList, commandQueue);
#endif

  MSGDEBUG(1, "Initializing UI objects ..." << sendmsg);

  // initialize all the interfaces
  uiList->init_UI();
}


// redraw the screen and update all things that need updatin'
int VMDupdate(int check_for_events) {
			
  MSGDEBUG(3, "Beginning main event loop processing ..." << sendmsg);

  if(uiList) {
    if(check_for_events) {
      uiList->check_event_UI();
      commandQueue->execute_all(uiList);
    }
    uiList->update_UI();
  } else {
    msgErr << "Help!  No UIList!!" << sendmsg;
    return FALSE;
  }

  if(scene && display) {
    scene->prepare(display);
    if (which_display != DISPLAY_CAVE) {  // in the CAVE, renders are done
      scene->draw(display);                  // by spawned processes
    }
  } else {
    msgErr << "Help!  No Scene OR DisplayDevice!!" << sendmsg;
    return FALSE;
  }
  
  // turn off the spinning vmd when molecules are loaded
  if (moleculeList->num() > 0)
    vmdTitle->off();

  return TRUE;
}


// given a string, start up a HTML viewer with the string as an argument.
// Return success.
int VMDreadHTML(char *cmd) {
  static pid_t childProc = (-1);

  if(!vmdHTMLViewer || strlen(vmdHTMLViewer) < 1) {
    msgErr << "No HTML viewer has been specified.  Set the environment ";
    msgErr << "variable VMDHTMLVIEWER to the name of a viewer installed ";
    msgErr << "on your system." << sendmsg;
    return FALSE;
  }

  msgInfo << "Loading help file ..." << sendmsg;

  // if a child process has already been started, kill it first ...
  if(childProc > 0)
    kill(childProc, SIGHUP);

  // fork off a new process now ...
  childProc = fork();

  // and deal with the three possible cases.
  if(childProc > 0) {
    // case 1: this is the parent ... nothing to do, return success.

  } else if(childProc == 0) {
    // case 2: this is the child ... exec the viewer if possible
    int retval = execlp(vmdHTMLViewer, vmdHTMLViewer, cmd, (char *)NULL);

    // if we're here, the exec failed ...
    msgErr << "Unable to start HTML viewer '" << cmd << "': cannot load.";
    msgErr << sendmsg;
    exit(1);

  } else {
    // case 3: the fork failed
    msgErr << "Unable to start HTML viewer '" << cmd << "': cannot fork.";
    msgErr << sendmsg;
    return FALSE;
  }

  // if here, we were successful.
  return TRUE;
}


// exit the program normally; first delete all the necessary objects
void VMDexit(char *exitmsg, int exitcode) {

  msgInfo << VERSION_MSG << "\n";
  if(exitmsg)
    msgInfo << exitmsg << "\n";
  else
    msgInfo << "Exiting ..." << sendmsg;

  // delete all objects we created during initialization

#ifdef VMDREMOTE
  // create the list of running simulations (initially empty)
  if(remoteList) delete remoteList;
  remoteList = NULL;
#endif

#ifdef VMDEXTERNAL
  if(uiExternal) delete uiExternal;
#endif

  if(fileRenderList) delete fileRenderList;
  if(trackerList) delete trackerList;
  if(mouse) delete mouse;
  if(uiText) delete uiText;
  if(uiList) delete uiList;
  if(uiVR) delete uiVR;
  if(geometryList) delete geometryList;
  if(moleculeList) delete moleculeList;
  if(commandQueue) delete commandQueue;
  if(vmdTitle) delete vmdTitle;
  if(stage) delete stage;
  if(axes) delete axes;
  if(lights) delete lights;
  if(colors) delete colors;
  if(scene) delete scene;
  if(display) delete display;
#ifdef VMDCAVE
  if (which_display == DISPLAY_CAVE) {  // call the CAVE specific exit
    CAVEExit();
  }
#endif
  exit(exitcode);
}


// exit the program abnormally; print out special message
void VMDdie(char *exitmsg) {
  msgErr << "Subroutine DIE called ..." << sendmsg;
  if(exitmsg)
    msgErr << exitmsg << sendmsg;
  VMDexit("Program terminating due to fatal error.", 1);
}

/* REVISION HISTORY:********************************************************
 *
 * $Log: Global.C,v $
 * Revision 1.46  1995/05/13  01:35:58  billh
 * Help is now displayed by launching an HTML viewer (Mosaic by default)
 * which will display the file /usr/local/lib/vmd/vmd_help.html
 *
 * Revision 1.45  95/05/11  22:49:24  billh
 * Uses new Mouse constructor; uiVR now created in Global; added VMDGL
 * option to allow separate specification of GUI and graphics methods.
 * 
 * Revision 1.44  95/04/06  17:54:50  dalke
 * Finished interface
 * 
 * Revision 1.43  1995/04/04  21:05:42  billh
 * Added VMDTMPDIR env variable, and routine VMDgetEnvironment for
 * processing env variables.  Made global char string myTempDir to
 * hold a directory for temporary files, and routine VMDtempFile to
 * generate a temp filename from just a single string.
 *
 * Revision 1.42  95/03/28  03:48:15  billh
 * CommandQueue changed to require that the constructor is given the pointer
 * to the UIList to use.
 * 
 * Revision 1.41  95/03/24  18:50:02  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.40  1995/03/17  22:25:59  billh
 * Stage now turned off at start.
 *
 * Revision 1.39  1995/03/04  05:13:15  billh
 * Added geometry menu.
 *
 * Revision 1.38  95/02/26  22:21:50  billh
 * Now creates tracker menu (TrackerFormsObj).
 * 
 * Revision 1.37  1995/02/22  08:43:51  dalke
 * added uiExternal
 *
 * Revision 1.36  1995/01/09  08:50:37  billh
 * Moved code to parse cmd-line options and read init and startup files to
 * the 'startup.C' file.
 *
 * Revision 1.35  1994/12/06  08:23:09  billh
 * Added color menu, and command to tell certain objects to use the ColorList
 *
 * Revision 1.34  94/11/24  07:27:03  dalke
 * Added fileRenderList
 * 
 * Revision 1.33  1994/11/22  08:29:22  billh
 * Forms only created when a VMDFORMS specified, and a WINDISPLAY requested.
 * Added EditFormsObj menu.
 *
 * Revision 1.32  1994/11/09  22:40:01  dalke
 * cleaned up CAVE calls
 *
 * Revision 1.31  1994/11/09  07:50:19  billh
 * changed CAVE memory allocation size
 *
 * Revision 1.30  1994/11/09  05:20:45  dalke
 * rearranged calls to CAVE scene and display
 *
 * Revision 1.29  1994/11/09  03:46:08  billh
 * If no display requested (or available due to no options compiled in),
 * creates an 'empty' display.
 *
 * Revision 1.28  94/11/02  08:37:37  billh
 * Took out unneeded msg printout.
 * 
 * Revision 1.27  1994/11/02  07:33:03  billh
 * Added code to get 'display', 'scrheight', and 'scrdist' init variables
 * from configList, and to set these values in the initial display.
 *
 * Revision 1.26  94/11/02  05:18:27  dalke
 * Added configList read from .vmd_init files
 * Added configList option to turn off the title
 * 
 * Revision 1.25  1994/11/01  11:24:02  dalke
 * Working on the CAVE
 *
 * Revision 1.24  1994/10/29  02:24:45  billh
 * Includes GraphicsFormsObj menu now.
 *
 * Revision 1.23  1994/10/28  18:36:05  billh
 * Updated due to conversion of Mouse to a general base class.  Window
 * events have also been made non-GL-specific.
 *
 * Revision 1.22  1994/10/22  18:24:34  dalke
 * Made VMDTitle disappear when a molecule is loaded
 *
 * Revision 1.21  1994/10/21  03:49:55  billh
 * Added remoteList global variable, and sim control menu.
 *
 * Revision 1.20  1994/10/20  01:32:29  billh
 * Added Animate menu.
 *
 * Revision 1.19  1994/10/07  08:39:23  dalke
 * added VMDTitle
 *
 * Revision 1.18  1994/10/04  06:38:23  billh
 * Added VMDDEBUG define, to set whether to include debugging statements
 *
 * Revision 1.17  1994/10/02  05:16:14  billh
 * Added RemoteFormsObj menu.
 *
 * Revision 1.16  94/10/01  03:09:18  billh
 * Improved setup of compiler directives.  Now the VMDFORMS define also
 * refers to using a GLDisplayDevice.  Also, Mouse object only defined
 * and used if VMDFORMS is defined.
 * 
 * Revision 1.15  1994/09/30  23:37:30  billh
 * put ifdef around remote definition.
 *
 * Revision 1.14  1994/09/30  23:26:41  billh
 * Added remote object reference; soon to add Files menu.
 *
 * Revision 1.13  1994/09/29  19:20:10  dalke
 * Added some (not all) CAVE routines
 *
 * Revision 1.12  1994/09/29  07:11:38  billh
 * Added DisplayFormsObj and MolFormsObj UI objects.
 *
 * Revision 1.11  94/09/24  20:30:15  billh
 * Updated to check for optional components, and compile accordingly;
 * added '-debug [n]' and '-? or -h' command-line options.
 * 
 * Revision 1.10  94/09/23  00:51:17  billh
 * Removed Enterprise; fixed named used for CAVE define.
 * 
 * Revision 1.9  94/09/22  05:26:42  dalke
 * Added "-disp cave" option and related CAVE code
 * 
 * Revision 1.8  1994/09/17  10:08:12  dalke
 * moved trakerList to UIVR, added uiVR
 *
 * Revision 1.7  1994/09/17  09:07:08  billh
 * Put in command to delete moleculeList when quitting.
 *
 * Revision 1.6  1994/09/17  09:02:48  billh
 * Put in moleculeList object, and code to create it.
 *
 * Revision 1.5  1994/09/17  05:00:13  dalke
 * Added UIVR to UIList
 *
 * Revision 1.4  1994/09/14  04:20:32  billh
 * Took out automatic turning on of main menu.
 *
 * Revision 1.3  1994/09/07  07:48:07  dalke
 * added trackerList
 *
 * Revision 1.2  1994/09/06  03:40:56  billh
 * Added code to parse command-line options, and to read init file and
 * startup file (init file processing must still be added).
 *
 * Revision 1.1  1994/08/24  03:10:37  billh
 * Initial revision
 *
 ***************************************************************************/
