/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: DisplayDevice.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.11 $	$Date: 1995/05/11 22:37:26 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * DisplayDevice - abstract base class for all particular objects which
 *	can process a list of drawing commands and render the drawing
 *	to some device (screen, file, preprocessing script, etc.)
 *
 ***************************************************************************/
#ifndef DISPLAYDEVICE_H
#define DISPLAYDEVICE_H

#include <math.h>
#include "Matrix4.h"
#include "Stack.h"
#include "ColorList.h"
#include "ColorUser.h"
#include "utilities.h"

class PopupMenu;


// constants for this object
#define DISP_LIGHTS	4


class DisplayDevice : public ColorUser {

public:
  // name of this device
  char *name;

  // enum for left or right eye
  enum DisplayEye { NOSTEREO, LEFTEYE, RIGHTEYE };

  // different buttons of the mouse
  enum Buttons { B_LEFT, B_MIDDLE, B_RIGHT, B2_LEFT, B2_MIDDLE, B2_RIGHT,
	B_F1, B_F2, B_F3, B_F4,  B_F5,  B_F6,
	B_F7, B_F8, B_F9, B_F10, B_F11, B_F12,
	B_ESC, TOTAL_BUTTONS };
  
  // possible window events
  enum EventCodes { WIN_REDRAW, WIN_LEFT, WIN_MIDDLE, WIN_RIGHT,
  	WIN_MOUSEX, WIN_MOUSEY, WIN_KEYBD,
	WIN_UPDATE, WIN_INPUTCHANGE, WIN_EVENTCODES, WIN_NOEVENT };

protected:
  //
  // flags for what capabilities this object has
  //

  // can this device display 2D as well as 3D?
  int has2D;

  // can this antialias or depth-cue?  If so, current state.
  int aaAvailable, cueingAvailable;
  int aaEnabled,   cueingEnabled;

  //
  // current display state
  //

  // color category for use by the display (if < 0, no category to use)
  int colorCat;

  // position and size of device, in 'pixels'
  long xOrig, yOrig, xSize, ySize;

  // background color
  float backColor[3];

  // current dimension of display (can change via prepare routines)
  int Dim;

  // current transformation matrix for the display (NOT the projection matrix)
  Stack<Matrix4> transMat;

  // drawing characteristics ... line style, sphere resolution, etc.
  int lineStyle, lineWidth;
  int sphereRes, sphereMode;

  //
  // eye position, clipping planes, and viewing geometry data
  //

  float eyePos[3];		// current location of the viewer's eye
  float nearClip, farClip;	// dist from eye to near and far clip plane
  float vSize;			// vertical size of 'screen'
  float zDist;			// distance to 'screen' relative to origin
  float Aspect;			// current aspect ratio
  float cpUp, cpDown, cpLeft, cpRight;	// distances to near frustrum base

  //
  // stereo display data
  //

  int inStereo;			// current stereo mode (0 = non-stereo)
  int stereoModes;		// total number of stereo modes (inc mono)
  char **stereoNames;		// pointer to stereo mode names
  float eyeSep;			// distance between eyes for stereo display
  float eyeDist;		// distance from eye to focal point
  float eyeDir[3];		// direction viewer is looking
  float upDir[3];		// direction which is 'up'
  float eyeSepDir[3];		// vector from eye position to right eye
  				// magnitude is 1/2 eyeSep 
  //
  // data for lights
  //

  int lightDefined[DISP_LIGHTS];
  int lightOn[DISP_LIGHTS];
  float lightColor[DISP_LIGHTS][3];
  float lightPos[DISP_LIGHTS][3];

  //
  // data for materials
  //

  int matDefined[MAXCOLORS];	// has the material been defined?
  int materialOn;		// which mat def are we using?
  int materialsActive;		// are we using mat def's at all?
  float matData[MAXCOLORS][COLOR_ITEMS];	// data for each material defn

  //
  // data for picking objects
  //

  float pickRegion;		// sensitivity of pick: 0 ... 1

  //
  // protected nonvirtual routines
  //
  
  // calculate the position of the near frustrum plane, based on curr values
  // of Aspect, vSize, zDist, nearClip and eyePosition
  void calc_frustrum(void);

  // calculate eyeSepDir, based on up vector and look vector
  // eyeSepDir = 1/2 * eyeSep * (lookdir x updir) / mag(lookdir x updir)
  void calc_eyedir(void);

  //
  // protected virtual routines
  //
  
  // virtual routines to deal with light sources at device level
  // all return success of operation
  virtual int do_define_light(int n, float *color, float *position);
  virtual int do_activate_light(int n, int turnon);

  // virtual routines to deal with materials at device level
  // all return success of operation
  virtual int do_define_material(int n, float *data);
  virtual int do_activate_material(int n, int turnon);

  // virtual function which is called when a new colorlist is provided.
  // This should NOT have to be changed by derived classes.
  virtual void do_use_colors(void);

  // do action due to the fact that a color for the given ColorList for
  // the specified category has changed
  // This should NOT have to be changed by derived classes.
  virtual void do_color_changed(ColorList *, int);

public:
  DisplayDevice(char *);
  virtual ~DisplayDevice(void);
  
  // copies over all relevant properties from one DisplayDevice
  // to another
  DisplayDevice& operator=(DisplayDevice &);

  //
  // query routines about the device's settings
  //
  
  int has_2D(void) { return has2D; }
  int dim(void) { return Dim; }
  int aa_available(void) { return aaAvailable; }
  int cueing_available(void) { return cueingAvailable; }
  int aa_enabled(void) { return aaEnabled; }
  int cueing_enabled(void) { return cueingEnabled; }

  //
  // event handling routines
  //

  // queue the standard events (need only be called once ... but this is
  // not done automatically by the window because it may not be necessary or
  // even wanted)
  virtual void queue_events(void);

  // test if there is an event ready; if so, returns the type of event ready
  virtual int test_events(void);

  // read the next event ... returns an event type (one of the above ones),
  // and a value.  Returns success, and sets arguments.
  // NOTE: THIS SHOULD NOT BLOCK ... IT SHOUULD RETURN FALSE IF NO EVENT TO
  // READ.
  virtual int read_event(long &, long &);

  // 
  //
  // get the current state of the device's pointer (i.e. cursor if it has one)
  //

  virtual int x(void);		// abs pos of cursor from lower-left corner
  virtual int y(void);		// same, for y direction
  virtual int button_down(int);	// whether a button is currently pressed

  // allow the user to define a special shape for the cursor ... the data
  // is assumed to consist of a single-dim array of 32 unsigned shorts; the
  // first 16 define a 16x16-bit (row-major) pattern, the last 16 are a
  // "second layer" which may be drawn in a different color
  // args: which shape (>0, with 0 the "default" case which cannot be changed)
  // the bitmap data, and the "hot spot" from the lower-left corner
  virtual void change_cursor_shape(int, unsigned short *, int, int);

  // set the Nth cursor shape as the current one.  If no arg given, the
  // default shape (n=0) is used.
  virtual void set_cursor(int = 0);

  //
  // get/set the background color
  //

  void get_background(float& r, float& g, float& b) {
    r = backColor[0];  g = backColor[1];  b = backColor[2];
  }
  void set_background(float r, float g, float b) {
    if(r >= 0.0 && r <= 1.0)  backColor[0] = r;
    if(g >= 0.0 && g <= 1.0)  backColor[1] = g;
    if(b >= 0.0 && b <= 1.0)  backColor[2] = b;
  }

  //
  // virtual routines to create and activate a popup (possibly pull-down) menu 
  //

  // given a PopupMenu definition, create it in the window (but do not
  // activate it yet).  Return success.
  virtual int menu_create(PopupMenu *);

  // activate a previously-created menu.  If the windowing system has no
  // current menu, or cannot do this operation, returns (-1); otherwise,
  // returns the return code for the select (-1 if none selected).
  virtual int menu_activate(void);

  // delete the given menu from the display.
  // If no argument is given, deletes the current menu (if any).
  // Returns success.
  virtual int menu_delete(PopupMenu * = NULL);

  //
  // routines to deal with the clipping planes and eye position
  //

  float aspect(void) { return Aspect; }
  float near_clip(void) const { return nearClip; }
  float far_clip(void) const { return farClip; }
  float clip_width(void) const { return (farClip - nearClip); }
  float addto_near_clip(float ac) { return set_near_clip(nearClip + ac); }
  float addto_far_clip(float ac) { return set_far_clip(farClip + ac); }
  float set_near_clip(float nc) {
    if(nc < farClip && nc > 0.0) {
      nearClip = nc;
      calc_frustrum();
    }
    return nearClip;
  }

  float set_far_clip(float fc) {
    if(fc > nearClip)
      farClip = fc;
    return farClip;
  }

  void move_eye(float x, float y, float z) {
    eyePos[0] = x;  eyePos[1] = y;  eyePos[2] = z;
    calc_frustrum();
  }

  void whereis_eye(float& x, float& y, float& z) const {
    x = eyePos[0];  y = eyePos[1];  z = eyePos[2];
  }

  virtual void reset_eye(void);

  //
  // virtual functions to turn on/off depth cuing and antialiasing
  //
  
  virtual void aa_on(void);
  virtual void aa_off(void);
  virtual void cueing_on(void);
  virtual void cueing_off(void);

  //
  // routines to get/set the 'screen' position
  //

  // return/set the distance from the origin to the screen
  float distance_to_screen(void) { return zDist; }
  void distance_to_screen(float zd) {
    zDist = zd;
    calc_frustrum();
  }
  
  // return the height of the screen
  float screen_height(void) { return vSize; }
  void screen_height(float vs) {
    if(vs > 0.0) {
      vSize = vs;
      calc_frustrum();
    }
  }

  // a) specify aspect ratio
  void set_screen_pos(float vsize, float zdist, float asp) {
    Aspect = asp;  vSize = vsize;  zDist = zdist;
    calc_frustrum();    
  }
  
  // b) have device provide aspect ratio
  void set_screen_pos(float vs, float zd) { set_screen_pos(vs, zd, aspect()); }
  
  // c) just specify aspect ratio
  void set_screen_pos(float asp) { set_screen_pos(vSize, zDist, asp); }
  
  // d) do not specify anything
  void set_screen_pos(void) { set_screen_pos(vSize, zDist); }

  //
  // routines to deal with stereo display
  //
  
  // change to a different stereo mode (0 means 'off')
  virtual void set_stereo_mode(int = 0);

  // current stereo mode ... 0 means non-stereo, others device-specific
  int stereo_mode(void) { return inStereo; }
  
  // number of different stereo modes supported ... 0 means no stereo
  int num_stereo_modes(void) { return stereoModes; }
  
  // return stereo name string, if possible
  char *stereo_name(int n) {
    char *retval = stereoNames[0];
    if(n >= 0 && n < stereoModes)
      retval = stereoNames[n];
    return retval;
  }

  // return focal length
  float eye_dist(void) const { return eyeDist; }
  
  // return eye separation
  float eyesep(void) const { return eyeSep; }

  // set eye separation
  float set_eyesep(float newsep) {
    if(newsep >= 0.0) {
      eyeSep = newsep;
      calc_eyedir();
    }
    return eyeSep;
  }

  // changes where the viewer is looking
  void change_look(float x, float y, float z) {
    eyeDir[0] = x;  eyeDir[1] = y;  eyeDir[2] = z;
    eyeDist = sqrtf(x*x + y*y + z*z);
    calc_eyedir();
  }

  // check the viewer direction
  void where_looking(float& x, float& y, float& z) const {
    x = eyeDir[0];  y = eyeDir[1];  z = eyeDir[2];
  }

  // changes where the up direction is
  void change_up(float x, float y, float z) {
    upDir[0] = x;  upDir[1] = y;  upDir[2] = z;
    calc_eyedir();
  }

  // check what the up direction is
  void where_up(float& x, float& y, float& z) const {
    x = upDir[0];  y = upDir[1];  z = upDir[2];
  }

  // find the direction to the right eye position
  void right_eye_dir(float& x, float& y, float& z) const {
    x = eyeSepDir[0];  y = eyeSepDir[1];  z = eyeSepDir[2];
  }  

  //
  // routines to deal with light sources
  //
  
  void define_light(int n, float *color, float *position);
  void activate_light(int n, int turnon);
  int light_active(int n) { return lightOn[n]; }

  //
  // routines to deal with material definitions
  //

  void define_material(int, float *);
  void activate_material(int n);	// which material
  void activate_material(void);		// current material used (if any)
  void deactivate_materials(void);	// turn off use of materials
  int materials_on(void) { return materialsActive; }
  int curr_material(void) { return materialOn; }

  //
  // routines to deal with getting/setting pick values
  //
  
  float pick_region(void) { return pickRegion; }
  void pick_region(float pr) { pickRegion = pr; }

  //
  // virtual routines to find characteristics of display itself
  //

  // return absolute 2D screen coordinates, given 2D or 3D world coordinates.
  virtual void abs_screen_loc_3D(float *, long *);
  virtual void abs_screen_loc_2D(float *, long *);

  // return relative 2D screen coordinates, given 2D or 3D world coordinates.
  void rel_screen_loc_3D(float *, float *);
  void rel_screen_loc_2D(float *, float *);

  // size of physical display object
  virtual void screen_size_mm(long& x, long& y);
  virtual void screen_size_pixel(long& x, long& y);

  // convert 2D absolute screen coords into relative coords, 0 ... 1
  void rel_screen_pos(float &x, float &y) {
    x = (x - (float)xOrig) / ((float)xSize);
    y = (y - (float)yOrig) / ((float)ySize);
  }

  // convert 2D relative screen coords into absolute coords
  void abs_screen_pos(float &x, float &y) {
    x = x * (float)xSize + (float)xOrig;
    y = y * (float)ySize + (float)yOrig;
  }

  // Given a 3D point (pos A),
  // and a 2D rel screen pos point (for pos B), computes the 3D point
  // which goes with the second 2D point at pos B.  Result returned in 3rd arg.
  virtual void find_3D_from_2D(float *, float *, float *);

  //
  // virtual routines to affect the device's transformation matrix
  //
  
  virtual void push(void);		// push device's curr state on stack
  virtual void pop(void);		// pop device's curr state from stack
  virtual void loadmatrix(Matrix4 &);	// replace trans matrix with given one
  virtual void multmatrix(Matrix4 &);	// multiply trans matrix with given one

  //
  // virtual routines for preparing to draw, drawing, and finishing drawing
  //
  
  virtual void prepare2D(int = TRUE);		// ready to draw 2D
  virtual void prepare3D(int = TRUE);		// ready to draw 3D
  virtual void clear(void);			// erase the device
  virtual void left(void);			// ready to draw left eye
  virtual void right(void);			// ready to draw right eye
  virtual void normal(void);			// ready to draw non-stereo
  virtual void update(int = TRUE);		// finish up after drawing
  virtual void reshape(void);			// refresh device after change
  
  // process list of draw commands
  virtual void render(void *);
  
  // pick objects based on given list of draw commands.
  // arguments are dimension of picking (2 or 3), position of pointer,
  // draw command list, and returned distance from object to eye position.
  // Returns ID code ('tag') for item closest to pointer, or (-1) if no pick.
  // If an object is picked, the eye distance argument is set to the distance
  // from the display's eye position to the object (after its position has been
  // found from the transformation matrix).  If the value of the argument when
  // 'pick' is called is <= 0, a pick will be generated if any item is near the
  // pointer.  If the value of the argument is > 0, a pick will be generated
  // only if an item is closer to the eye position than the value of the
  // argument.
  // For 2D picking, coordinates are relative position in window from
  //	lower-left corner (both in range 0 ... 1)
  // For 3D picking, coordinates are the world coords of the pointer.  They
  //	are the coords of the pointer after its transformation matrix has been
  //	applied, and these coordinates are compared to the coords of the objects
  //	when their transformation matrices are applied.
  virtual int pick(int, float *, void *, float &); 

};

#endif

/***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: DisplayDevice.h,v $
 * Revision 1.11  1995/05/11  22:37:26  billh
 * Moved log messages to the end of the file.
 * Added virtual functions to queue and query events, as well as to test
 * the state of buttons and to return the position of the cursor.
 *
 * Revision 1.10  95/03/24  18:48:59  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.9  1995/02/22  04:03:10  billh
 * Added virtual routine pick, which checks the given draw list for points
 * which can be selected.  Also added routines to convert absolute screen
 * coords <--> relative scaled screen coords.
 * Added virtual routine find_3D_from_2D, which takes a 3D point at pos A,
 * and 2D rel screen position at B, and returns the 3D point corresponding
 * to the 2D point at B in the same plane as point A.
 *
 * Revision 1.8  94/12/06  08:24:25  billh
 * Added ability to create color categories, due to being now derived from
 * ColorUser.
 * Added routines to have a new color list specified, and to check for when
 * a color in a relevant category is changed.
 * 
 * Revision 1.7  94/11/09  03:45:28  billh
 * Provided default versions for all virtual functions, so an 'empty' display
 * can be created that does essentially nothing (for testing purposes).
 * 
 * Revision 1.6  1994/11/02  07:31:36  billh
 * Added routines to set screen height and distance from origin.
 *
 * Revision 1.5  1994/10/05  03:36:24  dalke
 * fixed typo
 *
 * Revision 1.4  1994/10/04  20:29:22  billh
 * Changes to get to compile on HP's ... not there yet,though.
 *
 * Revision 1.3  1994/10/04  19:31:00  billh
 * Changed fsqrt to sqrtf.
 *
 * Revision 1.2  1994/09/28  17:05:05  billh
 * Added routines to get current distance to origin and screen height
 *
 * Revision 1.1  94/08/24  03:10:37  billh
 * Initial revision
 * 
 ***************************************************************************/
