/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: DispCmds.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.16 $	$Date: 1995/05/11 22:37:57 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * DispCmds - different display commands which take data and put it in
 *	a storage space provided by a given Displayable object.
 *
 * Notes:
 *	1. All coordinates are stored as 3 points (x,y,z), even if meant
 * for a 2D object.  The 3rd coord for 2D objects will be ignored.
 ***************************************************************************/
#ifndef DISPCMDS_H
#define DISPCMDS_H

#include <string.h>
#include "Displayable.h"
#include "Matrix4.h"
#include "ColorList.h"

// enum with all the possible commands
// draw commands with a _I suffix use indices into a DATA block, not the
// coordinates themselves
enum { DCLEAR, DPUSH, DPOP, DLOAD, DMULT, DTRANS, DROT, DSCALE, 
	DPOINT, DLINE, DCYLINDER, DSPHERE, DCONE,
	DPOINT_I, DLINE_I, DCYLINDER_I, DSPHERE_I, DCONE_I,
	DTRIANGLE, DSQUARE, DTRIANGLE_I, DSQUARE_I,
	DCOLORINDEX, DCOLORRGB, DCOLORDEF, DMATERIALS,
	DTEXT, DTEXTPOS, DTEXTPOS_I,
	DLIGHTDEF, DLIGHTONOFF,
	DSPHERERES, DSPHERETYPE, DLINEWIDTH, DLINESTYLE,
	DMOREDATA, DNEWDATA, DLINKLIST,
	DPICKPOINT, DPICKPOINT_I, DPICKLINE, DPICKLINE_I, DPICKBOX, DPICKBOX_I,
	DLASTCOMMAND };

// enum with different sphere and line types
enum { SOLIDSPHERE, POINTSPHERE };
enum { SOLIDLINE, DASHEDLINE };


class DispCmd {

protected:
  // the code and size for this Display Command
  int cmdCode, cmdSize;

  // data to put in storage; contains cmdSize byes
  void *Data;

public:
  // location in Displayable array where data was stored
  // public so others can change it if necessary.
  void *dataLoc;

public:    
  DispCmd(int code, int size);
  
  // NOTE: This is NOT virtual, since no derived classes need to have any
  // form of a destructor.  If one DOES have to have a specialized destructor,
  // this will need to be made virtual in order for it to be executed properly.
  ~DispCmd(void);

  // do the action
  void put(Displayable *dobj);

  // do the action AGAIN, if we've done it before; if not, same as put
  void reput(Displayable *dobj);
};

///////////////////////  DispCmd derived classes  

//*************************************************************

// create a block of floating-point data values to be used in drawing
// This version adds a new block of data to the current data blocks
class DispCmdData : public DispCmd {
public:
#ifndef VMDCAVE
   // pointer to float, used to store the data pointer
 float *dptr;
#endif

  // DOES NOT HAVE AN EMPTY CONSTRUCTOR
  DispCmdData(int n, float *d);

  // give new data, and just 'put' the command in again
  void putdata(float *d, Displayable *dobj);

  // put in new data, and reput the command
  void reputdata(float *d, Displayable *dobj);
};

//*************************************************************

// create a block of floating-point data values to be used in drawing
// This version make the given data block cancel out the usage of any
// previously current data blocks
class DispCmdNewData : public DispCmd {
public:
  // DOES NOT HAVE AN EMPTY CONSTRUCTOR
  DispCmdNewData(int n, float *d);
  
  // pointer to float, used to store the data pointer
#ifndef VMDCAVE
   // pointer to float, used to store the data pointer
 float *dptr;
#endif

  // give new data, and just 'put' the command in again
  void putdata(float *d, Displayable *dobj);

  // put in new data, and reput the command
  void reputdata(float *d, Displayable *dobj);
};

//*************************************************************

// command to clear the display
class DispCmdClear : public DispCmd {
public:
  DispCmdClear(void);
};

//*************************************************************

// command to push the display stacks
class DispCmdPush : public DispCmd {
public:
  DispCmdPush(void);
};

//*************************************************************

// command to pop the display stacks
class DispCmdPop : public DispCmd {
public:
  DispCmdPop(void);
};

//*************************************************************

// command to replace the transformation stack with the given matrix
class DispCmdLoad : public DispCmd {

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdLoad(void);

  DispCmdLoad(Matrix4 *newmat) ;

  // put in new data, and put the command
  void putdata(Matrix4 *newmat, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(Matrix4 *newmat, Displayable *dobj) ;
};

//*************************************************************

// command to multiply the transformation stack with the given matrix
class DispCmdMult : public DispCmd {

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdMult(void) ;

  DispCmdMult(Matrix4 *newmat) ;

  // put in new data, and put the command
  void putdata(Matrix4 *newmat, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(Matrix4 *newmat, Displayable *dobj);
};

//*************************************************************

// command to do a translation
class DispCmdTrans : public DispCmd {
protected:
  float tr[3];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdTrans(void);

  DispCmdTrans(float x, float y, float z);

  // put in new data, and put the command
  void putdata(float x, float y, float z, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(float x, float y, float z, Displayable *dobj) ;
};

//*************************************************************

// command to do a scaling
class DispCmdScale : public DispCmd {
protected:
  float tr[3];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdScale(void);

  DispCmdScale(float x, float y, float z) ;
  DispCmdScale(float s) ;

  // put in new data, and put the command
  void putdata(float x, float y, float z, Displayable *dobj);

  // put in new data, and reput the command
  void reputdata(float x, float y, float z, Displayable *dobj) ;
};

//*************************************************************

// command to do a rotation about a specific axis
class DispCmdRot : public DispCmd {
protected:
  float tr[2];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdRot(void) ;

  DispCmdRot(float t, char a);

  // put in new data, and put the command
  void putdata(float t, char a, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(float t, char a, Displayable *dobj);
};

//*************************************************************

// plot a point at the given position
class DispCmdPoint : public DispCmd {
protected:
  float pos[3];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdPoint(void) ;

  DispCmdPoint(float *newpos) ;

  // put in new data, and put the command
  void putdata(float *newpos, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(float *newpos, Displayable *dobj) ;
};


// plot a point at the given position, using indices
class DispCmdPointIndex : public DispCmd {
protected:
  int pos;

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdPointIndex(void) ;

  DispCmdPointIndex(int newpos) ;

  // put in new data, and put the command
  void putdata(int newpos, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(int newpos, Displayable *dobj) ;
};

//*************************************************************

// plot a sphere of specified radius at the given position
class DispCmdSphere : public DispCmd {
protected:
  float pos[4];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdSphere(void) ;

  DispCmdSphere(float *newpos, float rad) ;

  // put in new data, and put the command
  void putdata(float *newpos, float rad, Displayable *dobj);

  // put in new data, and reput the command
  void reputdata(float *newpos, float rad, Displayable *dobj) ;
};


// plot a sphere of specified radius at the given position, using indices
class DispCmdSphereIndex : public DispCmd {
protected:
  float pos[2];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdSphereIndex(void) ;

  DispCmdSphereIndex(int newpos, float rad);

  // put in new data, and put the command
  void putdata(int newpos, float rad, Displayable *dobj);

  // put in new data, and reput the command
  void reputdata(int newpos, float rad, Displayable *dobj) ;
};

//*************************************************************

// plot a line at the given position
class DispCmdLine : public DispCmd {
protected:
  float pos[6];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdLine(void);

  DispCmdLine(float *pos1, float *pos2) ;

  // put in new data, and put the command
  void putdata(float *pos1, float *pos2, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(float *pos1, float *pos2, Displayable *dobj);
};


// plot a line at the given position, using indices
class DispCmdLineIndex : public DispCmd {
protected:
  int pos[2];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdLineIndex(void) ;

  DispCmdLineIndex(int pos1, int pos2) ;

  // put in new data, and put the command
  void putdata(int pos1, int pos2, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(int pos1, int pos2, Displayable *dobj) ;
};

//*************************************************************

// draw a triangle, given the three points (computes the normals
//  from the cross product) and all normals are the same
//   -or-
// draw the triangle, given the three points and the three normals
class DispCmdTriangle : public DispCmd {
protected:
  float pos[24];

  // calculates the normal from the three vectors
  void do_calc(float *pos1, float *pos2, float *pos3) ;
  // given the vectors, set the elements of the array
  void set_array(float *pos1, float *pos2, float *pos3,
		  float *norm1, float *norm2, float *norm3);
public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdTriangle(void) ;

  DispCmdTriangle(float *pos1, float *pos2, float *pos3);
  DispCmdTriangle(float *pos1, float *pos2, float *pos3,
		  float *norm1, float *norm2, float *norm3);

  // put in new data, and put the command
  void putdata(float *pos1, float *pos2, float *pos3, Displayable *dobj);
  void putdata(float *pos1, float *pos2, float *pos3,
	       float *norm1, float *norm2, float *norm3, Displayable *dobj);
  
  // put in new data, and reput the command
  void reputdata(float *pos1, float *pos2, float *pos3, Displayable *dobj) ;
  void reputdata(float *pos1, float *pos2, float *pos3,
	       float *norm1, float *norm2, float *norm3, Displayable *dobj);
};


// draw a triangle, given the three points and normal, using indices
class DispCmdTriangleIndex : public DispCmd {
protected:
  int pos[4];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdTriangleIndex(void) ;

  DispCmdTriangleIndex(int norml, int pos1, int pos2, int pos3);
  // put in new data, and put the command
  void putdata(int norml, int pos1, int pos2, int pos3, Displayable *dobj);

  // put in new data, and reput the command
  void reputdata(int norml, int pos1, int pos2, int pos3, Displayable *dobj);
};

//*************************************************************

// draw a square, given 3 of four points
class DispCmdSquare : public DispCmd {
protected:
  float pos[15];
  
  void do_calc(float *pos1, float *pos2, float *pos3) ;
  
public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdSquare(void);

  DispCmdSquare(float *pos1, float *pos2, float *pos3);

  // put in new data, and put the command
  void putdata(float *pos1, float *pos2, float *pos3, Displayable *dobj);

  // put in new data, and reput the command
  void reputdata(float *pos1, float *pos2, float *pos3, Displayable *dobj);
};


// draw a triangle, given four points and normal, using indices
class DispCmdSquareIndex : public DispCmd {
protected:
  int pos[5];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdSquareIndex(void) ;

  DispCmdSquareIndex(int norml, int pos1, int pos2, int pos3, int pos4);

  // put in new data, and put the command
  void putdata(int norml, int pos1, int pos2, int pos3, int pos4, 
  	Displayable *dobj);

  // put in new data, and reput the command
  void reputdata(int norml, int pos1, int pos2, int pos3, int pos4,
  	Displayable *dobj);
};

//*************************************************************
// plot a cylinder at the given position
class DispCmdCylinder : public DispCmd {
protected:
#ifdef USE_SLOW_CYLINDERS
  float pos[8];
#else
  float *pos;
  int lastres;
#endif
  void make_cylinder_info(float *pos1, float *pos2, float rad, int res);

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdCylinder(void);

  DispCmdCylinder(float *pos1, float *pos2, float rad, int res) ;

  // put in new data, and put the command
  void putdata(float *pos1,float *pos2,float rad,int res,Displayable *dobj);

  // put in new data, and reput the command
  void reputdata(float *pos1,float *pos2,float rad,int res,Displayable *dobj) ;
};


// plot a cylinder at the given position, using indices
class DispCmdCylinderIndex : public DispCmd {
protected:
  float pos[4];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdCylinderIndex(void);

  DispCmdCylinderIndex(int pos1, int pos2, float rad, int res) ;

  // put in new data, and put the command
  void putdata(int pos1, int pos2, float rad, int res, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(int pos1, int pos2, float rad, int res, Displayable *dobj);
};

//*************************************************************

// plot a cone at the given position
class DispCmdCone : public DispCmd {
protected:
  float pos[8];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdCone(void) ;

  DispCmdCone(float *pos1, float *pos2, float rad, int res);

  // put in new data, and put the command
  void putdata(float *pos1,float *pos2,float rad,int res,Displayable *dobj);

  // put in new data, and reput the command
  void reputdata(float *pos1,float *pos2,float rad,int res,Displayable *dobj) ;
};

// plot a cone at the given position, using indices
class DispCmdConeIndex : public DispCmd {
protected:
  float pos[4];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdConeIndex(void) ;

  DispCmdConeIndex(int pos1, int pos2, float rad, int res);

  // put in new data, and put the command
  void putdata(int pos1, int pos2, float rad, int res, Displayable *dobj);

  // put in new data, and reput the command
  void reputdata(int pos1, int pos2, float rad, int res, Displayable *dobj);
};

//*************************************************************

// set the current color to the given one in a list; this specifies the index
class DispCmdColorIndex : public DispCmd {
protected:
  int color;

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdColorIndex(void) ;

  DispCmdColorIndex(int newcol) ;

  // put in new data, and put the command
  void putdata(int newcol, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(int newcol, Displayable *dobj) ;
};

//*************************************************************

// set the current color to the given rgb value
class DispCmdColorRGB : public DispCmd {
protected:
  float color[3];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdColorRGB(void) ;

  DispCmdColorRGB(float *col) ;

  // put in new data, and put the command
  void putdata(float *col, Displayable *dobj) ;
  
  // put in new data, and reput the command
  void reputdata(float *col, Displayable *dobj) ;
};

//*************************************************************

// define values for the Nth color
class DispCmdColorDef : public DispCmd {
protected:
  float data[COLOR_ITEMS + 1];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdColorDef(void) ;

  DispCmdColorDef(int n, float *cdat);

  // put in new data, and put the command
  void putdata(int n, float *cdat, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(int n, float *cdat, Displayable *dobj) ;
};

//*************************************************************

// command to turn on or off material characteristics display
class DispCmdMaterials : public DispCmd {
protected:
  int onoff;
  
public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdMaterials(void) ;

  DispCmdMaterials(int on) ;

  // put in new data, and put the command
  void putdata(int on, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(int on, Displayable *dobj) ;
};

//*************************************************************

// display text at the current text coordinates
// Note: This does NOT make a copy of the string, so make sure the text stays
// around until the 'put' routine is called!!!
class DispCmdText : public DispCmd {

public:
  // THIS OBJECT DOES NOT HAVE AN EMPTY DESTRUCTOR

  DispCmdText(char *newtxt) ;

  // put in new data, and put the command
  // be careful, the text must be the same length as before!
  void putdata(char *newtxt, Displayable *dobj) ;

  // put in new data, and reput the command
  // be careful, the text must be the same length as before!
  void reputdata(char *newtxt, Displayable *dobj) ;
};

//*************************************************************

// move the current text position to the given position
class DispCmdTextPos : public DispCmd {
protected:
  float pos[3];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdTextPos(void) ;

  DispCmdTextPos(float *pos1) ;

  // put in new data, and put the command
  void putdata(float *pos1, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(float *pos1, Displayable *dobj) ;
};


// move the current text position to the given position, using indices
class DispCmdTextPosIndex : public DispCmd {
protected:
  int pos;

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdTextPosIndex(void) ;

  DispCmdTextPosIndex(int pos1) ;

  // put in new data, and put the command
  void putdata(int pos1, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(int pos1, Displayable *dobj) ;
};

//*************************************************************

// define a light ... need the color and position of the light
class DispCmdLightDef : public DispCmd {
protected:
  float lightdata[7];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdLightDef(void) ;

  DispCmdLightDef(int num, float *color, float *pos) ;

  // put in new data, and put the command
  void putdata(int num, float *color, float *pos, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(int num, float *color, float *pos, Displayable *dobj) ;
};

//*************************************************************

// toggle a light on or off
class DispCmdLightOnOff : public DispCmd {
protected:
  int lightdata[2];
  
public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdLightOnOff(void) ;

  DispCmdLightOnOff(int num, int onoff) ;

  // put in new data, and put the command
  void putdata(int num, int onoff, Displayable *dobj) ;
  
  // put in new data, and reput the command
  void reputdata(int num, int onoff, Displayable *dobj) ;
};

//*************************************************************

// set the current sphere resolution to the given one
class DispCmdSphereRes : public DispCmd {
protected:
  int res;

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdSphereRes(void) ;

  DispCmdSphereRes(int newres) ;

  // put in new data, and put the command
  void putdata(int newres, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(int newres, Displayable *dobj) ;
};

//*************************************************************

// set the current sphere type to the given one
class DispCmdSphereType : public DispCmd {
protected:
  int type;

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdSphereType(void);

  DispCmdSphereType(int newtype) ;

  // put in new data, and put the command
  void putdata(int newtype, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(int newtype, Displayable *dobj) ;
};

//*************************************************************

// set the current line type to the given one
class DispCmdLineType : public DispCmd {
protected:
  int type;

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdLineType(void) ;
  DispCmdLineType(int newtype) ;

  // put in new data, and put the command
  void putdata(int newtype, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(int newtype, Displayable *dobj) ;
};

//*************************************************************

// set the current line width to the given one
class DispCmdLineWidth : public DispCmd {
protected:
  int width;

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdLineWidth(void) ;

  DispCmdLineWidth(int newwidth) ;

  // put in new data, and put the command
  void putdata(int newwidth, Displayable *dobj) ;

  // put in new data, and reput the command
  void reputdata(int newwidth, Displayable *dobj) ;
};

//*************************************************************

// indicate a point which may be picked, and the 'tag' which is associated
// with this point
class DispCmdPickPoint : public DispCmd {
protected:
  float postag[4];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdPickPoint(void) ;

  DispCmdPickPoint(float *, int) ;

  // put in new data, and put the command
  void putdata(float *, int, Displayable *) ;

  // put in new data, and reput the command
  void reputdata(float *, int, Displayable *) ;
};

//*************************************************************

// indicate a point which may be picked, and the 'tag' which is associated
// with this point.  This uses an indexed position instead of a specific one.
class DispCmdPickPointIndex : public DispCmd {
protected:
  int postag[2];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdPickPointIndex(void) ;

  DispCmdPickPointIndex(int, int) ;

  // put in new data, and put the command
  void putdata(int, int, Displayable *) ;

  // put in new data, and reput the command
  void reputdata(int, int, Displayable *) ;
};

//*************************************************************

// indicate a line which may be picked, and the 'tag' which is associated
// with this point
class DispCmdPickLine : public DispCmd {
protected:
  float postag[7];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdPickLine(void) ;

  DispCmdPickLine(float *, float *, int) ;

  // put in new data, and put the command
  void putdata(float *, float *, int, Displayable *) ;

  // put in new data, and reput the command
  void reputdata(float *, float *, int, Displayable *) ;
};

//*************************************************************

// indicate a line which may be picked, and the 'tag' which is associated
// with this point.  This uses an indexed position instead of a specific one.
class DispCmdPickLineIndex : public DispCmd {
protected:
  int postag[3];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdPickLineIndex(void) ;

  DispCmdPickLineIndex(int, int, int) ;

  // put in new data, and put the command
  void putdata(int, int, int, Displayable *) ;

  // put in new data, and reput the command
  void reputdata(int, int, int, Displayable *) ;
};

//*************************************************************

// indicate a box which may be picked, and the 'tag' which is associated
// with this point.
// This box is specified by giving two opposite corners.
class DispCmdPickBox : public DispCmd {
protected:
  float postag[7];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdPickBox(void) ;

  DispCmdPickBox(float *, float *, int) ;

  // put in new data, and put the command
  void putdata(float *, float *, int, Displayable *) ;

  // put in new data, and reput the command
  void reputdata(float *, float *, int, Displayable *) ;
};

//*************************************************************

// indicate a box which may be picked, and the 'tag' which is associated
// with this point.  This uses an indexed position instead of a specific one.
// This box is specified by giving two opposite corners.
class DispCmdPickBoxIndex : public DispCmd {
protected:
  int postag[3];

public:
  // empty constructor ... must be used with 'putdata' or 'reputdata'
  DispCmdPickBoxIndex(void) ;

  DispCmdPickBoxIndex(int, int, int) ;

  // put in new data, and put the command
  void putdata(int, int, int, Displayable *) ;

  // put in new data, and reput the command
  void reputdata(int, int, int, Displayable *) ;
};

#endif

/* REVISION HISTORY:********************************************************
 *
 * $Log: DispCmds.h,v $
 * Revision 1.16  1995/05/11  22:37:57  billh
 * Moved log messages to the end of the file.
 *
 * Revision 1.15  95/04/11  09:46:51  dalke
 * Changed triangle so the normal for each vertex an be defined
 * 
 * Revision 1.14  1995/04/04  21:03:19  billh
 * DispCmd base class destructor no longer virtual.
 *
 * Revision 1.13  95/03/24  18:48:53  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.12  1995/03/10  07:49:33  dalke
 * added fast cylinders by precomputing in the DispCmd
 * This makes things about 2.5 times faster for pure cylinder
 * drawing.
 *
 * Revision 1.11  1995/02/22  04:02:11  billh
 * Now supports picking objects with a 2D or 3D pointer.
 * Added DispCmdPick* drawing commands, which indicate points that may be
 * selected with a pointer, and which contain an associated tag.
 *
 * Revision 1.10  95/02/12  07:14:51  dalke
 * Added a "DLINKLIST" so that the display list can grow as needed
 * when more graphics space is needed.
 * 
 * Revision 1.9  1994/10/05  04:37:15  billh
 * Took out double backslash from text, even in comments.
 *
 * Revision 1.8  1994/10/02  23:40:28  dalke
 * moved the code to DispCmds.C
 *
 * Revision 1.7  1994/09/28  22:13:31  billh
 * No change.
 *
 * Revision 1.6  1994/09/26  18:53:28  dalke
 * Fixed text bug
 *
 * Revision 1.5  1994/09/24  20:08:09  billh
 * empty constructors now all initialize 'Data' properly.
 *
 * Revision 1.4  94/09/23  00:43:52  billh
 * Added new indexed-position commands, and added the routine 'putdata' and
 * 'reputdata' to every display command.
 * 
 * Revision 1.3  94/09/03  11:04:54  dalke
 * Precomputed triangle norm, and added square
 * 
 * Revision 1.2  1994/08/30  07:18:09  dalke
 * Added a triangle command
 *
 * Revision 1.1  1994/08/24  03:10:37  billh
 * Initial revision
 *
 ***************************************************************************/
