/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CommandQueue.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.8 $	$Date: 1995/05/11 22:12:43 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * This stores all the Commands to be run in a queue.  The idea is that
 * the various events add Commands to the command queue then they
 * are read off the queue and the UIs are notified.
 *
 * Commands may be logged to a file, if desired.
 *
 ***************************************************************************/
#ifndef COMMANDQUEUE_H
#define COMMANDQUEUE_H

#include <stdio.h>
#include <string.h>
#include "ResizeArray.h"
class Command;
class UIList;


/* NOTES:
 *	1) To add new commands to queue, use routine 'append(Command *)',
 *		inherited since this is a ResizeArray<> object.
 *	2) To do something, use 'execute' routine, with a given UIList obj.
 *		This will execute the top command and inform all the UIs in
 *		the UIList.
 *	3) 'execute_all' will do all the commands until the queue is empty.
 */

class CommandQueue {

  private:
    // the command list itself
    ResizeArray<Command *> cmdlist;

    // are we logging commands to a file?  If so, which one?
    int loggingCmds;
    FILE *logfile;

    // UIList object to use
    UIList *uil;

    // execute the given command.  Return success of command.
    int do_execute(Command *);
   
    // delete the current element in the linked list.  Needed since we
    // must delete the Command object stored as well as the list element.
    void delete_current(void);

  public:
    // constructor/destructor
    CommandQueue(UIList *);
    ~CommandQueue(void);

    // check/change which UIList object to use
    UIList *uilist(void) { return uil; }
    void use_uilist(UIList *u) { uil = u; }

    //
    // log file routines
    //
    
    // check if logging
    int logging(void) { return loggingCmds; }
    
    // turn on logging ... open file, and write header comments
    void log_on(char *);
    
    // turn off logging ... close the file
    void log_off(void);

    //
    // command processing routines
    //
    
    // return number of commands yet to execute
    int num(void) { return cmdlist.num(); }

    // add a new command to the list ... always adds to queue, does not
    // execute.  Return whether it could be added to queue.
    int append(Command *);

    // add a new command to the list ...if we have a UIList already, run it,
    // otherwise queue it.  Return success if run, or (-1) if queued.
    int runcommand(Command *);

    // execute the first command in the queue ... return success of command or
    // FALSE if no commands available
    int execute(UIList *UIL = NULL);

    // execute ALL the commands in the queue
    void execute_all(UIList *IUL = NULL);
};

#endif

/* REVISION HISTORY:********************************************************
 *
 * $Log: CommandQueue.h,v $
 * Revision 1.8  1995/05/11  22:12:43  billh
 * Added new virtual function 'create_text', which is used only when the
 * commands are being logged to a file.  This eliminates alot of extra
 * work when text equivalents of commands are not needed (i.e. when not
 * logging commands).
 *
 * Revision 1.7  95/03/28  03:39:04  billh
 * Moved routines to .C file; added 'runcommand' routine to allow a command
 * to be executed immediately instead of always going on the queue.
 * 
 * Revision 1.6  95/03/24  18:48:28  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.5  1995/01/08  22:08:03  billh
 * Uses config.h instead of version.h
 *
 * Revision 1.4  1994/10/05  06:58:16  billh
 * Converted to use a ResizeArray instead of a DLinkList in the command queue
 * and ui list.
 *
 * Revision 1.3  1994/10/04  20:29:22  billh
 * Changes to get to compile on HP's ... not there yet,though.
 *
 * Revision 1.2  1994/09/12  20:52:37  billh
 * Made some debugging messages L3 instead of L2 (to reduce L2 verbosity).
 *
 * Revision 1.1  94/08/24  03:10:37  billh
 * Initial revision
 *  
 ***************************************************************************/
