/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CommandQueue.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.2 $	$Date: 1995/05/11 22:12:31 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * This stores all the Commands to be run in a queue.  The idea is that
 * the various events add Commands to the command queue then they
 * are read off the queue and the UIs are notified.
 *
 * Commands may be logged to a file, if desired.
 *
 ***************************************************************************/

#include "CommandQueue.h"
#include "Command.h"
#include "UIList.h"
#include "utilities.h"
#include "config.h"


///////////////////////////  constructor
CommandQueue::CommandQueue(UIList *u) : cmdlist(64) {
  uil = u;
  loggingCmds = FALSE;
  logfile = NULL;
}
    

///////////////////////////  destructor
// we must remove all commands, and delete them as well.
// if logging, must close file
CommandQueue::~CommandQueue(void) {
  while(cmdlist.num() > 0)
    delete_current();
  if(logging())
    log_off();
}


////////////////////////////  private routines  ////////////////////////

// execute the given command.  Return success of command
int CommandQueue::do_execute(Command *cmd) {

  MSGDEBUG(3,"CQ: Executing '" << *cmd << "' ..." << sendmsg);
	
  // if logging, write to log file
  if(logging() && cmd->has_text())
    fprintf(logfile,"%s\n", cmd->text());

  // do the command ...
  int retval = cmd->execute();
  
  // ... and report to UIList the action has been done
  if(uil)
    uil->act_on_command_UI(cmd->gettype(), cmd, retval); 

  // report the success of the commmand
  return retval;
}


// delete the current element in the linked list.  Needed since we
// must delete the Command object stored as well as the list element.
void CommandQueue::delete_current(void) {
  if(cmdlist.num() > 0) {
    delete cmdlist[0];	// delete the Command object store
    cmdlist.remove(0);
  }
}


////////////////////////////  public routines  ////////////////////////

// turn on logging ... open file, and write header comments
void CommandQueue::log_on(char *fname) {
  if(!logging()) {
    if((logfile = fopen(fname, "w")) != NULL) {
      msgInfo << "Logging commands to '" << fname << "'." << sendmsg;
      loggingCmds = TRUE;
      fprintf(logfile,"# %s\n", VERSION_MSG);
      fprintf(logfile,"# Log file '%s', created by user %s\n", fname,
		  username());
    } else {
      msgErr << "Cannot open log file '" << fname << "' for writing."
	     << sendmsg;
    }
  } else {
    msgErr << "Cannot open log file '" << fname << "' ... already "
	   << "logging commands to a file." << sendmsg;
  }
}


// turn off logging ... close the file
void CommandQueue::log_off(void) {
  if(logging()) {
    fprintf(logfile,"# %s\n", VERSION_MSG);
    fprintf(logfile,"# end of log file.\n");
    fclose(logfile);
    loggingCmds = FALSE;
    msgInfo << "Log file closed." << sendmsg;
  } else {
    msgErr << "Not currently logging commands." << sendmsg;
  }
}


// add a new command to the list ... if we have a UIList already, execute it,
// otherwise queue it.  Return success if run, or (-1) if queued.
int CommandQueue::runcommand(Command *cmd) {
  register int retval = (-1);

  if(!cmd)
    return FALSE;

  if(uil) {
    // execute the command, and then delete it
    retval = do_execute(cmd);
    delete cmd;
  } else {
    cmdlist.append(cmd);
  }

  return retval;
}


// add a new command to the list ... always adds to queue, does not
// execute.  Return whether it could be added to queue.
int CommandQueue::append(Command *cmd) {

  if(cmd)
    cmdlist.append(cmd);
  else
    return FALSE;

  // if here, things went OK
  return TRUE;
}


// execute the first command in the queue ... return success of command, or
// FALSE if no commands available
int CommandQueue::execute(UIList *UIL) {
  register int retval = FALSE;
  UIList *olduil = uil;

  if(cmdlist.num() > 0) {
    // execute the command, and then delete it
    if(UIL)
      uil = UIL;			// set UIList to use if necessary
    retval = do_execute(cmdlist[0]);
    delete_current();
    uil = olduil;			// restore proper UIList
  }
  
  return retval;
}


// execute ALL the commands in the queue
void CommandQueue::execute_all(UIList *UIL) {
  while (cmdlist.num() > 0)
    execute(UIL);
}


/* REVISION HISTORY:********************************************************
 *
 * $Log: CommandQueue.C,v $
 * Revision 1.2  1995/05/11  22:12:31  billh
 * Added new virtual function 'create_text', which is used only when the
 * commands are being logged to a file.  This eliminates alot of extra
 * work when text equivalents of commands are not needed (i.e. when not
 * logging commands).
 *
 * Revision 1.1  95/03/28  03:39:58  billh
 * Initial revision
 * 
 * Revision 1.6  95/03/24  18:48:28  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.5  1995/01/08  22:08:03  billh
 * Uses config.h instead of version.h
 *
 * Revision 1.4  1994/10/05  06:58:16  billh
 * Converted to use a ResizeArray instead of a DLinkList in the command queue
 * and ui list.
 *
 * Revision 1.3  1994/10/04  20:29:22  billh
 * Changes to get to compile on HP's ... not there yet,though.
 *
 * Revision 1.2  1994/09/12  20:52:37  billh
 * Made some debugging messages L3 instead of L2 (to reduce L2 verbosity).
 *
 * Revision 1.1  94/08/24  03:10:37  billh
 * Initial revision
 *  
 ***************************************************************************/
