/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CmdTracker.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 1995/05/11 22:06:22 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Code for the various tracker commands.
 *
 ***************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "CmdTracker.h"
#include "TrackerList.h"
#include "Tracker.h"
#include "UIVR.h"
#include "CommandQueue.h"
#include "Global.h"
#include "utilities.h"

////////////////////////////////////////////////////////////////////
///////////////////////  text processors
////////////////////////////////////////////////////////////////////

// text callback routine for 'tracker'; return TRUE if an error occurs.
int text_cmd_tracker(int argc, char **argv, CommandQueue *cmdQueue, int id) {

  // so the first word [0] is "tracker".  Here are the options
  // tracker list         -- list running trackers
  // tracker list avail   -- list available trackers
  // tracker start number [on remotemachine] -- start avail tracker "number"
  //                            possible remotely
  // tracker pause number [on remotemachine] -- pause tracker "number"
  // tracker location number sensor -- print the current location of 
  //                            a sensor on the tracker "number"

  if (argc < 2 || argc > 5)  // 2 and 5 are obviously out of bounds, now
    return TRUE;
    
  // "tracker list" and "tracker list avail"
  if (!strupncmp(argv[1], "list", CMDLEN)) {  // ############
    if (argc == 2) { // tracker list
      cmdQueue->append(new CmdTrackerList(id));
    } else if ( !strupncmp(argv[2], "avail", CMDLEN)) {
      cmdQueue->append(new CmdTrackerListAvail(id));
    } else
      return TRUE;

  } else if (!strupncmp(argv[1], "start", CMDLEN)) {
    // tracker start number [on remotemachine]  #################
    if (argc !=3 && !(argc == 5 && strupncmp(argv[3], "on", CMDLEN)))
      return TRUE;

    // send a start message to the appropriate tracker
    if (argc == 3)
      cmdQueue->append(new CmdTrackerStart(atoi(argv[2]), id));
    else
      cmdQueue->append(new CmdTrackerStart(atoi(argv[2]), argv[4], id));

  } else if (!strupncmp(argv[1], "location", CMDLEN)) {
    // tracker location number sensor  ######################
    if (argc != 4)
      return TRUE;

    cmdQueue->append(new CmdTrackerLoc(atoi(argv[2]), atoi(argv[3]), id));

  } else if (!strupncmp(argv[1], "pause", CMDLEN)) {
    // tracker pause 'tracker num'  ###########
    if (argc != 3)
      return TRUE;

    cmdQueue->append(new CmdTrackerPause(atoi(argv[2]), id));

  } else if (!strupncmp(argv[1], "unpause", CMDLEN)) {
    // tracker unpause 'tracker num'  ###########
    if (argc != 3)
      return TRUE;

    cmdQueue->append(new CmdTrackerUnpause(atoi(argv[2]), id));

  } else
    return TRUE;
    
  // if here, everything worked out ok
  return FALSE;
}

/////////////////////////////  utility functions  //////////////////////////

// prints an error message and returns FALSE when the tracker is
// outside the allowed range of running ones.  Otherwise, it returns TRUE
int valid_running_tracker_range(int tracker) {
  int maxtracker;
  maxtracker = (trackerList ? trackerList -> num() - 1 : (-1));
  if (tracker < 0 || tracker > maxtracker) {
    if (maxtracker <= 0)
      msgErr << "There are no trackers running." << sendmsg;
    else {
      msgErr << "Illegal tracker number '" << tracker << "'." << sendmsg;
      msgErr << "The running trackers number from 0 to " << maxtracker;
      msgErr << "." << sendmsg;
    }
    return FALSE;
  }
  return TRUE;  // all a-okay
}


// prints an error message and returns FALSE when the tracker is
// outside the allowed range of available ones.  Otherwise, it returns TRUE
int valid_avail_tracker_range(int tracker) {
  int maxtracker;
  maxtracker = (trackerList ? trackerList -> num_avail() - 1 : (-1));
  if (tracker < 0 || tracker > maxtracker) {
    if (maxtracker <= 0)
      msgErr << "There are no trackers available." << sendmsg;
    else {
      msgErr << "Illegal tracker number '" << tracker << "'." << sendmsg;
      msgErr << "The available trackers number from 0 to " << maxtracker;
      msgErr << "." << sendmsg;
    }
    return FALSE;
  }
  return TRUE;  // all a-okay
}


// prints an error message and returns FALSE when the sensor is
// outside the allowed range.  Otherwise, it returns TRUE
int valid_sensor_range(int tracker, int sensor) {
  int maxsensor;

  // first make sure the tracker is OK
  if(valid_running_tracker_range(tracker)) {
    maxsensor = (trackerList -> tracker(tracker)) -> numSensors() - 1;
    if (sensor < 0 || sensor > maxsensor) {
      if (maxsensor <= 0) {
        msgErr << "There are no sensors available for tracker " << tracker;
        msgErr << "." << sendmsg;
      } else {
        msgErr << "Illegal sensor number '" << sensor << "'." << sendmsg;
        msgErr << "The sensors number from 0 to " << maxsensor;
        msgErr << "." << sendmsg;
      }
      return FALSE;
    }
    return TRUE;  // all a-okay
  }
  return FALSE;
}


/////////////////////////// Start a tracker
CmdTrackerStart::CmdTrackerStart(int newavailnum, char *newhostname, int id)
	: Command(Command::TRACKER_START, id) {
  hostname = stringdup(newhostname);
  availnum = newavailnum;
}

CmdTrackerStart::CmdTrackerStart(int newavailnum, int id)
	: Command(Command::TRACKER_START, id) {
  hostname = stringdup("");
  availnum = newavailnum;
}

CmdTrackerStart::~CmdTrackerStart(void) {
  delete [] hostname;
}

void CmdTrackerStart::create_text(void) {
  *cmdText << "tracker start " << availnum;
  if(strlen(hostname) > 0)
    *cmdText << " on " << hostname;
  *cmdText << ends;
}

int CmdTrackerStart::do_execute(void) {
  if(valid_avail_tracker_range(availnum))
    return (trackerList -> start_avail(availnum, hostname)) == TRACKER_OK;
  else
    return FALSE;    
}


/////////////////////////// Pause a tracker, given its number on the list
CmdTrackerPause::CmdTrackerPause(int tmptrackernum, int fromUIid)
	: Command(Command::TRACKER_PAUSE, fromUIid) {
  trackernum = tmptrackernum;
}

void CmdTrackerPause::create_text(void) {
  *cmdText << "tracker pause " << trackernum << ends;
}

int CmdTrackerPause::do_execute(void) {
  if(valid_running_tracker_range(trackernum)) {
    trackerList -> tracker(trackernum) -> pause();
    return TRUE;
  } else {
    return FALSE;
  }
}


/////////////////////////// Pause a tracker, given its number on the list
CmdTrackerUnpause::CmdTrackerUnpause(int tmptrackernum, int fromUIid)
	: Command(Command::TRACKER_UNPAUSE, fromUIid) {
  trackernum = tmptrackernum;
}

void CmdTrackerUnpause::create_text(void) {
  *cmdText << "tracker unpause " << trackernum << ends;
}

int CmdTrackerUnpause::do_execute(void) {
  if(valid_running_tracker_range(trackernum)) {
    trackerList -> tracker(trackernum) -> unpause();
    return TRUE;
  } else {
    return FALSE;
  }
}


/////////////////////////// list all the available trackers
CmdTrackerListAvail::CmdTrackerListAvail(int fromUIid)
	: Command(Command::TRACKER_LIST_AVAIL, fromUIid) { }

void CmdTrackerListAvail::create_text(void) {
  *cmdText << "tracker list avail" << ends;
}

int CmdTrackerListAvail::do_execute(void) {
  int i = trackerList -> num_avail();
  if (i==0) {
    msgInfo << "  No trackers are available." << sendmsg;
  } else {
    msgInfo << "Available trackers:\n" << sendmsg;
    for (int j=0; j<i; j++)
      msgInfo << " " << j << "   " << trackerList->avail_name(j) << sendmsg;
  }
  return TRUE;
}


/////////////////////////// list all running trackers
CmdTrackerList::CmdTrackerList(int fromUIid)
	: Command(Command::TRACKER_LIST, fromUIid) { }

void CmdTrackerList::create_text(void) {
  *cmdText << "tracker list" << ends;
}

int CmdTrackerList::do_execute(void) {
  int i = trackerList -> num();
  if (i == 0 ) {
    msgInfo << "  No trackers are running.\n" << sendmsg;
  } else {
    msgInfo << "Running trackers:\n" << sendmsg;
    for (int j=0; j<i; j++) {
      msgInfo << " " << j;
      if (trackerList->tracker(j)->paused())
        msgInfo << " (paused) ";
      else
        msgInfo << "          ";
      msgInfo  << trackerList -> name(j) << '\n' << sendmsg;
    }
  }
  return TRUE;
}


////////////////////// print the current location of a tracker/ sensor pair
CmdTrackerLoc::CmdTrackerLoc(int tracker, int sensor, int fromUIid)
	: Command(Command::TRACKER_LOC, fromUIid) {
  trackernum = tracker;
  sensornum = sensor;
}

void CmdTrackerLoc::create_text(void) {
  *cmdText << "tracker location " << trackernum << " " << sensornum << ends;
}

// return the current location (x,y,z) of a tracker/sensor pair
int CmdTrackerLoc::do_execute(void) {
  float x,y,z;
  if(valid_sensor_range(trackernum, sensornum)) {
    (trackerList -> tracker(trackernum)) -> get_position(sensornum, x, y, z);
    msgInfo << "Tracker: " << trackernum << "  Sensor: " << sensornum;
    msgInfo << "   \nx= " << x << " y= " << y << " z= "<< z << sendmsg;
    return TRUE;
  }
  return FALSE;
}

/* REVISION HISTORY:********************************************************
 *
 * $Log: CmdTracker.C,v $
 * Revision 1.3  1995/05/11  22:06:22  billh
 * Added 'create_text' function to create text representation "on demand".
 * Added text callback functions which are added by UIText, and which are
 * called when a text command is entered and the first word is recognized.
 *
 * Revision 1.2  95/03/24  18:48:10  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.1  1994/09/15  07:04:49  dalke
 * Initial revision
 *
 ****************************************************************************/
