/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CmdTool.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.5 $	$Date: 1995/05/11 22:06:20 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Commands which control the tool 
 *
 ***************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "CmdTool.h"
#include "GrabTool.h"
#include "UIVR.h"
#include "CommandQueue.h"
#include "Global.h"
#include "utilities.h"

#define MIN_SCALE  0.01
#define MIN_LENGTH 0.05
#define MIN_DETAIL 1
#define MAX_SCALE  20
#define MAX_LENGTH 20
#define MAX_DETAIL 10
#define MAX_OFFSET 20

////////////////////////////////////////////////////////////////////
///////////////////////  text processors
////////////////////////////////////////////////////////////////////

// text callback routine for 'tool'; return TRUE if an error occurs.
int text_cmd_tool(int argc, char **argv, CommandQueue *cmdQueue, int id) {
   if (argc < 3)
     return TRUE;

   int toolid = atoi(argv[1]);

   if ((!strupncmp(argv[2],"info",CMDLEN)||!strupncmp(argv[2],"set",CMDLEN))){
     // tool number info |  print out the info
     // tool number set  |    about a tool
     if (argc != 3)
       return TRUE;
     cmdQueue->append(new CmdToolInfo(toolid, id));

   } else if (!strupncmp(argv[2], "push", CMDLEN)) {
     // tool number push newtool
     if (argc != 4)
       return TRUE;
     cmdQueue->append(new CmdToolPush(toolid, argv[3], id));

   } else if (!strupncmp(argv[2], "pop", CMDLEN)) {
     // tool number pop
     if (argc != 3)
       return TRUE;
     cmdQueue->append(new CmdToolPop(toolid, id));

   } else if (!strupncmp(argv[2], "replace", CMDLEN)) {
     // tool number replace newtool  -- a pop then a push
     if (argc != 4)
       return TRUE;
     cmdQueue->append(new CmdToolReplace(toolid, argv[3], id));

   } else if (!strupncmp(argv[2], "set", CMDLEN) ||
		!strupncmp(argv[2], "add", CMDLEN) ) {
     // tool number [{set,add} [size newsize] [length newlength]
     //               {set,add} [offset x y z]  [detail newdetail]]+
     int pos = 2;

     // for each of the inputs, toss a Command on the commandQueue
     while (pos < argc) {
       if (!strupncmp(argv[pos], "set",CMDLEN)) {
         if (!strupncmp(argv[pos+1], "scale",CMDLEN)) {
           cmdQueue->append(new CmdToolSetSize(toolid, 
                    atof(argv[pos+2]), 0.0, 0, id));
           pos += 3;
         } else if (!strupncmp(argv[pos+1], "length", CMDLEN)) {
           cmdQueue->append(new CmdToolSetSize(toolid,
                    0.0, atof(argv[pos+2]), 0, id));
           pos += 3;
         } else if (!strupncmp(argv[pos+1], "detail", CMDLEN)) {
           cmdQueue->append(new CmdToolSetSize(toolid,
                    0.0, 0.0, atoi(argv[pos+2]), id));
           pos += 3;
         } else if (!strupncmp(argv[pos+1], "offset", CMDLEN)) {
           if (pos + 5 <= argc) {
             cmdQueue->append(new CmdToolSetOffset(toolid,
                atof(argv[pos+2]),atof(argv[pos+3]),atof(argv[pos+4]),id));
             pos += 5;
           } else
             return TRUE;
         } else
           return TRUE;

       } else if (!strupncmp(argv[pos], "add",CMDLEN)) {
         if (!strupncmp(argv[pos+1], "scale",CMDLEN)) {
           cmdQueue->append(new CmdToolAddSize(toolid, 
                   atof(argv[pos+2]), 0.0, 0, id));
           pos += 3;
         } else if (!strupncmp(argv[pos+1], "length", CMDLEN)) {
           cmdQueue->append(new CmdToolAddSize(toolid,
                   0.0, atof(argv[pos+2]), 0, id));
           pos += 3;
         } else if (!strupncmp(argv[pos+1], "detail", CMDLEN)) {
           cmdQueue->append(new CmdToolAddSize(toolid,
                   0.0, 0.0, atoi(argv[pos+2]), id));
           pos += 3;
         } else if (!strupncmp(argv[pos+1], "offset", CMDLEN)) {
           if (pos + 5 <= argc) {
             cmdQueue->append(new CmdToolAddOffset(toolid,
                    atof(argv[pos+2]), atof(argv[pos+3]), atof(argv[pos+4]),
                    id ) );
             pos += 5;
           } else
             return TRUE;
         } else
           return TRUE;
       } else
         return TRUE;
     }

   } else
     return TRUE;

   // if here, everything worked out ok
   return FALSE;
}


/////////////////////////  utility routines  ///////////////////////////

// check the tool number to see if it is in range
// return FALSE and print an erro if it is not
static int check_tool_range(int toolid)
{
  int n = uiVR->tools.num();
  if (toolid < 0 || toolid >= n) {
     msgErr<<"The valid range of tools are: " << 0 << " to " << n-1 <<sendmsg;
     return FALSE;
  }
  return TRUE;
}


// create a new tool based on the given tool name ... if unrecognized,
// print an error and return NULL
static Tool *create_new_tool(char *toolname) {

  if (!strupncmp(toolname, "grabber", CMDLEN)) {
    return (new GrabTool("grabber", scene));

  }

  // no recognizable tool found ... return NULL
  msgErr << "Unknown Tool '" << toolname << "' specified." << sendmsg;
  return NULL;
}


//////////////////// return the base characteristics of a Tool
CmdToolInfo::CmdToolInfo(int which_tool, int UIid)
  : Command(Command::TOOL_INFO, UIid) {
  toolid = which_tool;
}

void CmdToolInfo::create_text(void) {
  *cmdText << "tool " << toolid << " info" << ends;
}

int CmdToolInfo::do_execute(void) {
  if (!check_tool_range(toolid))
    return FALSE;
  msgInfo << "Info for tool #" << toolid+1 << "\n";
  msgInfo << "  Scale: " << uiVR->tools[toolid]->scale() << "\n";
  msgInfo << "  Length: " << uiVR->tools[toolid]->length() << "\n";
  msgInfo << "  Detail: " << uiVR->tools[toolid]->detail() << "\n";
  float *offset = uiVR->tools[toolid]->offset();
  msgInfo << "  Offset is: " << offset[0] << "  " << offset[1] << "  " <<
                  offset[2] << ".\n" << sendmsg;
  return TRUE;
}


//////////////////// set the tool's base characteristics to a specific number
CmdToolSetSize::CmdToolSetSize(int which_tool,
                   float new_scale,     // overall scaling factor >0
                   float new_length,   // length of tool >0
                   int new_detail,    // detail ranges from 1 to 10 (best)
                   int UIid)
 : Command (Command::TOOL_SIZE, UIid) {
  toolid = which_tool;
  scale = new_scale;
  length = new_length;
  detail = new_detail;
}

void CmdToolSetSize::create_text(void) {
  *cmdText << "tool " << toolid << " set";
  if (scale > 0)
    *cmdText << " scale " << scale;
  if (length > 0)
    *cmdText << " length " << length;
  if (detail > 0)
    *cmdText << " detail " << detail;
  *cmdText << ends;
}

int CmdToolSetSize::do_execute(void) {
  if (!check_tool_range(toolid))
    return FALSE;
  ToolControl *tmptool = uiVR->tools[toolid];

  if (scale > 0) {
  tmptool->scale(scale);
  if (tmptool->scale() < MIN_SCALE)
    tmptool->scale(MIN_SCALE);
   else if (tmptool->scale() > MAX_SCALE)
    tmptool->scale(MAX_SCALE);
  }
  if (length > 0) {
  tmptool->length(length);
  if (tmptool->length() < MIN_LENGTH)
    tmptool->length(MIN_LENGTH);
   else if (tmptool->length() > MAX_LENGTH)
    tmptool->length(MAX_LENGTH);
  }
  if (detail > 0) {
  tmptool->detail(detail);
  if (tmptool->detail() < MIN_DETAIL)
    tmptool->detail(MIN_DETAIL);
   else if (tmptool->detail() > MAX_DETAIL)
    tmptool->detail(MAX_DETAIL);
  }
  return TRUE;
}


////////////////// alter the tool's base characteristics by a relative number
CmdToolAddSize::CmdToolAddSize(int which_tool,
                   float new_scale,     // overall scaling factor >0
                   float new_length,   // length of tool >0
                   int new_detail,    // detail ranges from 1 to 10 (best)
                   int UIid)
 : Command (Command::TOOL_SIZE, UIid) {
  toolid = which_tool;
  scale = new_scale;
  length = new_length;
  detail = new_detail;
}

void CmdToolAddSize::create_text(void) {
  *cmdText << "tool " << toolid << " add";
  if (scale > 0)
    *cmdText << " scale " << scale;
  if (length > 0)
    *cmdText << " length " << length;
  if (detail > 0)
    *cmdText << " detail " << detail;
  *cmdText << ends;
}

int CmdToolAddSize::do_execute(void) {
  if (!check_tool_range(toolid))
    return FALSE;
  ToolControl *tmptool = uiVR->tools[toolid];
  tmptool->scale(scale + tmptool->scale());
  if (tmptool->scale() < MIN_SCALE)
    tmptool->scale(MIN_SCALE);
   else if (tmptool->scale() > MAX_SCALE)
    tmptool->scale(MAX_SCALE);

  tmptool->length(length + tmptool->length());
  if (tmptool->length() < MIN_LENGTH)
    tmptool->length(MIN_LENGTH);
   else if (tmptool->length() > MAX_LENGTH)
    tmptool->length(MAX_LENGTH);

  tmptool->detail(detail + tmptool->detail());
  if (tmptool->detail() < MIN_DETAIL)
    tmptool->detail(MIN_DETAIL);
   else if (tmptool->detail() > MAX_DETAIL)
    tmptool->detail(MAX_DETAIL);

  return TRUE;
}


//////////////////// change the offset to some new position
CmdToolSetOffset::CmdToolSetOffset(int which_tool, float *newpos, int UIid)
 : Command(Command::TOOL_OFFSET, UIid) {
  toolid = which_tool;
  delta[0] = newpos[0];
  delta[1] = newpos[1];
  delta[2] = newpos[2];
}

CmdToolSetOffset::CmdToolSetOffset(int which_tool, float x, float y, float z,
   int UIid) : Command(Command::TOOL_OFFSET, UIid)
{
  toolid = which_tool;
  delta[0] = x;
  delta[1] = y;
  delta[2] = z;
}

void CmdToolSetOffset::create_text(void) {
  *cmdText << "tool " << toolid << " set offset " << delta[0] << " " <<
         delta[1] << " " << delta[2] << ends;
}

int CmdToolSetOffset::do_execute(void) {
  if (!check_tool_range(toolid))
    return FALSE;
  if (delta[0] >  MAX_OFFSET) delta[0] =  MAX_OFFSET;
  if (delta[0] < -MAX_OFFSET) delta[0] = -MAX_OFFSET;
  if (delta[1] >  MAX_OFFSET) delta[1] =  MAX_OFFSET;
  if (delta[1] < -MAX_OFFSET) delta[1] = -MAX_OFFSET;
  if (delta[2] >  MAX_OFFSET) delta[2] =  MAX_OFFSET;
  if (delta[2] < -MAX_OFFSET) delta[2] = -MAX_OFFSET;
  
  uiVR->tools[toolid]->offset(delta);
  return TRUE;
}


//////////////////// change the offset by some relative amount
CmdToolAddOffset::CmdToolAddOffset(int which_tool, float *newpos, int UIid)
 : Command(Command::TOOL_OFFSET, UIid) {
  toolid = which_tool;
  delta[0] = newpos[0];
  delta[1] = newpos[1];
  delta[2] = newpos[2];
}

CmdToolAddOffset::CmdToolAddOffset(int which_tool, float x, float y, float z,
   int UIid) : Command(Command::TOOL_OFFSET, UIid) {
  toolid = which_tool;
  delta[0] = x;
  delta[1] = y;
  delta[2] = z;
}

void CmdToolAddOffset::create_text(void) {
  *cmdText << "tool " << toolid << " add offset " << delta[0] << " " <<
         delta[1] << " " << delta[2] << ends;
}

int CmdToolAddOffset::do_execute(void) {
  if (!check_tool_range(toolid))
    return FALSE;
  add(delta, delta, uiVR->tools[toolid]->offset());
  if (delta[0] >  MAX_OFFSET) delta[0] =  MAX_OFFSET;
  if (delta[0] < -MAX_OFFSET) delta[0] = -MAX_OFFSET;
  if (delta[1] >  MAX_OFFSET) delta[1] =  MAX_OFFSET;
  if (delta[1] < -MAX_OFFSET) delta[1] = -MAX_OFFSET;
  if (delta[2] >  MAX_OFFSET) delta[2] =  MAX_OFFSET;
  if (delta[2] < -MAX_OFFSET) delta[2] = -MAX_OFFSET;
  
  uiVR->tools[toolid]->offset(delta);
  return TRUE;
}


////////////////////////// Change the current tool ///////////////////

//////////////////// put another tool on the ToolControl stack
CmdToolPush::CmdToolPush(int which_tool, char *newtool, int UIid )
   : Command(Command::TOOL_PUSH, UIid) {
  toolid = which_tool;
  tool = stringdup(newtool);
}

CmdToolPush::~CmdToolPush(void) {
  delete [] tool;
}

void CmdToolPush::create_text(void) {
  *cmdText << "tool " << toolid << " push " << tool << ends;
}

int  CmdToolPush::do_execute(void) {
  if (!check_tool_range(toolid))
    return FALSE;

  // check for a recognizable tool name
  Tool *ntool = create_new_tool(tool);
  if(ntool)
    uiVR->tools[toolid] -> push_tool(ntool);

  return (ntool != NULL);  
}


//////////////////// take a tool off the ToolControl stack
CmdToolPop::CmdToolPop(int which_tool, int UIid )
   : Command(Command::TOOL_POP, UIid) {
  toolid = which_tool;
}

void CmdToolPop::create_text(void) {
  *cmdText << "tool " << toolid << " pop" << ends;
}

int CmdToolPop::do_execute(void) {
  if (!check_tool_range(toolid))
    return FALSE;

  uiVR->tools[toolid] -> pop_tool();
  return TRUE;
}


//////////////////// replace the top tool with another
CmdToolReplace::CmdToolReplace(int which_tool, char *newtool, int UIid )
   : Command(Command::TOOL_PUSH, UIid) {
  toolid = which_tool;
  tool = stringdup(newtool);
}

CmdToolReplace::~CmdToolReplace(void) {
  delete [] tool;
}

void CmdToolReplace::create_text(void) {
  *cmdText << "tool " << toolid << " replace " << tool << ends;
}

int  CmdToolReplace::do_execute(void) {
  if (!check_tool_range(toolid))
    return FALSE;

  // check for a recognizable tool name
  Tool *ntool = create_new_tool(tool);
  if(ntool)
    uiVR->tools[toolid] -> replace_tool(ntool);

  return (ntool != NULL);  
}


/* REVISION HISTORY:********************************************************
 *
 * $Log: CmdTool.C,v $
 * Revision 1.5  1995/05/11  22:06:20  billh
 * Added 'create_text' function to create text representation "on demand".
 * Added text callback functions which are added by UIText, and which are
 * called when a text command is entered and the first word is recognized.
 *
 * Revision 1.4  95/03/24  18:48:07  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.3  1994/11/07  06:57:10  dalke
 * Got rid of "soft" stack -- I now change tools via a Command
 *
 ***************************************************************************/
