/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: CmdColor.C,v $
 *      $Author: billh $        $Locker:  $                $State: Exp $
 *      $Revision: 1.3 $      $Date: 1995/05/11 22:05:41 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Command objects for affecting molecules.
 *
 ***************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "CmdColor.h"
#include "ColorList.h"
#include "Scene.h"
#include "DisplayDevice.h"
#include "CommandQueue.h"
#include "Global.h"
#include "utilities.h"


// the following defines commands for the Cmdtypes:
// COLOR_NAME, COLOR_CHANGE, COLOR_SCALE_METHOD, COLOR_SCALE_MIDPOINT,
// COLOR_SCALE_MIN, COLOR_SCALE_MAX

////////////////////////////////////////////////////////////////////
///////////////////////  text processors
////////////////////////////////////////////////////////////////////

// text callback routine for 'color'; return TRUE if an error occurs.
int text_cmd_color(int argc, char **argv, CommandQueue *cmdQueue, int id) {

  if(argc < 2)
    return TRUE;

  if(!strupncmp(argv[1], "change", CMDLEN) && argc > 3) {
    int ctype;
    float r = 0.5, g = 0.5, b = 0.5;

    if(!strupncmp(argv[2], "alpha", CMDLEN))
      ctype = ALPHA_INDEX;
    else if(!strupncmp(argv[2], "shininess", CMDLEN))
      ctype = SHININESS_INDEX;
    else if(!strupncmp(argv[2], "rgb", CMDLEN))
      ctype = COLOR_INDEX;
    else if(!strupncmp(argv[2], "ambient", CMDLEN))
      ctype = AMBIENT_INDEX;
    else if(!strupncmp(argv[2], "diffuse", CMDLEN))
      ctype = DIFFUSE_INDEX;
    else if(!strupncmp(argv[2], "specular", CMDLEN))
      ctype = SPECULAR_INDEX;
    else
      return TRUE;

    if(argc == 4) {
      cmdQueue->append(new CmdColorChange(ctype, argv[3], id));
    } else {
      if(isdigit(*(argv[4])) || *(argv[4]) == '.') {
        r = atof(argv[4]);
        if(argc != 5 && argc != 7)
          return TRUE;
        else {
          g = atof(argv[5]);
          b = atof(argv[6]);
        }
        cmdQueue->append(new CmdColorChange(ctype, argv[3], r, g, b, id));
      } else {
        if(argc != 5)
          return TRUE;
        cmdQueue->append(new CmdColorChange(ctype, argv[3], argv[4], id));
      }
    }

  } else if(!strupncmp(argv[1], "scale", CMDLEN)) {
    if(argc == 4) {
      if(!strupncmp(argv[2], "method", CMDLEN)) {
        int i;
        for(i=0; i < ColorList::SCALE_TOTAL; i++) {
	  if(!strupncmp(argv[3], colorScaleMethod[i], CMDLEN)) {
	    cmdQueue->append(new CmdColorScaleMethod(i, id));
	    break;
	  }
	}
	if(i == ColorList::SCALE_TOTAL)
	  return TRUE;
      } else if(!strupncmp(argv[2], "midpoint", CMDLEN)) {
        cmdQueue->append(new CmdColorScaleMidpoint(atof(argv[3]), id));
      } else if(!strupncmp(argv[2], "min", CMDLEN)) {
        cmdQueue->append(new CmdColorScaleMin(atof(argv[3]), id));
      } else if(!strupncmp(argv[2], "min", CMDLEN)) {
        cmdQueue->append(new CmdColorScaleMax(atof(argv[3]), id));
      } else
        return TRUE;
    } else
      return TRUE;
    
  } else if(argc == 3 || argc == 4) {
    char *clr = (argc == 3 ? NULL : argv[3]);
    cmdQueue->append(new CmdColorName(argv[1], argv[2], clr, id));

  } else
    return TRUE;
    
  // if here, everything worked out ok
  return FALSE;
}



//////////////////  utility routines

// given a string which is either a color name or number, return regular
// color index.  If error, return < 0 and print message.
int find_color_index(char *cname) {
  int retval = (-1);

  if(colors)
    retval = colors->colorNames.typecode(cname);

  if(retval < 0)
    msgErr << "Unknown color name '" << cname << "'." << sendmsg;
  
  return retval;
}


//////////////////  commands

///////////////  change the color index for a specifed name in a specified
///////////////  category
int CmdColorName::do_execute(void) {
  int retval;
  if(retval = (colCatIndex >= 0 && colNameIndex >= 0)) {
    if(newColor >= 0) {
      // change color associated with name
      (colors->color_category(colCatIndex))->set_data(colNameIndex, newColor);

      // tell the displayables and the display a color has been changed
      scene->color_changed(colors, colCatIndex);
      display->color_changed(colors, colCatIndex);
    } else {
      int curclr = (colors->color_category(colCatIndex))->data(colNameIndex);
      msgInfo << "Color of " <<(colors->color_categories()).name(colCatIndex);
      msgInfo << " ";
      msgInfo << (colors->color_category(colCatIndex))->name(colNameIndex);
      msgInfo << " is " << colors->colorNames.name(curclr);
      msgInfo << " (" << curclr << ")" << sendmsg;
    }
  }
  
  return retval;
}

void CmdColorName::create_text(void) { }

// constructor: category name, item name, new color
CmdColorName::CmdColorName(char *cCat, char *cName, char *cCol, int newUIid)
	: Command(Command::COLOR_NAME, newUIid) {
  colNameIndex = newColor = (-1);
  if(colors) {
    colCatIndex = (colors->color_categories()).typecode(cCat);
    if(colCatIndex < 0) {
      msgErr << "Unknown color category '" << cCat << "'." << sendmsg;
    } else {
      colNameIndex = (colors->color_category(colCatIndex))->typecode(cName);
      if(colNameIndex < 0) {
         msgErr << "Unknown name '" << cName << "' for which to set color."
	 	<< sendmsg;
      } else {
        newColor = find_color_index(cCol);
      }
    }
  }

  // leave this in the constructor since it uses data not stored for later
  *cmdText << "color " << cCat << " " << cName;
  if(cCol)
    *cmdText << " " << cCol;
  *cmdText << ends;
}


///////////////  change the rgb settings for a specified color
int CmdColorChange::do_execute(void) {
  int retval;
  if(retval = (colIndex >= 0 && (useRGB || newColIndex >= 0))) {
    if(setDefault) {
      colors->reset_color(REGCOLOR(colIndex));
    } else {
      if(!useRGB) {
        // get status of 2nd color
        float *curcol = colors->color_data(REGCOLOR(newColIndex)) + ctype;
        newR = curcol[0];
        newG = curcol[1];
        newB = curcol[2];
      }
      if(ctype == ALPHA_INDEX)
        colors->change_alpha(REGCOLOR(colIndex), newR);
      else if(ctype == SHININESS_INDEX)
        colors->change_shininess(REGCOLOR(colIndex), newR);
      else if(ctype == COLOR_INDEX)
        colors->change_color(REGCOLOR(colIndex), newR, newG, newB);
      else if(ctype == AMBIENT_INDEX)
        colors->change_ambient(REGCOLOR(colIndex), newR, newG, newB);
      else if(ctype == DIFFUSE_INDEX)
        colors->change_diffuse(REGCOLOR(colIndex), newR, newG, newB);
      else if(ctype == SPECULAR_INDEX)
        colors->change_specular(REGCOLOR(colIndex), newR, newG, newB);
    }
  }
  return retval;
}

void CmdColorChange::create_text(void) { }

void CmdColorChange::my_create_text(char *c1, char *c2) {
  *cmdText << "color change ";
  if(ctype == ALPHA_INDEX)
    *cmdText << "alpha ";
  else if(ctype == SHININESS_INDEX)
    *cmdText << "shininess ";
  else if(ctype == COLOR_INDEX)
    *cmdText << "rgb ";
  else if(ctype == AMBIENT_INDEX)
    *cmdText << "ambient ";
  else if(ctype == DIFFUSE_INDEX)
    *cmdText << "diffuse ";
  else if(ctype == SPECULAR_INDEX)
    *cmdText << "specular ";
  *cmdText << c1;
  if(!setDefault) {
    if(!useRGB) {
      *cmdText << " " << c2;
    } else {
      *cmdText << " " << newR;
      if(!(ctype == ALPHA_INDEX || ctype == COLOR_INDEX))
        *cmdText << " " << newG << " " << newB;
    }
  }
  *cmdText << ends;
}

// constructor 1: color name, R, G, B
CmdColorChange::CmdColorChange(char *cCol, float r, float g, float b,
	int newUIid) : Command(Command::COLOR_CHANGE, newUIid) {
  ctype = COLOR_INDEX;
  newColIndex = (-1);
  useRGB = TRUE;
  setDefault = FALSE;
  newR = r;
  newG = g;
  newB = b;
  colIndex = find_color_index(cCol);
  my_create_text(cCol, NULL);
}

// constructor 2: color name, color to use for settings
CmdColorChange::CmdColorChange(char *cCol, char *cCol2, int newUIid)
	: Command(Command::COLOR_CHANGE, newUIid) {
  ctype = COLOR_INDEX;
  useRGB = FALSE;
  setDefault = FALSE;
  colIndex = find_color_index(cCol);
  if(colIndex >= 0)
    newColIndex = (colIndex >= 0 ? find_color_index(cCol2) : (-1));
  my_create_text(cCol, cCol2);
}


// constructor 3: color name, nothing else ... resets color to
// build-in default value
CmdColorChange::CmdColorChange(char *cCol, int newUIid)
	: Command(Command::COLOR_CHANGE, newUIid) {
  ctype = COLOR_INDEX;
  setDefault = TRUE;
  colIndex = find_color_index(cCol);
  my_create_text(cCol, NULL);
}


// constructors 4 ... 6: same as above, but specify what color characteristic
// (shininess, diffuse, etc) to change
CmdColorChange::CmdColorChange(int ct, char *cCol, float r, float g, float b,
	int newUIid) : Command(Command::COLOR_CHANGE, newUIid) {
  ctype = ct;
  newColIndex = (-1);
  useRGB = TRUE;
  setDefault = FALSE;
  newR = r;
  newG = g;
  newB = b;
  colIndex = find_color_index(cCol);
  my_create_text(cCol, NULL);
}

CmdColorChange::CmdColorChange(int ct, char *cCol, char *cCol2, int newUIid)
	: Command(Command::COLOR_CHANGE, newUIid) {
  ctype = ct;
  useRGB = FALSE;
  setDefault = FALSE;
  colIndex = find_color_index(cCol);
  if(colIndex >= 0)
    newColIndex = (colIndex >= 0 ? find_color_index(cCol2) : (-1));
  my_create_text(cCol, cCol2);
}

CmdColorChange::CmdColorChange(int ct, char *cCol, int newUIid)
	: Command(Command::COLOR_CHANGE, newUIid) {
  ctype = ct;
  setDefault = TRUE;
  colIndex = find_color_index(cCol);
  my_create_text(cCol, NULL);
}


///////////////  change the method used to calculate the color scale
int CmdColorScaleMethod::do_execute(void) {
  int retval;
  if(retval = (colors != NULL && newMethod >= 0))
    colors->scale_method(newMethod);
  return retval;
}

void CmdColorScaleMethod::create_text(void) {
  *cmdText << "color scale method " << colorScaleMethod[newMethod] << ends;
}

// constructor: new method
CmdColorScaleMethod::CmdColorScaleMethod(int nm, int newUIid)
	: Command(Command::COLOR_SCALE_METHOD, newUIid) {
  newMethod = nm;
}


///////////////  change the midpoint of the color scale
int CmdColorScaleMidpoint::do_execute(void) {
  int retval;
  if(retval = (colors != NULL))
    colors->scale_midpoint(newval);
  return retval;
}

void CmdColorScaleMidpoint::create_text(void) {
  *cmdText << "color scale midpoint " << newval << ends;
}

// constructor: new val
CmdColorScaleMidpoint::CmdColorScaleMidpoint(float nv, int newUIid)
	: Command(Command::COLOR_SCALE_MIDPOINT, newUIid) {
  newval = nv;
}


///////////////  change the minimum color of the color scale range
int CmdColorScaleMin::do_execute(void) {
  int retval;
  if(retval = (colors != NULL))
    colors->scale_min(newval);
  return retval;
}

void CmdColorScaleMin::create_text(void) {
  *cmdText << "color scale min " << newval << ends;
}

// constructor: new val
CmdColorScaleMin::CmdColorScaleMin(float nv, int newUIid)
	: Command(Command::COLOR_SCALE_MIN, newUIid) {
  newval = nv;
}


///////////////  change the maximum color of the color scale range
int CmdColorScaleMax::do_execute(void) {
  int retval;
  if(retval = (colors != NULL))
    colors->scale_max(newval);
  return retval;
}

void CmdColorScaleMax::create_text(void) {
  *cmdText << "color scale max " << newval << ends;
}

// constructor: new val
CmdColorScaleMax::CmdColorScaleMax(float nv, int newUIid)
	: Command(Command::COLOR_SCALE_MAX, newUIid) {
  newval = nv;
}


/* REVISION HISTORY:********************************************************
 *
 * $Log: CmdColor.C,v $
 * Revision 1.3  1995/05/11  22:05:41  billh
 * Added 'create_text' function to create text representation "on demand".
 * Added text callback functions which are added by UIText, and which are
 * called when a text command is entered and the first word is recognized.
 *
 * Revision 1.2  95/03/24  18:47:40  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.1  1994/12/06  08:36:55  billh
 * Initial revision
 *
 ***************************************************************************/
