/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Buttons.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.7 $	$Date: 1995/05/11 21:36:47 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Provides an interface to request a set of buttons of various types to
 * have a common method of query.  A Button object is created by providing
 * a DisplayDevice which knows how to read the state of specific buttons.
 * Each Button object is told which 'bank' of buttons to use, which then
 * appear as buttons 0, 1, and 2 for the user.  There are several different
 * button 'banks':
 *      POINTER_BANK_0 : the mouse buttons (0=left,1=middle,2=right)
 *      POINTER_BANK_1 : an alternate bank of mouse buttons; if the device
 *                       only has one set of mouse buttons, this is the
 *                       same as POINTER_BANK_0.
 *      KEY_BANK_0 : functions keys set #1 (0=F1, 1=F2, 2=F3)
 *      KEY_BANK_1 : functions keys set #2 (0=F5, 1=F6, 2=F7)
 *      KEY_BANK_2 : functions keys set #3 (0=F9, 1=F10, 2=F11)
 *      ESC_BANK : the escape key ... all three buttons are the same (ESC)
 ***************************************************************************/

#include <string.h>
#include "Buttons.h"
#include "DisplayDevice.h"
#include "utilities.h"


/////////////////////////  constructor: number of buttons this object contains
Buttons::Buttons(DisplayDevice *d, int b) {

  // default values
  has_changed = FALSE;
  myBank = POINTER_BANK_0;
  bankStart = DisplayDevice::B_LEFT;
  for(int i=0; i < NUM_BUTTONS; i++)
    state[i] = changes[i] = FALSE;

  // set values from arguments
  dispDev = d;
  bank(b);
}


// change which bank to use
void Buttons::bank(int b) {

  switch(b) {
    case POINTER_BANK_0:        bankStart = DisplayDevice::B_LEFT; break;
    case POINTER_BANK_1:        bankStart = DisplayDevice::B2_LEFT; break;
    case KEY_BANK_0:            bankStart = DisplayDevice::B_F1; break;
    case KEY_BANK_1:            bankStart = DisplayDevice::B_F5; break;
    case KEY_BANK_2:            bankStart = DisplayDevice::B_F9; break;
    case ESC_BANK:              bankStart = DisplayDevice::B_ESC; break;
    default: return;
  }

  // if here, b was a legal value
  myBank = b;
}


// check the current state of all the buttons, and save them (TRUE=pressed)
void Buttons::check_event(void) {

  int dispButtons = (myBank == ESC_BANK ? 1 : NUM_BUTTONS);

  // get the state of each button and store it, and check if it has changed
  has_changed = FALSE;
  for(int i=0; i < dispButtons; i++) {
    int new_state = dispDev->button_down(bankStart + i);
    changes[i] = new_state - state[i];
    state[i] = new_state;
    has_changed = has_changed || (changes[i] != 0);
  }
}


/* REVISION HISTORY:********************************************************
 *
 * $Log: Buttons.C,v $
 * Revision 1.7  1995/05/11  21:36:47  billh
 * Moved log message to end of file.
 * Redone entirely; now uses the routine 'button_down' in DisplayDevice
 * to make Buttons completely independent of the windowing system.
 *
 * Revision 1.6  95/03/24  18:47:16  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.5  1994/11/08  09:12:15  dalke
 * set up CaveButtons correctly
 *
 * Revision 1.4  1994/10/04  20:29:22  billh
 * Changes to get to compile on HP's ... not there yet,though.
 *
 * Revision 1.3  1994/09/25  22:37:56  billh
 * Changed cave include path, and put in check for VMDCAVE define.
 *
 * Revision 1.2  1994/09/17  10:53:18  dalke
 * added RCS header
 *
 ***************************************************************************/
