/*****************************************************************************/
/* Copyright 1992-1994, Mark C. Surles.  All Rights Reserved                 */
/*****************************************************************************/
/* Interface for Tree class.

   The Tree class is the base class for a 'generic' tree data structure.  All
   nodes in the tree have a pointer to its parent.

   Tree leaves hold a value.  The value is a pointer to void so callers or
   derived classes must explicitly cast the value to its given type.  Leaves
   cannot have children.

   Internal nodes (not leaves) have a list of children Trees.  The list is
   given in the order of those specified at initialization and subsequent calls
   to 'add_child'.  There is not limit on the number of children.  Internal
   nodes do not have a value.
/*****************************************************************************/

#ifndef TreeH
#define TreeH

#include "global.h"
#define NULL 0
class Tree
{
  int    array_dimension;    /* length of kid array */

 protected:
  int    num_kids;           /* number of children */
  Tree **kid;                /* array of pointers to children Trees */
  void  *leaf_value;         /* pointer to node value */
  Tree  *elder;              /* parent */

 public:
  Tree(Tree *parent = NULL);
  Tree(Tree **kids, int, Tree *parent = NULL);
  Tree(void* value, Tree *parent = NULL);		  /* a leaf node */
  ~Tree();

  void set_parent(Tree*);
  virtual void add_child(Tree*);
  virtual void set_value(void*);

  /*************************** Querey routines *****************************/
  Tree* parent();
  Tree* child(int);
  int   number_children();
  void* value();
  int   is_leaf();

  /*************************** Diagnostic routines *************************/
  virtual void dump(char *s=0);
};


/*****************************************************************************/
/*			    Description of methods                           */
/*****************************************************************************/
/*				Public methods                               
/* 

  Tree(Tree *parent = NULL);
    Constructor.  Creates a Tree node that does not have any children.  The
    node can be made into a leaf or a internal node.


  Tree(Tree **kids, int n, Tree *parent = NULL);
    Constructor.  Creates a Tree node with 'n' children.  'kids' lists the
    pointers to the children.


  Tree(void* value, Tree *parent = NULL);
    Constructor.  Creates a Tree node that is a leaf.  'value' is saves as the
    leaf's value.


  ~Tree();
    Destructor.  Removes children, if any.


  void set_parent(Tree *p);
    Sets the node's parent to 'p'.


  void add_child(Tree *c);
    Adds 'c' to the list of children.  The node is no longer a leaf.


  void set_value(void *v);
    If the node has no children, the value is set to 'v'; otherwise, nothing
    happens.

/*                               Querey routines
  Tree* parent();
    Returns a pointer to the node's parent.


  Tree* child(int i);
    Returns a pointer the to i'th child, if it exists; otherwise, returns a 
    NULL pointer.


  int number_children();
    Returns the number of children.


  void* value();
    If the node is a leaf (has no children), the value is returned; otherwise,
    a NULL pointer is returned.


  int is_leaf();
    Returns TRUE if and only if the node is a leaf (i.e. has no children).


/*                            Diagnostic routines
  void dump(char *s=0);
    Diagnostic.  Prints 's' before printing values of member data to standard
    output.


/*****************************************************************************/

#endif
