/*****************************************************************************/
/* Copyright 1992, 1993.  Mark C. Surles.  All Rights Reserved               */
/*****************************************************************************/
/* Interface for Geometry class and its derived classes.

   The Geometry class is the the base class of 'generic' geometrical objects.
   The main intent in the creation of this class is to allow non-graphics
   modules to create geometric (pseudo-graphical) objects without the overhead
   of writing and always linking graphics code.  Regarding Sculpt, this class
   allows the minimizer module to create Geometry objects.  If the system is
   compiled without graphics, the implementation of this class does nothing.
   If it is compiled with graphics, the implementation of this class creates
   a DispObj, a graphical representation of the Geometry, and registers it with
   the display module.  This allows a graphical and non-graphical very of the
   minimizer to be created by chaning a flag at compilation.

/*****************************************************************************/

#ifndef GeometryH
#define GeometryH

#include "global.h"
#include "Tree.h"

class Id;

/* Types needed by GeometryList when constructing DispObjs */
#define INVALID_TYPE    -1
#define GEOMETRY_TYPE    0
#define G_TETRA_TYPE     1
#define G_TACK_TYPE      2
#define G_CYLINDER_TYPE  3
#define G_TUBE_TYPE      4
#define G_TEXT_TYPE      5
#define G_VECTOR_TYPE    6
#define G_TREE_TYPE      7
#define G_SHELL_TYPE     8

/*****************************************************************************/
/* Base class.
/*****************************************************************************/
class Geometry 
{
  char          visib;
  unsigned char r, g, b, a;
  double        sc;

 public:
  /* data */
  Id *id;
  

  /* methods */
  Geometry()                  { id = NULL; visib = TRUE; r=g=b=a= 255; sc = 1;}
  ~Geometry()                 {}
  virtual char  type()        { return GEOMETRY_TYPE; }
  virtual char* read_buffer(char*);
  virtual int   write_buffer(char*);

  /* set routines */
  void set_color(unsigned char r_in, unsigned char g_in, unsigned char b_in,
		 unsigned char a_in = 255)
                               { r = r_in;  g = g_in;  b = b_in;  a = a_in; }
  void set_scale(double s)     { sc = s; }
  void set_visibility(char v)  { visib = v ? TRUE : FALSE; }


  /* get routines */
  char   get_visibility()      { return visib; }
  double get_scale()           { return sc; }
  void   get_color(unsigned char *ro, unsigned char *go, unsigned char *bo,
		   unsigned char *ao)
                               { *ro = r;  *go = g;  *bo = b;  *ao = a; }
  
  virtual void dump(char *s = 0);
};

/*****************************************************************************/
/*			  Derived classes of Geometry                        */
/*****************************************************************************/

/*****************************************************************************/
/* A 'solid' or wireframe tetrahedron at point number 'ndx'.
/*****************************************************************************/
class TetraGeom : public Geometry
{
  int  ndx;     /* point number */
  char solid;   /* solid or wireframe */
 public:
  TetraGeom(int coord_ndx, int is_solid = 1);
  TetraGeom();
  ~TetraGeom()                {}

  char* read_buffer(char*);
  int   write_buffer(char*);
  char  is_solid()             { return solid; }
  int   coord_index()          { return ndx; }
  char  type()                 { return G_TETRA_TYPE; }

  void  dump(char *s = 0);
};


/*****************************************************************************/
/* A thumbtack either at point 'ndx' or position 'pos'.
/*****************************************************************************/
class TackGeom : public Geometry
{
  char   use_coord;
  int    ndx;         /* point number if 'use_coord' is TRUE */
  Point3 pos;         /* position if 'use_coord' is FALSE */
 public:
  TackGeom(int coord_ndx);
  TackGeom(Point3*);
  TackGeom();
  ~TackGeom()                 {}
  char*   read_buffer(char*);
  int     write_buffer(char*);

  char    use_coord_index()   { return use_coord; }
  int     coord_index()       { return ndx; }
  Point3* position()          { return &pos; }
  char    type()              { return G_TACK_TYPE; }

  void dump(char *s = 0);
};

/*****************************************************************************/
/* A cylinder with a given length attached between two points.  Each point is
/* either specified with a point index or a cartesian position.  If 'is_coil'
/* is TRUE, a coil (spring) wraps around the cylinder.
/*****************************************************************************/
class CylinderGeom : public Geometry
{
  char   use_coord1, use_coord2;
  int    ndx1, ndx2;              /* point numbers */
  Point3 pos1, pos2;              /* point positions */
  double rad;                     /* radius */
  int    is_coil;
 public:
  CylinderGeom();
  CylinderGeom(int coord_ndx1, int coord_ndx2, double rad, int coil = FALSE);
  CylinderGeom(Point3*, Point3*, double rad, int coil = FALSE);
  CylinderGeom(int from_ndx, Point3 *to, double rad, int coil = FALSE);
  ~CylinderGeom()                 {}
  char*   read_buffer(char*);
  int     write_buffer(char*);

  char    use_coord_index1()  { return use_coord1; }
  char    use_coord_index2()  { return use_coord2; }
  int     coord_index1()      { return ndx1; }
  int     coord_index2()      { return ndx2; }
  Point3* position1()         { return &pos1; }
  Point3* position2()         { return &pos2; }
  double  radius()            { return rad; }
  int     coiled()            { return is_coil; }
  char    type()              { return G_CYLINDER_TYPE; }

  void dump(char *s = 0);
};


/*****************************************************************************/
/* A tube attached between points.  For example the points [0 1 2 3] specify a
/* tube between 0 and 1 and between 2 and 3.  The points can either be listed
/* with point indices or cartesian positions.
/*****************************************************************************/
typedef Point3 PointPair[2];			  /* for positions */
typedef int    CoordPair[2];			  /* for indices */

class TubeGeom : public Geometry
{
  char       use_coord;
  int        entries;   /* number of rows in following array */
  CoordPair *coords;    /* point indices */
  PointPair *points;    /* point positions */
  double     radius;
  int        sides;

 public:
  TubeGeom(CoordPair *coords_in, int num_elements, double rad=0.2, int side=8);
  TubeGeom(PointPair *pos,       int num_elements, double rad=0.2, int side=8);
  TubeGeom();
  ~TubeGeom();
  char*      read_buffer(char*);
  int        write_buffer(char*);

  char       use_coord_index()   { return use_coord; }
  int        number_entries()    { return entries; }
  CoordPair* coordinates()       { return coords; }
  PointPair* positions()         { return points; }
  double     tube_radius()       { return radius; }
  int        number_sides()      { return sides;  }

  char       type()              { return G_TUBE_TYPE; }

  void dump(char *s = 0);
};


/*****************************************************************************/
/* VectorGeom is a line between two points.  Each point can be specified with
/* a point index or a cartesian coordinate.  The class also has 'update'
/* routines.  This lets the caller change the points without remove the object
/* and creating another.
/*****************************************************************************/
class VectorGeom : public Geometry
{
  char   use_ndx1, use_ndx2;
  int    ndx1, ndx2;				  /* point index */
  Point3 p1, p2;				  /* point position */
 public:
  VectorGeom(int coord_ndx1, int coord_ndx2);
  VectorGeom(int coord_ndx1, Point3*);
  VectorGeom(Point3 *from, Point3 *to);
  VectorGeom();
  ~VectorGeom()                {}
  char*   read_buffer(char*);
  int     write_buffer(char*);

  char    use_coord_index1()  { return use_ndx1 ? TRUE : FALSE; }
  char    use_coord_index2()  { return use_ndx2 ? TRUE : FALSE; }
  int     coord_index1()      { return ndx1; }
  int     coord_index2()      { return ndx2; }
  Point3* position1()         { return &p1; }
  Point3* position2()         { return &p2; }

  void    update(int,int);	    /* change indices */
  void    update(int,Point3*);	    /* change index and position */
  void    update(Point3*,Point3*);  /* change positions */

  char type()                 { return G_VECTOR_TYPE; }

  void dump(char *s = 0);
};


/*****************************************************************************/
/* Text attached to a point index of position.
/*****************************************************************************/
class TextGeom : public Geometry
{
  char   *string;
  char    use_coord;				  /* the text */
  int     ndx;					  /* point index */
  Point3  pos;					  /* point position */

 public:

  TextGeom (int coord_ndx, char *str);
  TextGeom (Point3*, char *str);
  TextGeom();
  ~TextGeom();
  char*   read_buffer(char*);
  int     write_buffer(char*);

  int     coord_index()       { return ndx; }
  char   *get_string()        { return string; }
  Point3 *position()          { return &pos; }
  char    type()              { return G_TEXT_TYPE; }
  char    use_coord_index()   { return use_coord; }

  void dump (char *s = 0);
};


/*****************************************************************************/
/* A spherical shell with radius 'rad', centered about the point index 
/* center_ndx'.  The shell can have polygonal, wireframe, or dot (point)
/* attributes.  The north pole of the shell is aligned along the vector from
/* the center to the 'dir_ndx' point.  
/*
/* A percentage of the shell (counted from the north pole) can be active (on).
/* A percentage of the shell (counted from the north pole) can also be
/* 'dented', not drawn.
/*****************************************************************************/
#define SHG_POINT   0
#define SHG_LINE    1
#define SHG_POLYGON 2

class ShellGeom : public Geometry
{
  int    center_ndx, dir_ndx;
  double rad, cover, dent;
  int    render;

 public:

  ShellGeom (int cntr_ndx, double rad,
             double cover = 1.0, double dent = 0.0,
             int to_ndx = -1, int rend = SHG_LINE);

  ShellGeom (int cntr_ndx, double rad, int rend = SHG_LINE);
  ShellGeom();
  ~ShellGeom();
  char*  read_buffer(char*);
  int    write_buffer(char*);

  int    center_index()     { return center_ndx; }
  double radius()           { return rad; }
  double coverage()         { return cover; }
  double dentage()          { return dent; }
  int    direction_index()  { return dir_ndx; }
  int    render_mode()      { return render; }

  char   type()             { return G_SHELL_TYPE; }

  void dump (char *s = 0);
};


/*****************************************************************************/
/* A TreeGeom is a Tree of Geometry.  The class is multiply derived from Tree
/* and Geometry.
/*****************************************************************************/
class TreeGeom : public Geometry, public Tree
{
 public:
  TreeGeom(TreeGeom *parent = NULL);
  TreeGeom(TreeGeom **kids, int, TreeGeom *parent = NULL);
  TreeGeom(Geometry *leaf, TreeGeom *parent = NULL);
  ~TreeGeom();
  char*   read_buffer(char*);
  int     write_buffer(char*);
  char    type()              { return G_TREE_TYPE; }

  TreeGeom* parent();
  TreeGeom* child(int);
  Geometry* value();

  void dump (char *s = 0);
};


char* read_geometry(char*, Geometry**);

#endif
