/*****************************************************************************/
/* Copyright 1992-1994, Mark C. Surles, SDSC.  All Rights Reserved */
/*****************************************************************************/
/* Interface for machine-independent communication in Sculpt.

   This is a header file for routines that read and write simple items (e.g.
   ints, floats, points) out of or into a buffer (array of bytes).

   The read_block performs a blocking read.  It does not return until 'to_read'
   number of bytes have been read.

   The read_message reads a 'message' from an open socket and places the
   results in the buffer.

   A 'message' in this scheme has the following format.
       1.  The first integer contains the number of bytes that remain in
           the message.
       2.  The remainder of the message contains data written through the
           buffer_write_* routines.

   The data in a 'buffer' is in IEEE format.  The parameters to the routines
   are machine implmentations of 'int', 'float', etc.  The implementation of
   these routines performs the necessary conversion, if any, between IEEE and
   machine-specific formats.


   buffer_write_* routines write the argument in the buffer and return the
   number of bytes written.

   buffer_read_* routines read the item from the buffer into the first
   argument.  The new position of the buffer is returned.

   Two side-effects:  buffer_read_string and buffer_read_Id both create new
   data (a character string or an Id) that are returned.
 *****************************************************************************/

#ifndef COMMUNICATIONH
#define COMMUNICATIONH

#include "global.h"

class Id;

void  write_block(int sock,  char *buff,    int to_write);
void  read_block(int  sock,  char *buff,    int to_read);
int   read_message(int sock, char *message, int message_max);

int   buffer_write_int(char *buff,    int);
int   buffer_write_float(char *buff,  float);
int   buffer_write_string(char *buff, char *str);
int   buffer_write_point3(char *buff, Point3*, int num = 1);
int   buffer_write_Id(char *buff,     Id*);

char* buffer_read_int(int*,       char *buff);
char* buffer_read_float(float*,   char *buff);
char* buffer_read_string(char**,  char *buff);
char* buffer_read_point3(Point3*, char *buff);
char* buffer_read_Id(Id**,        char *buff);

#define INIT_MESSAGE "running"

#endif
