/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: startup.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 95/03/24 18:52:49 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Startup code for VMD ... used only by Global files, contains variables
 * and routines to read initial data files and scripts, and parse
 * command-line options.
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	startup.C,v $
 * Revision 1.3  95/03/24  18:52:49  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.2  1995/01/09  09:41:18  billh
 * The -height and -dist cmdline options do NOT check to make sure the
 * next argument does not start with a -
 *
 * Revision 1.1  1995/01/09  08:54:17  billh
 * Initial revision
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /private/auto143000131/vmdsrc/vmd/billh/src/RCS/startup.C,v 1.3 95/03/24 18:52:49 billh Exp $";
#endif

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "startup.h"
#include "Global.h"
#include "CommandQueue.h"
#include "UIText.h"
#include "CmdMol.h"
#include "MoleculeFile.h"
#include "ConfigList.h"
#include "utilities.h"

//
// externally-accessed variables
//
int which_display;			// type of display to use
float displayHeight, displayDist;	// height and distance of display
int displaySize[2], displayLoc[2];	// X,Y size and position of display
int showTitle;				// how to show the title at start

char *displayTypeNames[NUM_DISPLAY_TYPES] = { "WIN", "CAVE", "TEXT" };
char *titleTypeNames[NUM_TITLE_TYPES] = { "OFF", "ON" };

//
// flags for whether these variables have been set yet
//
int have_which_display = FALSE;
int have_displayHeight = FALSE, have_displayDist = FALSE;
int have_displaySize = FALSE, have_displayLoc = FALSE;
int have_showTitle = FALSE;

//
// filenames for init and startup files
//
static char *dataPath = NULL;
static char *initFileStr = NULL;
static char *startupFileStr = NULL;

//
// filenames for initial molecule to load
//
static char *pdbFileName = NULL;
static char *psfFileName = NULL;
static char *dcdFileName = NULL;


/////////////////////////  routines  ///////////////////////////////

// initialize variables which indicate how VMD starts up, and
// parse the command-line options
void VMDparseCommandLine(int argc, char **argv) {

  // initialize variables
  which_display = INIT_DEFDISPLAY;
  showTitle = INIT_DEFTITLE;
  displayHeight = INIT_DEFHEIGHT;
  displayDist = INIT_DEFDIST;
  displaySize[0] = displayLoc[0] = (-1);

  // find default path to VMD data files, etc.
  char *envtxt;
  if((envtxt = getenv(VMDENVVAR)) != NULL)
    dataPath = stringdup(envtxt);
  else
    dataPath = stringdup(DEF_VMDENVVAR);
  // strip trailing / characters ...
  while(strlen(dataPath) > 0 && dataPath[strlen(dataPath)-1] == '/')
    dataPath[strlen(dataPath)-1] = '\0';

  // go through the arguments
  int ev = 1;
  while(ev < argc) {
    if(*(argv[ev]) != '-' || !strupcmp(argv[ev], "-pdb")) {
      if(pdbFileName) {
	msgErr << "More than one PDB file specified.  Using original file '";
	msgErr << pdbFileName << "'." << sendmsg;
      } else {
        if(*(argv[ev]) != '-')
          pdbFileName = stringdup(argv[ev]);
	else if(argc > (ev + 1) && *(argv[ev+1]) != '-')
          pdbFileName = stringdup(argv[++ev]);
        else
          msgErr << "-pdb must also specify a filename." << sendmsg;
      }

    } else if(!strupcmp(argv[ev], "-psf")) {
      if(psfFileName) {
	msgErr << "More than one PSF file specified.  Using original file '";
	msgErr << psfFileName << "'." << sendmsg;
      } else if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
        psfFileName = stringdup(argv[++ev]);
      } else {
        msgErr << "-psf must also specify a filename." << sendmsg;
      }

    } else if(!strupcmp(argv[ev], "-dcd")) {
      if(dcdFileName) {
	msgErr << "More than one DCD file specified.  Using original file '";
	msgErr << dcdFileName << "'." << sendmsg;
      } else if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
        dcdFileName = stringdup(argv[++ev]);
      } else {
        msgErr << "-dcd must also specify a filename." << sendmsg;
      }

    } else if(!strupcmp(argv[ev], "-dist")) {
      if(argc > (ev + 1)) {
        displayDist = atof(argv[++ev]);
	have_displayDist = TRUE;
      } else
        msgErr << "-dist must also specify a distance." << sendmsg;

    } else if(!strupcmp(argv[ev], "-height")) {
      if(argc > (ev + 1)) {
        displayHeight = atof(argv[++ev]);
	have_displayHeight = TRUE;
      } else
        msgErr << "-height must also specify a distance." << sendmsg;

    } else if(!strupcmp(argv[ev], "-pos")) {
      if(argc > (ev + 2) && *(argv[ev+1]) != '-' && *(argv[ev+2]) != '-') {
        displayLoc[0] = atoi(argv[++ev]);
        displayLoc[1] = atoi(argv[++ev]);
	have_displayLoc = TRUE;
      } else
        msgErr << "-pos must also specify an X Y pair." << sendmsg;

    } else if(!strupcmp(argv[ev], "-size")) {
      if(argc > (ev + 2) && *(argv[ev+1]) != '-' && *(argv[ev+2]) != '-') {
        displaySize[0] = atoi(argv[++ev]);
        displaySize[1] = atoi(argv[++ev]);
	have_displaySize = TRUE;
      } else
        msgErr << "-size must also specify an X Y pair." << sendmsg;

    } else if(!strupcmp(argv[ev], "-init")) {
      // use next argument as init file name
      if(argc > (ev + 1) && *(argv[ev+1]) != '-')
        initFileStr = stringdup(argv[++ev]);
      else
        msgErr << "-init must also have a new file name specified." << sendmsg;

    } else if(!strupcmp(argv[ev], "-startup")) {
      // use next argument as startup config file name
      if(argc > (ev + 1))
        startupFileStr = stringdup(argv[++ev]);
      else
        msgErr << "-startup must also have a new file name specified."
	       << sendmsg;

    } else if(!strupcmp(argv[ev], "-nt")) {
      // do not print out the program title
      showTitle = TITLE_OFF;
      have_showTitle = TRUE;

    } else if(!strupcmp(argv[ev], "-debug")) {
      // turn on debugging
      msgDebug.on(TRUE);
      if(argc > (ev + 1) && argv[ev+1][0] != '-') {
        // also set the debugging level to the one given
        msgDebug.output_level(atoi(argv[++ev]));
      }

    } else if (!strupcmp(argv[ev], "-disp")) {  // startup Display
      ev++;
      have_which_display = TRUE;
      if (!strupcmp(argv[ev], "cave")) {  // use the cave 
        which_display = DISPLAY_CAVE;
      } else if (!strupcmp(argv[ev], "win")) { // use the GLDisplay (default)
        which_display = DISPLAY_WIN;
      } else if (!strupcmp(argv[ev], "text")) { // do not use any display
        which_display = DISPLAY_TEXT;
      } else if (!strupcmp(argv[ev], "none")) { // do not use any display
        which_display = DISPLAY_TEXT;
      } else {
        have_which_display = FALSE;
        msgErr << "-disp options are 'win' (default) 'cave' or 'text | none'";
	msgErr << sendmsg;
      }

    } else if (!strupcmp(argv[ev], "-?") || !strupcmp(argv[ev], "-h")) {
      // print out command-line option summary
      msgInfo << "Available command-line options:" << sendmsg;
      msgInfo << "<filename> (with no - option)     Load specified PDB file\n";
      msgInfo << "\t-dcd <filename>     Load specified DCD trajectory file\n";
      msgInfo << "\t-debug [n]          Turn on debugging, at level n\n";
      msgInfo << "\t-disp <win | cave | text | none> Specify display device";
      msgInfo << sendmsg;
      msgInfo << "\t-dist <d>           Distance from origin to screen";
      msgInfo << sendmsg;
      msgInfo << "\t-h (or -?)          Display this command-line summary\n";
      msgInfo << "\t-height <h>         Height of display screen";
      msgInfo << sendmsg;
      msgInfo << "\t-init <filename>    Specify initialization file";
      msgInfo << sendmsg;
      msgInfo << "\t-pdb <filename>     Load specified PDB file\n";
      msgInfo << "\t-pos <X> <Y>        Lower-left corner position of display";
      msgInfo << sendmsg;
      msgInfo << "\t-psf <filename>     Load specified PSF structure file\n";
      msgInfo << "\t-nt			No title display at start" << sendmsg;
      msgInfo << "\t-size <X> <Y>       Size of display" << sendmsg;
      msgInfo << "\t-startup <filename> Specify startup script file";
      msgInfo << sendmsg;
      exit(0);

    } else {
      // unknown option
      msgErr << "Unknown command-line option '" << argv[ev] << "'"
#ifdef VMDCAVE
             << " ... ignoring - it could be a CAVE option."
#endif
	     << sendmsg;
    }
    ev++;
  }
  
  // command-line options have been parsed ... any init status variables that
  // have been given initial values will have flags saying so, and their
  // values will not be changed when the init file(s) is parsed.
}


// process an init file, by reading it in and checking for keywords.  Items
// that have had their value set already will not be changed.
void VMDprocessInitFile(char *fname) {
  ConfigList *configList;
  char *dispArgv[64], *dispStr, *strList;
  int dispArgc;

  // read in the file, which is of the form <keyword> = <value>
  configList = new ConfigList;
  if(! configList->read(fname)) {
    // file not found ... just return
    delete configList;
    return;
  }
  
  // check for proper keywords, and set values if found
  if(!have_which_display && (strList = configList->find("DISPLAY"))) {
    for(int i=0; i < NUM_DISPLAY_TYPES; i++) {
      if(!strupcmp(strList, displayTypeNames[i])) {
	which_display = i;
	break;
      }
    }
  }
  
  if(!have_showTitle && (strList = configList->find("TITLE"))) {
    for(int i=0; i < NUM_TITLE_TYPES; i++) {
      if(!strupcmp(strList, titleTypeNames[i])) {
	showTitle = i;
	break;
      }
    }
  }
  
  if(!have_displayHeight && (strList = configList->find("SCRHEIGHT")))
    displayHeight = atof(strList);

  if(!have_displayDist && (strList = configList->find("SCRDIST")))
    displayDist = atof(strList);

  if(!have_displayLoc && (strList = configList->find("SCRPOS"))) {
    dispStr = NULL;
    if((dispStr = str_tokenize(strList, &dispArgc, dispArgv)) != NULL
    		&& dispArgc == 2) {
      displayLoc[0] = atoi(dispArgv[0]);
      displayLoc[1] = atoi(dispArgv[1]);
    } else {
      msgErr << "Illegal SCRPOS initialization setting '" << strList << "'.";
      msgErr << sendmsg;
    }
    if(dispStr)  delete [] dispStr;
  }
  
  if(!have_displaySize && (strList = configList->find("SCRSIZE"))) {
    dispStr = NULL;
    if((dispStr = str_tokenize(strList, &dispArgc, dispArgv)) != NULL
    		&& dispArgc == 2) {
      displaySize[0] = atoi(dispArgv[0]);
      displaySize[1] = atoi(dispArgv[1]);
    } else {
      msgErr << "Illegal SCRSIZE initialization setting '" << strList << "'.";
      msgErr << sendmsg;
    }
    if(dispStr)  delete [] dispStr;
  }

  // done with config list .. delete it
  delete configList;
}



// read in the init file(s).  The order of searching is the following, and EACH
// file found in this list will be processed.
//	1. global data dir as specified by dataPath
//	2. $HOME
//	3. .
// If an init file was specified on the command line, the above three searches
// are skipped, only the specified file is processed.
void VMDreadInit(void) {
  char namebuf[512], *envtxt;
  
  // use an init file if previously specified, 
  if(initFileStr) {
    VMDprocessInitFile(initFileStr);
    delete [] initFileStr;
    return;
  }
  
  // 1. look in global data dir for the file.
  strcpy(namebuf, dataPath);
  strcat(namebuf, "/");
  strcat(namebuf, VMD_INITFILE);
  VMDprocessInitFile(namebuf);
  
  // 2. look in home directory
  if (envtxt = getenv("HOME")) {
    strcpy(namebuf, envtxt);
    strcat(namebuf, "/");
    strcat(namebuf, VMD_INITFILE);
    VMDprocessInitFile(namebuf);
  }
  
  // 3. look in current directory
  strcpy(namebuf, VMD_INITFILE);
  VMDprocessInitFile(namebuf);

}



// read in the startup script, execute it, and then execute any other commands
// which might be necessary (i.e. to load any molecules at start)
// This searches for the startup file in the following
// places (and in this order), reading only the FIRST one found:
//		1. Current directory
//		2. Home directory
//		3. 'Default' directory (here, /usr/local/vmd)
// If a name was given in the -startup switch, that file is checked for ONLY.
void VMDreadStartup(void) {
  char namebuf[512], *envtxt;
  int found = FALSE;
  struct stat statbuf;

  // check if the file is available
  if(startupFileStr) {	// name specified by -startup
    if(stat(startupFileStr, &statbuf) == 0) {
      found = TRUE;
      strcpy(namebuf, startupFileStr);
    }
    delete [] startupFileStr;
  } else {	// search in different directories, for default file
    // first, look in current dir
    strcpy(namebuf, VMD_STARTUP);
    if(stat(namebuf, &statbuf) == 0) {
      found = TRUE;
    } else {
      // not found in current dir; look in home dir
      if((envtxt = getenv("HOME")) != NULL)
        strcpy(namebuf,envtxt);
      else
        strcpy(namebuf,".");
      strcat(namebuf,"/");
      strcat(namebuf,VMD_STARTUP);
      if(stat(namebuf, &statbuf) == 0) {
        found = TRUE;
      } else {
        // not found in home dir; look in default dir
	strcpy(namebuf, dataPath);
	strcat(namebuf,"/");
	strcat(namebuf,VMD_STARTUP);
        if(stat(namebuf, &statbuf) == 0) {
          found = TRUE;
	}
      }
    }
  }

  // if the file was found, read in the commands there
  if(found) {
    MSGDEBUG(1, "Startup file '" << namebuf << "'found." << sendmsg);
    uiText->read_from_file(namebuf);
  } else {
    MSGDEBUG(1, "No startup file found." << sendmsg);
  }
  
  //
  // execute any other commands needed at start
  //
  
  // read in molecule if requested via command-line switches
  if(pdbFileName || (psfFileName && dcdFileName)) {
    int sftype, cftype;
    char *strfile, *coorfile;
    sftype = (psfFileName ? MoleculeFile::PSF : MoleculeFile::PDB);
    cftype = (dcdFileName ? CoorFileData::DCD : CoorFileData::PDB);
    strfile = (psfFileName ? psfFileName : pdbFileName);
    coorfile = (psfFileName ? (dcdFileName ? dcdFileName : pdbFileName) : NULL);
    commandQueue->append(new CmdMolNew(strfile, sftype, coorfile, cftype));
  }
  
  // delete filename storage
  if(pdbFileName)  delete [] pdbFileName;
  if(psfFileName)  delete [] psfFileName;
  if(dcdFileName)  delete [] dcdFileName;
  if(dataPath)  delete [] dataPath;
}
