/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Timestep.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.7 $	$Date: 95/03/24 18:51:57 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Timestep class, which stores coordinates, energies, etc. for a
 * single timestep.
 *
 * Note: As more data is stored for each step, it should go in here.  For
 * example, H-Bonds could be calculated each step.
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	Timestep.C,v $
 * Revision 1.7  95/03/24  18:51:57  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.6  1994/11/11  16:30:37  billh
 * init routine now recalculates even when previously Initialized.
 *
 * Revision 1.5  94/11/10  17:19:32  billh
 * Only deletes patchData if number of patches changes.
 * 
 * Revision 1.4  94/11/09  02:50:28  billh
 * Added finished support for displaying patches for a remote molecule.
 * 
 * Revision 1.3  94/10/05  04:38:23  billh
 * Took out double backslash from text, even in comments.
 * 
 * Revision 1.2  1994/09/24  20:10:05  billh
 * Added 'maxrad' member, which is the radius of a sphere which will
 * entirely contain the positions for this timestep, if it were centered
 * at the center of volume.
 *
 * Revision 1.1  94/09/17  09:11:36  billh
 * Initial revision
 * 
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /private/auto143000131/vmdsrc/vmd/billh/src/RCS/Timestep.C,v 1.7 95/03/24 18:51:57 billh Exp $";
#endif

#include <math.h>
#include "Timestep.h"
#include "Inform.h"
#include "Atom.h"

///////////////////////////  constructor  

Timestep::Timestep(int n, float DT, float *newdata) {

  MSGDEBUG(2,"Creating new timestep with " << n << " atoms, dt=" << DT);
  MSGDEBUG(2,sendmsg);

  for(int i=0; i < TSENERGIES; energy[i++] = 0.0);
  num = n;
  dt = DT;
  pos = new float[ATOMCOORDS * num];
  if(newdata)
    data = newdata;
  else
    data = new float[ATOMEXTRA * num];
  needDataDelete = (newdata == NULL);
  scale_factor = 1.0;

  numPatches = 0;
  patchData = NULL;

  Initialized = FALSE;
}

///////////////////////////  destructor  

Timestep::~Timestep(void) {
  if(needDataDelete && data)
    delete [] data;
    
  if(patchData)
    delete [] patchData;
}

//////////////////////////  public routines  

// create storage to store n patches; returns pointer to beginning
// of data block
float *Timestep::create_patch_storage(int n) {

  // delete old storage, if necessary
  if(patchData) {
    // see if we already have the proper size for patches
    if(numPatches == n)
      return patchData;

    // number differs; remove old storage
    delete [] patchData;
    patchData = NULL;
    numPatches = 0;
  }
  
  // create new storage, if n is legal
  if(n > 0) {
    patchData = new float[n * TSPATCHDATA];
    numPatches = n;
  }
  
  return patchData;
}


// calculate the max/min values for all the quantities, and anything else
// required based on atom coordinates, etc.  Used for scaling
// and translating purposes.
void Timestep::init(void) {
  int i, j;
  float *mpos = pos, *mdata = data, *mpatch = patchData;

  // only do this if there are atoms
  if(!num)
    return;		

  // inialize min/max positions
  for(i=0; i < ATOMCOORDS; i++) {
    minpos[i] = maxpos[i] = pos[i];		// take values for first atom
    COV[i] = 0.0;
  }  
  for(i=0; i < ATOMEXTRA; i++) {
    mindata[i] = maxdata[i] = data[i];		// take values for first atom
  }

  // calc ranges for all atoms
  for(i=0; i < num; i++) {
  
    for(j=0; j < ATOMCOORDS; j++) {
      COV[j] += mpos[j];
      if(mpos[j] < minpos[j])
        minpos[j] = mpos[j];
      else if(mpos[j] > maxpos[j])
        maxpos[j] = mpos[j];
    }

    for(j=0; j < ATOMEXTRA; j++) {
      if(mdata[j] < mindata[j])
        mindata[j] = mdata[j];
      else if(mdata[j] > maxdata[j])
        maxdata[j] = mdata[j];
    }

    mpos += ATOMCOORDS;
    mdata += ATOMEXTRA;
  }
  
  // calculate ranges for patch data
  if(numPatches > 0 && mpatch) {
  
    // initialize min and max values
    for(i=0; i < TSPATCHDATA; i++)
      minpatch[i] = maxpatch[i] = mpatch[i];
      
    // calc ranges
    for(i=0; i < numPatches; i++) {

      for(j=0; j < TSPATCHDATA; j++) {
        if(mpatch[j] < minpatch[j])
	  minpatch[j] = mpatch[j];
	else if(mpatch[j] > maxpatch[j])
	  maxpatch[j] = mpatch[j];
      }
      
      mpatch += TSPATCHDATA;
    }
  }

  // calculate center-of-volume and scale factor
  scale_factor = maxpos[0] - minpos[0];
  for(i=0; i < ATOMCOORDS; i++) {
    COV[i] /= (float)num;
    if((maxpos[i] - minpos[i]) > scale_factor)
      scale_factor = maxpos[i] - minpos[i];
  }
  scale_factor = 1.5 / scale_factor;

  // calculate enclosing sphere radius
  maxrad = 0.0;
  float newrad = 0.0;
  mpos = pos;
  for(i=0; i < num; i++) {
    float x = *(mpos++) - COV[0];
    float y = *(mpos++) - COV[1];
    float z = *(mpos++) - COV[2];
    newrad = sqrtf(x*x + y*y + z*z);
    if(newrad > maxrad)
      maxrad = newrad;
  }

  MSGDEBUG(3,"Timestep ranges calculated: for " << num << " atoms:"<< sendmsg);
  MSGDEBUG(3,"  Item       Min       Max" << sendmsg);
  MSGDEBUG(3,"--------- --------- -----------" << sendmsg);
  for(i=0; i < ATOMCOORDS; i++) {
    MSGDEBUG(3,"  pos[" << i << "]     " << minpos[i] << "\t " << maxpos[i]);
    MSGDEBUG(3,sendmsg);
  }
  for(i=0; i < ATOMEXTRA; i++) {
    MSGDEBUG(3," data[" << i << "]     " << mindata[i] << "\t " << maxdata[i]);
    MSGDEBUG(3,sendmsg);
  }
  if(numPatches > 0) {
    for(i=0; i < TSPATCHDATA; i++) {
      MSGDEBUG(3,"  patch[" << i << "]   " << minpatch[i] << "\t ");
      MSGDEBUG(3,maxpatch[i] << sendmsg);
    }
  }
  
  Initialized = TRUE;
}
