/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Remote.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.12 $	$Date: 95/03/24 18:51:32 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Remote object, which maintains a connection to a remote computer that
 * provides data and structures.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	Remote.C,v $
 * Revision 1.12  95/03/24  18:51:32  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.11  1994/11/10  17:26:51  billh
 * Fixed stupid bugs.
 *
 * Revision 1.10  94/11/10  17:19:09  billh
 * Added option to save every Nth transferred frame.
 * 
 * Revision 1.9  1994/11/02  01:36:40  billh
 * When receiving a timestep, must call 'get_next_ts' in Remote first, get
 * all the data, and when done, call 'done_with_ts' in Remote.
 *
 * Revision 1.8  94/10/28  18:32:36  billh
 * Temporarily took out commands to deallocate structure with list of
 * remote programs.
 * 
 * Revision 1.7  1994/10/21  04:14:45  billh
 * Added include of stdlib.h
 *
 * Revision 1.6  1994/10/21  03:51:20  billh
 * Several bug fixes, and added routines to get and change settings.
 *
 * Revision 1.5  1994/10/19  20:25:03  billh
 * Changed order of operations in destructor to prevent core dump when
 * deleting mdcomm structures.
 *
 * Revision 1.4  1994/10/05  04:38:23  billh
 * Took out double backslash from text, even in comments.
 *
 * Revision 1.3  1994/10/02  05:16:14  billh
 * Fixed some problems with over-zealous error checking.
 *
 * Revision 1.2  1994/10/01  11:01:37  billh
 * Several bug fixes due to putting in MoleculeRemote code.
 *
 * Revision 1.1  94/10/01  03:08:43  billh
 * Initial revision
 *  
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /private/auto143000131/vmdsrc/vmd/billh/src/RCS/Remote.C,v 1.12 95/03/24 18:51:32 billh Exp $";
#endif

#include <string.h>
#include <stdlib.h>
#include "Remote.h"
#include "Inform.h"
#include "utilities.h"


// static string storage
static char *no_err_msg = "No error.";

// static status storage
static char *statusStrings[Remote::TOTAL_STATUS] = {
	"No Connection", "Selecting Application",
	"Editing Parameters", "Running Simulation" };

// string descriptions, and types, for available simulation settings
char *remoteSettingName[Remote::TOTAL_SETTINGS] = {
	" Trans Rate: ", "   Timestep: ", "Temperature: ", "Keep Rate:" };
char *remoteSettingKeyword[Remote::TOTAL_SETTINGS] = {
	"rate", "dt", "temp", "keep" };
char remoteSettingType[Remote::TOTAL_SETTINGS] = { 'i', 'f', 'f', 'i' };


// static flags for this class
int Remote::run_mdc_init = FALSE;

/****************************************************************************
 * NOTES:
 *
 * A remote connection proceeds in three parts:
 *	1) Make initial connection to remote manager.  This requires setting
 * the remote machine and username, and is done in the constructor.  This
 * results in this object getting a list of possible remote applications (if
 * new app is to be started), or a list of running apps (if a connection to
 * a running program is to be made).
 *
 *	2) If a new app is to be started, a parameter list is fetched and
 * edited.  If a connection to running app is required, this step is not
 * necessary.
 *
 *	3) Connect to the program, possible starting it first.
 ****************************************************************************/

///////////////////////////  constructor  
Remote::Remote(char *newuser, char *machine) {

  // make copies of machine and username
  computer = stringdup(machine);
  username = stringdup(newuser);

  // initialize our internal variables
  startingNewApp = TRUE;		// default is to start a new app
  editingParams = FALSE;		// no parameters retrieved yet
  madeConnection = FALSE;		// no connection yet either
  application = (-1);			// if < 0, no app defined yet
  availApps = 0;
  availJobs = 0;
  availOptions = 0;
  returnValue = MDC_EOK;
  remoteStatus = NO_CONNECT;
  
  // modifiable parameters
  transferRate = 1;
  timeStep = 0.001;
  Temperature = 300.0;
  saveFrameRate = 0;
  
  // initialization for mdcomm library
  proglist = NULL;
  joblist = NULL;
  optlist = NULL;
  consumer = NULL;
  md_struct = NULL;
  if(!run_mdc_init) {
    if(mdc_init() != 0) {
      msgErr << "Cannot initialize connection library." << sendmsg;
      remoteStatus = CONNECT_ERROR;
      returnValue = MDC_EINIT;
    } else {
      run_mdc_init = TRUE;
    }
  }
  
  // get data about available jobs and applications
  if(returnValue == MDC_EOK) {
    remoteStatus = SELECTING_APP;
    
    // get list of apps
    if((proglist = mdc_avail_apps(computer)) == NULL) {
      msgErr << "Cannot retrieve available application from " << computer;
      msgErr << "." << sendmsg;
      remoteStatus = CONNECT_ERROR;
      returnValue = mdc_errno;
    } else {
      // find number of applications in list
      struct mdc_proglist *proglistscan = proglist;
      while(proglistscan) {
        availApps++;
	proglistscan = proglistscan->next;
      }
    }
    
    // get list of running jobs
    if(returnValue == MDC_EOK) {
      if((joblist = mdc_ps(computer, username)) == NULL) {
        // there are no running jobs for the user that can be found
	msgWarn << "There are no simulations running on " << computer;
	msgWarn << " for user " << username << "." << sendmsg;
      } else {
        // find number of running applications
	struct mdc_joblist *joblistscan = joblist;
        while(joblistscan) {
          availJobs++;
	  joblistscan = joblistscan->next;
        }
      }
    }
  }
}


///////////////////////////  destructor  
Remote::~Remote(void) {
  // free up any other mdcomm-allocated space
  // NOTE: THESE ARE COMMENTED OUT DUE TO STRANGE BUG ... THIS IS A
  // MEMORY LEAK IF NOT FIXED SOMETIME ... wfh
/*
  if(proglist)
    mdc_proglist_dealloc(proglist);
  if(joblist)
    mdc_joblist_dealloc(joblist);
*/
  proglist = NULL;
  joblist = NULL;

  // kill the processes, if they are running
  remote_close(FALSE);
  
  // free other allocated storage
  if(username) delete [] username;
  if(computer) delete [] computer;
}


//////////////////////////  public routines  

// return the currently selected application name.  NULL if error.
char *Remote::app(void) {

  // are we running?
  if(application < 0)
    return NULL;
    
  // running from previous job, or new job?
  if(startingNewApp)
    return avail_app_name(application);
  else
    return running_job_name(application);
}


// current status of connection (not the same as the error status string)
char *Remote::status(void) {
  return statusStrings[remoteStatus];
}


// name of molecule to simulate (taken from STR keyword, if available)
char *Remote::mol_name(void) {
  // check for STR keyword
  for(int i=0; i < options(); i++) {
    if(!strcmp(option_keyword(i), "STR"))
      return option_default(i);
  }
  
  // if here, not found ... return NULL
  return NULL;
}


// get parameters, for the Nth available app.  Return success.
// This can only work properly if we have just created this object, and
// have not yet selected an app to work with.  This is used to start a new
// job, not to connect to a running one.
int Remote::get_parameters(int n) {

  // are we running?
  if(application >= 0 || n < 0 || n >= availApps)
    return FALSE;
    
  // get parameters, if possible
  if((optlist = md_get_usage(computer, avail_app_name(n))) != NULL) {
    // got parameters, switch to editing parameters state
    availOptions = optlist->count;
    application = n;
    startingNewApp = TRUE;
    editingParams = TRUE;
    madeConnection = FALSE;
    remoteStatus = EDITING_PARAMS;
    returnValue = MDC_EOK;
    return TRUE;
  }
  
  // if here, could not get options
  returnValue = mdc_errno;
  return FALSE;
}


// attach to the Nth running job.  Return success.
// This can only work properly if we have just created this object, and
// have not yet selected an app to work with.  This is used to connect to a
// previously-running job, instead of starting a new one.
int Remote::attach_to_job(int n) {

  // are we running?
  if(application >= 0 || n < 0 || n >= availJobs)
    return FALSE;
    
  // make connection, if possible
  struct mdc_joblist *joblistscan = joblist;
  int jobcount = 0;
  while(jobcount++ < n)
    joblistscan = joblistscan->next;
  consumer = dynd_connect(computer,(joblistscan->job)->port,&md_struct,&state);
  if(consumer) {
    // connection was successful
    application = n;
    startingNewApp = FALSE;
    editingParams = FALSE;
    madeConnection = TRUE;
    remoteStatus = CONNECTED;
    returnValue = MDC_EOK;
    return TRUE;
  }
  
  // if here, connection failed
  returnValue = mdc_errno;
  return FALSE;
}


// if a new simulation is being run, and get_parameters has been called with
// the parameters all set up, this makes a connection and runs the sim.
// return success.
int Remote::start_new_simulation(void) {

  // are we currently editing parameters?
  if(!editingParams)
    return FALSE;
    
  // make a connection using the current parameters
  int bufsiz = 0;		// actually just ignored my mdcomm
  consumer = dynd_exec_and_connect(computer, username, optlist,
  			&md_struct, &state, bufsiz);
  if(consumer) {
    // connection was successful
    editingParams = FALSE;
    madeConnection = TRUE;
    remoteStatus = CONNECTED;
    returnValue = MDC_EOK;
    return TRUE;
  }
  
  // if here, connection failed
  returnValue = mdc_errno;
  return FALSE;
}


// close: finishes the connection, return to original state
// if arg = TRUE, and an application is running, this will disconnect from
// the job but let it keep running.  Otherwise, this will kill the remote
// application.
void Remote::remote_close(int keepRunning) {

  // are we currently connected?
  if(madeConnection)  {
    // do we let the remote application keep running?
    if(keepRunning)
      mdc_cmd_disconnect(consumer);
    else
      mdc_cmd_kill(consumer);
      
    mdc_arena_dealloc(consumer);
    consumer = NULL;
  }

  // deallocate memory for other lists
  if(optlist)
    mdc_optlist_dealloc(optlist);
  optlist = NULL;
  availOptions = 0;

  // return variables to initial state
  startingNewApp = TRUE;		// default is to start a new app
  editingParams = FALSE;		// no parameters retrieved yet
  madeConnection = FALSE;		// no connection yet either
  application = (-1);			// if < 0, no app defined yet
  returnValue = MDC_EOK;
  remoteStatus = NO_CONNECT;
  transferRate = 1;
  timeStep = 0.001;
  Temperature = 300.0;
  saveFrameRate = 10;

  return;
}


// return index of for given setting, -1 if error
int Remote::setting_index(char *p) {
  int n = 0;
  while(n < avail_settings()) {
    if(!strcmp(p, remoteSettingKeyword[n]))
      return n;
    n++;
  }
  return (-1);
}


// change the value of the specified setting.  return success.
int Remote::change_setting(char *p, char *newval) {
  int n;

  // can only do so if sim is running and n is valid
  if(madeConnection && (n = setting_index(p)) >= 0) { 
    if(n == TRANSFER_RATE) {
      int nv = atoi(newval);
      mdc_cmd_rate(consumer, nv);
      transferRate = nv;
    } else if(n == TEMPERATURE) {
      float nv = atof(newval);
      mdc_cmd_temperature(consumer, nv);
      Temperature = nv;
    } else if(n == TIMESTEP) {
      float nv = atof(newval);
      mdc_cmd_dt(consumer, nv);
      timeStep = nv;
    } else if(n == SAVEFRAME_RATE) {
      saveFrameRate = atoi(newval);
    } else
      return FALSE;		// unknown setting
  } else
    return FALSE;		// no connection, or bad parameter
    
  return TRUE;
}


// write the value of the specified setting into the given string storage.
// return success.
int Remote::get_setting(char *p, char *newval) {
  int n;

  // can only do so if n is valid
  if((n = setting_index(p)) >= 0) { 
    if(n == TRANSFER_RATE)
      sprintf(newval, "%d", transferRate);
    else if(n == TEMPERATURE)
      sprintf(newval, "%f", Temperature);
    else if(n == TIMESTEP)
      sprintf(newval, "%f", timeStep);
    else if(n == SAVEFRAME_RATE)
      sprintf(newval, "%d", saveFrameRate);
    else
      return FALSE;		// unknown setting
  } else
    return FALSE;		// bad parameter
    
  return TRUE;
}


// write the value of the specified setting into the given string storage.
// return success.
int Remote::get_setting(int n, char *newval) {
  // can only do so if n is valid
  if(n == TRANSFER_RATE)
    sprintf(newval, "%d", transferRate);
  else if(n == TEMPERATURE)
    sprintf(newval, "%f", Temperature);
  else if(n == TIMESTEP)
    sprintf(newval, "%f", timeStep);
  else if(n == SAVEFRAME_RATE)
    sprintf(newval, "%d", saveFrameRate);
  else
    return FALSE;		// unknown setting
    
  return TRUE;
}


// process list of options of the form "<keyword> = <value>" in a file
// use config routines to process file data
// return success.
int Remote::read_param_file(FILE *f) {
  int argc, slen, retval;
  char *tokstr, *argv[64];
  char strbuf[256];

  if(!optlist)
    return FALSE;

  // loop over the file, reading until EOF
  while(fgets(strbuf,255,f)) {
    if((slen=strlen(strbuf)) > 0) {
      if(strbuf[slen-1] == '\n')
        strbuf[slen-1] = '\0';
    }
    if((tokstr = command_tokenize(strbuf, &argc, argv)) != NULL) {
      retval = set_option(argv[0], argv[2]);
      delete [] tokstr;
      if(!retval)
        return FALSE;
    }
  }
  
  return TRUE;
}


// write current list of options to a file, in form "<keyword> = <value>"
// return success.
int Remote::write_param_file(FILE *f) {
  int i;
  
  if(!optlist)
    return FALSE;
  
  // only write parameters that have assigned values to the file
  for(i=0; i < optlist->count; i++) {
    if((optlist->opt[i])->def)
      fprintf(f,"%s = %s\n",(optlist->opt[i])->keyword,(optlist->opt[i])->def);
  }
  
  return TRUE;
}


// check for a new timestep, and return TRUE if available.
int Remote::next_ts_available(void) {
  int retval;
  
  if(madeConnection && returnValue == MDC_EOK) {
    if((retval = dynd_probe(consumer)) == MDC_EOK) {
      MSGDEBUG(1,"A timestep is available from '" << host() << "' ... ");

      return TRUE;
    } else if(retval == (-1)) {
      returnValue = mdc_errno;
      remoteStatus = READ_ERROR;
      msgErr << "Problem encounted while probing for new timestep from ";
      msgErr << host() << ":\n   " << error_status_desc() << sendmsg;
    } else {
      MSGDEBUG(2,"No timestep is available from '" << host() << "' ... ");
    }
  }
      
  // if here, problem
  return FALSE;
}


// gets timestep data, if available.  returns TRUE if successful.
int Remote::get_next_ts(void) {

  if(next_ts_available()) {
    dynd_next_ts(consumer,&state);
    remoteStatus = CONNECTED;
    
    MSGDEBUG(1,"read step " << *(state.step) << sendmsg);

    return TRUE;
  }
  
  // if here, no data available
  return FALSE;
}


// signal a new timestep can be sent.  Should be called after get_next_ts
// has returned true, and the data has been fetched properly.
void Remote::done_with_ts(void) {
  // this just assumes a timestep has been fetched with get_next_ts previously
  dynd_freebuf(consumer);
}


// query return status description
char *Remote::error_status_desc(int st) {
  if(st == MDC_EOK)
    return no_err_msg;
  else
    return mdc_errlist[st];
}


// query return status description of current error
char *Remote::error_status_desc(void) {
  if(returnValue == MDC_EOK)
    return no_err_msg;
  else
    return mdc_errlist[returnValue];
}


// return pid and uid of the Nth job
void Remote::running_job_ids(int n, int &pid, int &uid) {

  if(n < 0 || n >= availJobs)
    return;
    
  struct mdc_joblist *joblistscan = joblist;
  int jobcount = 0;
  while(jobcount++ < n)
    joblistscan = joblistscan->next;

  pid = (joblistscan->job)->jid;
  uid = (joblistscan->job)->uid;
}


// return name of Nth available job's program
char *Remote::running_job_name(int n) {

  if(n < 0 || n >= availJobs)
    return NULL;
    
  struct mdc_joblist *joblistscan = joblist;
  int jobcount = 0;
  while(jobcount++ < n)
    joblistscan = joblistscan->next;

  return (joblistscan->job)->prog;
}


// return name of Nth available job's program
char *Remote::avail_app_name(int n) {

  if(n < 0 || n >= availApps)
    return NULL;
    
  struct mdc_proglist *proglistscan = proglist;
  int appcount = 0;
  while(appcount++ < n)
    proglistscan = proglistscan->next;

  return proglistscan->prog;
}


// strings used by option_string routine
static char *optiontypeString[3] = { "f", "i", "s" };
static char formattedOptionString[256];


// complete string description of Nth option, including keyword, value,
// required, and type
char *Remote::option_string(int n) {

  if(n < 0 || n >= availOptions) {
    strcpy(formattedOptionString, "");
  } else {
    sprintf(formattedOptionString, "%12.12s: %30.30s %s %s ==> %s",
    	option_keyword(n), option_desc(n), (option_req(n) ? "REQ" : "opt"),
	optiontypeString[option_type(n)],
	(option_default(n) ? option_default(n) : ""));
  }
  
  return formattedOptionString;
}


