/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Pickable.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 95/03/24 18:51:17 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A Pickable object is one which contains data which may be selected by
 * using a pointer device in an open DisplayDevice.  Each Pickable registers
 * itself with a PickList object which indicates it has items in its
 * drawing list which may be picked, and which indicates it should be told
 * when something has been successfully selected.  There are one or many
 * picking modes, which are represented by derivations of the PickMode class.
 * When picking is done, it is done while in a current pick mode, which
 * controls what to do with the information.
 *
 * A Pickable must provide versions of virtual routines which tell what
 * pick modes the object is interested in.
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	Pickable.h,v $
 * Revision 1.3  95/03/24  18:51:17  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.2  1995/03/17  22:27:25  billh
 * Added virtual function 'pickable_on' to indicate if a Pickable object
 * is currently being displayed.
 *
 * Revision 1.1  1995/02/22  03:56:04  billh
 * Initial revision
 *
 ***************************************************************************/
#ifndef PICKABLE_H
#define PICKABLE_H

#include "utilities.h"
extern class PickList;
extern class PickMode;
extern class DisplayDevice;


class Pickable {

protected:
  // the picklist with which we registered
  PickList *pickList;

  // creates a new pick mode of a type specified by the code integer argument.
  // this is typically only used by add_pick_mode; the code is both defined
  // by and used by the derived class
  virtual PickMode *create_pick_mode(int);

public:
  // constructor: PickList with which to register.  If NULL, do nothing
  Pickable(PickList * = NULL);
  
  // destructor: unregister with the picklist
  virtual ~Pickable(void);

  //
  // public utility routines
  //
  
  // register with the given PickList.  If already registered, deregisters
  // with the old one first
  void register_with_picklist(PickList *);

  // check to see if a PickMode with the given name has been created yet
  // for the PickList.  If so, return the index of the mode.  If not,
  // this routine calls the virtual create_pick_mode with the 2nd argument,
  // which is a code for the derived class to know what kind of PickMode
  // object to create.  Returns (-1) if some sort of error occurs.
  int add_pick_mode(char *, int);

  //
  // public virtual informative routines
  //
  
  // return if we are interested in the given pick mode or not
  virtual int want_pick_mode(int);
  
  // return our list of draw commands with picking draw commands in them
  virtual void *pick_cmd_list(void);
  
  // return name of pickable
  virtual char *pick_name(void);

  // return whether the pickable object is being displayed
  virtual int pickable_on(void);

  //
  // public virtual pick action routines
  //

  // called when a pick is begun:
  //	args = display to use, obj picked, button, mode, tag, dim, pos
  // For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
  // For 3D version: x,y,z are transformed position of pointer
  virtual void pick_start(DisplayDevice *, Pickable *,
  				int, int, int, int, float *);
  
  // called when a pick moves:
  //	args = display to use, obj picked, button, mode, tag, dim, pos
  // For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
  // For 3D version: x,y,z are transformed position of pointer
  virtual void pick_move(DisplayDevice *, Pickable *,
  				int, int, int, int, float *);
  
  // called when a pick ends:
  //	args = display to use, obj picked, button, mode, tag, dim, pos
  // For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
  // For 3D version: x,y,z are transformed position of pointer
  virtual void pick_end(DisplayDevice *, Pickable *,
  				int, int, int, int, float *);

};

#endif
