/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: PickList.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 95/03/24 18:51:05 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The PickList object, which maintains a list of Pickable objects and
 * has the ability to find and deal with items picked by a pointer device.
 *
 * Each Scene is derived from PickList.
 *
 * Each PickList has a list of possible picking modes; these are created by
 * the request of Pickable objects, which then get notified when an
 * object is picked.  Only Pickables which are interested in the specified
 * mode are told of the picking result.
 *
 * For each pick mode, there is a corresponding PickMode object; PickMode
 * is a base class for the particular type of picking mode requested.
 * Pickable objects may request to have a new pick mode added to this list,
 * and if they do so they must provide an instance of the PickMode object
 * they require.  If the mode exists, the objects may replace the mode or
 * may just use the current one.  When modes are removed, the instance is
 * deleted.
 *
 * Only one picking operation may be in effect at a time.  A picking operation
 * consists of these steps:
 *	1. A pick is started by a pointer, by queueing a CmdPickStart command.
 *	2. pick_start is called, which determines if something is selected.
 *		If so, returns tag of item, and sets internal flags to
 *		indicate what is being picked, and how.
 *		2.1 Each Pickable which is interested in the current mode is
 *			told when the pick starts, and all later steps.
 *		2.2 The object for the current pick mode is also told of
 *			the start of the pick.
 *	3. pick_move is called whenever the pointer moves, by queueing the
 *		CmdPickMove command.  This continues until the picking is
 *		finished.  The PickMode object and interested Pickables are
 *		also told of the move.
 *	4. pick_end is called by queueing CmdPickEnd; this behaves similarly
 *		the pick_move.  When finished, the internal flags are reset
 *		to indicate that picking is finished, and a new one can begin.
 * NOTE: multiple concurrent picks could be implemented later by providing
 * an object which maintains separate sets of the internal flags and variables,
 * for each picking device.  This would also require another argument to the
 * pick_* routines, to indicate which picking device is being used.  Also,
 * PickMode and Pickable object involved in another picking operation would
 * have to be excluded from being informed of other picking operations.
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	PickList.C,v $
 * Revision 1.3  95/03/24  18:51:05  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.2  1995/03/17  22:27:52  billh
 * Now calls pickable_on when checking for a picked item; items which are not
 * on are not checked if they can be picked.
 *
 * Revision 1.1  1995/02/22  03:56:04  billh
 * Initial revision
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /private/auto143000131/vmdsrc/vmd/billh/src/RCS/PickList.C,v 1.3 95/03/24 18:51:05 billh Exp $";
#endif

#include "PickList.h"
#include "DisplayDevice.h"
#include "Pickable.h"
#include "PickModeQuery.h"
#include "ResizeArray.h"
#include "NameList.h"
#include "Inform.h"

//////////////////////////// constructor  ////////////////////////////
PickList::PickList(void) : pickableObjs(32), pickableInterested(32) {

  // reset all current picking flags
  currPickMode = currPickDim = currPickTag = (-1);
  currPickModePtr = NULL;
  currPickable = NULL;

  // add as a default a PickModeQuery pick mode
  add_pick_mode("Query", new PickModeQuery);
}


// destructor: delete all PickMode objects currently in list
PickList::~PickList(void) {
  register int i, nm = num_pick_modes();
  for(i=0; i < nm; i++)
    delete (pick_mode(i));
}


//////////////////////////// public routines  ////////////////////////////

// adds a Pickable to the current list, if it is not already in the list
// Return TRUE if the item is being added for the first time.
int PickList::add_pickable(Pickable *p) {

  // find if this is in the list already.
  int indx = pickableObjs.find(p);
  
  // if it is not, append it
  if(indx < 0) {
    indx = pickableObjs.append(p);
    pickableInterested[indx] = FALSE;
  }

  // return the index of the pickable object
  return indx;
}


// remove the given pickable from the list; return TRUE if it was in the list
int PickList::remove_pickable(Pickable *p) {

  // find where this object is
  int indx = pickableObjs.find(p);

  // if it is in the list, remove it
  if(indx >= 0) {
    pickableObjs.remove(indx);
    pickableInterested.remove(indx);
  }

  // return whether something was removed
  return (indx >= 0);
}


// add a new pick mode, with the given name.  Return index of this mode.
// If a mode of the name exists already, delete the old one and replace it
// with the new one provided.
int PickList::add_pick_mode(char *name, PickMode *pm) {

  // find if this pick mode already exists
  int indx = pickModes.typecode(name);
  
  // if it does, delete the pick mode there and put in new one
  if(indx >= 0) {
    delete (pick_mode(indx));
    pickModes.set_data(indx, pm);
  } else {
    indx = pickModes.add_name(name, pm);
  }
  
  // return the index of this name/pick mode pair
  return indx;
}


/////////////////////////////////////////////////////////////////////
// routines to handle starting a pick, moving during a pick,
// ending of a pick
/////////////////////////////////////////////////////////////////////

// using the given display, this checks to see if any object is under
// the given pointer position.  This does not set any internal picking
// flags, it just checks to see if something has been picked.  Returns
// a pointer to the Pickable object selected if successful, or NULL if
// nothing is picked.  If successful, returns 'tag' of item in final 
// argument.
// arguments: display device, dim, position, returned tag
Pickable *PickList::pick_check(DisplayDevice *d, int dim, float *pos, int &tag){
  Pickable *currObj, *retobj = NULL;
  float eyedist = (-1.0);
  int currtag, rettag = (-1);
  int i, np = num_pickable();

  if(!d || !np)
    return NULL;

  // use left eye settings for picking; if not stereo, will just be normal
  d->left();
      
  // for all Pickable objects, check to see if they have a picked object
  for(i=0; i < np; i++) {
    currObj = pickable(i);
    if(currObj->pickable_on()) {
      currtag = d->pick(dim, pos, currObj->pick_cmd_list(), eyedist);
      if(currtag >= 0) {
        // a new object closer to the eye position was found.  Save it.
        retobj = currObj;
        rettag = currtag;
      }
    }
  }
  
  // clean up after setting stereo mode, but do not do buffer swap
  d->update(FALSE);
      
  // for now, only check left eye.  Can later see if checking right eye helps
  // as well.
  
  // finished; report results
  if(retobj)
    tag = rettag;
  return retobj;
}


// called when a pick is begun: display device, button, mode, dim, pos
// returns 'tag' of closest object, or (-1) if nothing is picked.
// When a pick is started, the internal flags for this object are set,
// and no other item can be picked until pick_end is called.
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
int PickList::pick_start(DisplayDevice *d, int b, int m, int dim, float *pos) {
  Pickable *p, *closePickable;
  int i, tag = (-1), np = num_pickable();

  // make sure we're not already picking something
  if(picking())
    return (-1);

  // check if something has been actually picked
  if((closePickable = pick_check(d, dim, pos, tag)) != NULL) {

    // set all variables to show that we're picking something
    currPickMode = m;
    currPickDim = dim;
    currPickTag = tag;
    currPickModePtr = pick_mode(m);
    currPickable = closePickable;
    
    // use left eye settings for picking; if not stereo, will just be normal
    d->left();
      
    // find which pickables are interested, and let them know about the pick
    for(i=0; i < np; i++) {
      p = pickable(i);
      if((pickableInterested[i] = p->want_pick_mode(m)) == TRUE)
        p->pick_start(d, currPickable, b, m, currPickTag, dim, pos);
    }

    // let the current pick mode object know as well
    currPickModePtr->pick_start(d, currPickable, b, currPickTag, dim, pos);

    // clean up after setting stereo mode, but do not do buffer swap
    d->update(FALSE);   
  }

  return tag;
}


// called when a pick moves: display device, button, mode, dim, pos
// Returns TRUE if a pick is currently active, FALSE otherwise.
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
int PickList::pick_move(DisplayDevice *d, int b, int m, int dim, float *pos) {
  int i, np = num_pickable();

  // make sure we're already picking something
  if(!picking() || currPickMode != m || currPickDim != dim)
    return FALSE;

  // use left eye settings for picking; if not stereo, will just be normal
  d->left();
      
  // for all interested pickables, notify them of the move
  for(i=0; i < np; i++) {
    if(pickableInterested[i])
      pickable(i) -> pick_move(d, currPickable, b, m, currPickTag, dim, pos);
  }
  
  // let the current pick mode object know as well
  currPickModePtr->pick_move(d, currPickable, b, currPickTag, dim, pos);

  // clean up after setting stereo mode, but do not do buffer swap
  d->update(FALSE);
      
  return TRUE;
}


// called when a pick ends: display device, button, mode, dim, pos
// Returns TRUE if a pick is currently active, FALSE otherwise.
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
int PickList::pick_end(DisplayDevice *d, int b, int m, int dim, float *pos) {
  int i, np = num_pickable();

  // make sure we're already picking something
  if(!picking() || currPickMode != m || currPickDim != dim)
    return FALSE;

  // use left eye settings for picking; if not stereo, will just be normal
  d->left();
      
  // for all interested pickables, notify them the picking is finished
  for(i=0; i < np; i++) {
    if(pickableInterested[i]) {
      pickable(i) -> pick_end(d, currPickable, b, m, currPickTag, dim, pos);
    }
  }

  // let the current pick mode object know as well
  currPickModePtr->pick_end(d, currPickable, b, currPickTag, dim, pos);

  // clean up after setting stereo mode, but do not do buffer swap
  d->update(FALSE);
      
  // reset all current status variables, and return
  currPickMode = currPickDim = currPickTag = (-1);
  currPickModePtr = NULL;
  currPickable = NULL;
  
  return TRUE;
}
