/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Mouse.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.8 $	$Date: 95/03/24 18:50:57 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Mouse UI object, which maintains the current state of the 
 * mouse, including what it is currently used for, and how much it has moved
 * from one measurement to the next.  This also deals with any pop-up or
 * pull-down menus available by using the mouse, as well as picking objects.
 *
 * A three-button mouse is assumed here, with the following usage:
 *	1) Buttons 1 and 2 : manipulation and picking.
 *	2) Button 3 (right): pop-up menu
 *
 * This is the general base class definition; specific versions for each
 * windowing/graphics system must be supplied.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	Mouse.C,v $
 * Revision 1.8  95/03/24  18:50:57  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.7  1995/02/22  03:57:58  billh
 * Now supports picking objects with a 2D or 3D pointer.
 *
 * Revision 1.6  94/10/28  18:33:29  billh
 * Converted to a general base class (non-GL-specific) which can be used
 * throughout the program.  A GL-specific Mouse is in GLMouse.h
 * 
 * Revision 1.5  1994/10/19  20:22:20  billh
 * Added pop-up menu commands to turn forms on and off.
 *
 * Revision 1.4  1994/10/05  05:33:01  billh
 * arguments changed to allow to compile on HP's.
 *
 * Revision 1.3  1994/09/12  20:53:07  billh
 * Added 'view reset' command.
 *
 * Revision 1.2  94/08/26  00:01:05  billh
 * Improved pop-up menu; now produces submenus to attach mouse to lights,
 * set the stereo mode, toggle lights on/off, and a couple other utilities.
 * 
 * Revision 1.1  1994/08/24  03:10:37  billh
 * Initial revision
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /private/auto143000131/vmdsrc/vmd/billh/src/RCS/Mouse.C,v 1.8 95/03/24 18:50:57 billh Exp $";
#endif

#include "Mouse.h"
#include "Inform.h"
#include "CmdDisplay.h"

// string names for the move modes
static char *mm_names[5] = {"Rotate","Translate","Scale","Lights","Pick"};


// constructor
Mouse::Mouse(UIList *uil, CommandQueue *cq) : UIObject("Mouse", uil, cq) {

  MSGDEBUG(1, "Creating Mouse ..." << sendmsg);

  // set the default translation and rotation increments
  transInc = 0.002;
  rotInc = 1.0/15.0;
  scaleInc = 0.0002;
  RotVelScale = 0.4;
  b1Moved = b2Moved = TRUE;
  b1Down = b2Down = FALSE;
  currX = currY = oldX = oldY = 0;
  stop_rotation();
  moveObj = 0;
  moveMode = ROTATION;
  activePickMode = (-1);
  pickInProgress = FALSE;
}


// destructor
Mouse::~Mouse(void) {
  move_mode(ROTATION);
}


// return whether the given button is down (TRUE) or up (FALSE)
int Mouse::button_down(int) {
  // by default, act as if there are no buttons
  return FALSE;
}


// set the mouse move mode to the given state; return success
int Mouse::move_mode(MoveMode mm, int mobj) {

  MSGDEBUG(2,"Mouse: setting mouse mode to " << mm << ", mobj " << mobj);
  MSGDEBUG(2,sendmsg);

  // we cannot change the picking mode if an active pick is going on
  if(picking())
    return FALSE;

  // set active picking mode if necessary
  if(mm == PICKING) {
    // if we are changing to a picking mode, mobj is the new mode to use
    if(!pick_mode(mobj))
      return FALSE;
  } else {
    // not a picking mode; reset setting
    activePickMode = (-1);
  }

  // stop rotating no matter what move we've changed to
  stop_rotation();

  // signal we are/are not moving a light around
  if(moveMode == LIGHT) {
    // turn off old light
    addcommand(new CmdDisplayLightHL(moveObj, FALSE, id()));
  }

  // special actions based on the new mode
  if(mm == LIGHT) {
    // turn on new light number mobj
    moveObj = mobj;
    addcommand(new CmdDisplayLightHL(moveObj, TRUE, id()));
  }

  // change the mouse mode now
  moveMode = mm;
  return TRUE;
}

  
// set the active pick mode for this mouse
// (does things like change cursor style)
// return success.
int Mouse::pick_mode(int n) {

  MSGDEBUG(2,"Mouse: setting pick mode to " << n << sendmsg);
  
  // change the setting for the pick mode, if not currently picking something
  if(picking())
    return FALSE;

  activePickMode = n;
  return TRUE;
}
  

// string name for the given move mode
char *Mouse::move_mode_name(MoveMode mm) {
  return mm_names[mm];
}


// stop rotation of object
void Mouse::stop_rotation(void) {
  xRotVel = yRotVel = zRotVel = 0.0;
}



/////////////////////// virtual routines for UI init/display  /////////////
   
// reset the mouse to original settings
void Mouse::reset(void) {
  scaling = 1.0;
  stop_rotation();
  b1Down = button_down(B_LEFT);
  b2Down = button_down(B_MIDDLE);
  b1Moved = b2Moved = FALSE;
  currX = oldX = x();
  currY = oldY = y();
}
