/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Matrix4.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.6 $	$Date: 95/03/24 18:50:27 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * 4 x 4 Matrix, used for a transformation matrix.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	Matrix4.h,v $
 * Revision 1.6  95/03/24  18:50:27  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.5  1995/02/22  04:07:51  billh
 * Added new constructor for just a float * argument - takes the given
 * single-dim array of floats, assumed to be of proper size, and copies
 * it into internal matrix storage as 4x4 matrix.
 *
 * Revision 1.4  1994/12/09  01:20:49  khamer
 * Added inverse.
 *
 * Revision 1.3  1994/10/28  18:32:04  billh
 * Only includes GL library for Matrix def if using VMDFORMS option.
 * Otherwise, provides own typedef.
 *
 * Revision 1.2  1994/10/04  20:29:22  billh
 * Changes to get to compile on HP's ... not there yet,though.
 *
 * Revision 1.1  1994/08/24  03:10:37  billh
 * Initial revision
 *
 ***************************************************************************/

#ifndef MATRIX_FOUR_H
#define MATRIX_FOUR_H

#include <iostream.h>
#include "Inform.h"

// define Matrix type, either via the GL version or our own
#ifdef VMDFORMS
#include <gl/gl.h>
#else
  typedef float Matrix[4][4];
#endif

// local defines
#ifndef MYPI
#define MYPI		3.141592654
#endif

// Degree-to-Radians and Radians-to-Degrees Conversion macros
#define DEGTORAD(a)	(a*MYPI/180.0)
#define RADTODEG(a)	(a*180.0/MYPI)

// dimension of this matrix
#define MATRIX_DIM	4


class Matrix4 {

public:	
  // the matrix itself (this is public so it can be accessed for speed)
  Matrix mat;

public:
  Matrix4(void) { identity(); }
  Matrix4(float f) { constant(f); }
  Matrix4(float *m);
  Matrix4(const Matrix4& m) { loadmatrix(m); }
  ~Matrix4(void) { }

  // print this matrix to the given ostream
  friend ostream& operator<<(ostream &, Matrix4 &);
  friend Inform& operator<<(Inform &, Matrix4 &);

  // multiplies a 3D point (first arg) by the Matrix, returns in second arg
  Matrix4& multpoint3d(float[3], float[3]);

  // multiplies a 4D point (first arg) by the Matrix, returns in second arg
  Matrix4& multpoint4d(float[4], float[4]);

  // clears the matrix (resets it to identity)
  Matrix4& identity(void);
  
  // sets the matrix so all items are the given constant value
  Matrix4& constant(float);
  
  // return the inverse of the matrix, that is, 
  // the inverse of the rotation, the inverse of the scaling, and 
  // the opposite of the translation vector.
  Matrix4& inverse(const Matrix4 &);
  Matrix4& inverse(void) { 
    Matrix4 temp(*this);
    return inverse(temp);
  }
  
  // replaces this matrix with the given one
  Matrix4& loadmatrix(const Matrix4 &);
  Matrix4& operator=(const Matrix4 &m) { return loadmatrix(m); }

  // premultiply the matrix by the given matrix
  Matrix4& multmatrix(const Matrix4 &);
  Matrix4& operator*=(const Matrix4 &m) { return multmatrix(m); }
  friend Matrix4 operator*(const Matrix4 &m1, const Matrix4 &m2) {
    Matrix4 m(m1); m.multmatrix(m2); return m;
  }

  // adds the given matrix to this matrix
  Matrix4& addmatrix(const Matrix4 &);
  Matrix4& operator+=(const Matrix4 &m) { return addmatrix(m); }

  // adds the given constant to this matrix
  Matrix4& addconst(float);
  Matrix4& operator+=(float f) { return addconst(f); }

  // performs a rotation around an axis (char == 'x', 'y', or 'z')
  Matrix4& rotate(short a, char axis) {	// angle in tenths of degrees
    return rot((float)a / 10.0, axis);
  }
  Matrix4& rot(float, char);		// angle in degrees

  // performs a translation
  Matrix4& translate(float, float, float);

  // performs scaling
  Matrix4& scale(float, float, float);
  Matrix4& scale(float f) { return scale(f, f, f); }
  Matrix4& operator*=(float f) { return scale(f, f, f); }

  // sets this matrix to represent a perspective
  Matrix4& perspective(short, float, float, float);

  // sets this matrix to represent a window perspective
  Matrix4& window(float, float, float, float, float, float);

  // sets this matrix to a 3D orthographic matrix
  Matrix4& ortho(float, float, float, float, float, float);

  // sets this matrix to a 2D orthographic matrix
  Matrix4& ortho2(float, float, float, float);

  /* This subroutine defines a viewing transformation with the eye located in
   * polar coordinates looking at the world origin with twist rotation about
   * the line of site.  Precisely, polarview does:
   * polarview = rot(-azim,z)*rot(-inc,x)*rot(-twist,z)*trans(0.0,0.0,-dist)
   */
  Matrix4& polarview(float, short, short, short);

  /* This subroutine defines a viewing transformation with the eye at the point
   * (vx,vy,vz) looking at the point (px,py,pz).  Twist is the right-hand
   * rotation about this line.  The resultant matrix is multiplied with
   * the top of the transformation stack and then replaces it.  Precisely,
   * lookat does:
   * lookat = trans(-vx,-vy,-vz)*rotate(theta,y)*rotate(phi,x)*rotate(-twist,z)
   */
  Matrix4& lookat(float, float, float, float, float, float, short);

};

#endif
