/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Light.C,v $
 *	$Author: dalke $	$Locker: billh $		$State: Exp $
 *	$Revision: 1.6 $	$Date: 1994/11/09 06:34:32 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A Displayable3D object which represents a light source at infinity, along
 * some line from the origin, of some color.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: Light.C,v $
 * Revision 1.6  1994/11/09  06:34:32  dalke
 * Now uses 'LOAD' instead of 'NONE' for matrix method.
 *
 * Revision 1.5  1994/11/02  01:35:38  billh
 * Highlight drawn with dotted line of width 3.
 *
 * Revision 1.4  94/10/01  03:03:59  billh
 * Removed 'update' routine.
 * 
 * Revision 1.3  94/09/24  20:09:28  billh
 * Now contain single display commands which are used many times,
 * instead of re-creating and destroying disp command objects.
 * 
 * Revision 1.2  1994/09/12  20:51:43  billh
 * Changed due to new Displayable constructor arguments.
 *
 * Revision 1.1  94/08/24  03:10:37  billh
 * Initial revision
 * 
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /Net/cairo/Users/billh/vmd_new/src/RCS/Light.C,v 1.6 1994/11/09 06:34:32 dalke Exp billh $";
#endif

#include <stdio.h>
#include "Light.h"
#include "Inform.h"
#include "ColorList.h"
#include "Displayable.h"
#include "DispCmds.h"
#include "DisplayDevice.h"

// color for the light if we are using a ColorList
#define DEFLIGHTCOL	REGWHITE


// constructor
Light::Light(Scene *sc, int ln, float *c, float *p)
	: Displayable3D(LOAD, "Light", sc, 0), htxt(" ") {
  // initialize variables
  lightOn = FALSE;
  need_define = need_onoff = TRUE;
  need_highlight = highlightOn = FALSE;
  needDefineCleanup = FALSE;
  colorCat = colorIndex = (-1);

  // set number of this light
  if(ln < 0 || ln >= DISP_LIGHTS)
    num = 0;
  else
    num = ln;
  
  // store color and position
  for(int i=0; i < 3; i++) {
    color[i] = c[i];
    pos[i] = p[i];
  }

  // create command to load identity matrix on dispay's stack
  loadIdentMat = new DispCmdLoad(&ident);

  // Displayable characteristics
  rot_off();
  scale_off();
  glob_trans_off();
  cent_trans_off();
}


/////////////////////////  protected virtual routines /////////////////

// virtual function which is called when a new colorlist is provided.
// This creates or gets the category # for a Light color category
void Light::do_use_colors(void) {
  char cname[4];

  // create (or possibly just the the # for) a color category for lights
  colorCat = colorList->add_color_category("Light");

  // add a color object to this category for our own light
  sprintf(cname,"%d",num);
  colorIndex=(colorList->color_category(colorCat))->add_name(cname,DEFLIGHTCOL);

  msgInfo << "New light color category " << colorCat << " for color ";
  msgInfo << cname << ", index = " << colorIndex << sendmsg;

  // indicate we need to recreate the command list
  need_define = TRUE;
}


// do action due to the fact that a color for the given ColorList for
// the specified category has changed
void Light::do_color_changed(ColorList *changelist, int clr) {
  if(changelist == colorList && clr == colorCat) {
    // color changed for us, recreate command list
    need_define = TRUE;

    msgInfo << "Colors updated for light " << num << sendmsg;
  }
}

//////////////////////////  public routines  ////////////////////////////

// extra action added to Displayable's rotating commands, 
// to signal a redef of the
// light is needed, and to update the rotation matrix
void Light::add_rotation(float x, char axis) {
  need_define = TRUE;
  rot_on();
  add_rot(x,axis);
  rot_off();
}

void Light::add_rotation(Matrix4 &m) {
  need_define = TRUE;
  rot_on();
  add_rot(m);
  rot_off();
}

void Light::set_rotation(Matrix4 &m) {
  need_define = TRUE;
  rot_on();
  set_rot(m);
  rot_off();
}


// origin position
static float lightOrigin[3] = { 0.0, 0.0, 0.0 };


// create the command list
void Light::create_cmdlist(void) {
  float *usecolor;

  // do reset first
  reset_disp_list();

  // see if we need to define the light source
  if(need_define)
    // put in define light command
    if(colorCat < 0)
      usecolor = color;
    else {
      usecolor = colorList->color(
      	(colorList->color_category(colorCat))->data(colorIndex));
      msgInfo << "Color " << num << " (index " << colorIndex;
      msgInfo << ") creating cmdlist with color ";
      msgInfo << usecolor[0] << ", " << usecolor[1] << ", " << usecolor[2];
      msgInfo << sendmsg;
    }
    deflight.putdata(num, usecolor, pos, this);

  // see if we need to turn the light on or off
  if(need_onoff)
    // put in on/off command
    onofflight.putdata(num, lightOn, this);

  // draw line from origin to light if we are highlighting
  if(highlightOn) {
    cmdLineType.putdata(DASHEDLINE, this);
    cmdLineWidth.putdata(3, this);
    hcol.putdata(color, this);
    hline.putdata(lightOrigin, pos, this);
    htxtpos.putdata(pos, this);
    char txt[2];
    txt[0] = '0' + num;
    txt[1] = '\0';
    htxt.putdata(txt, this);
  }
  
  // set drawing flags
  needDefineCleanup = (need_define || need_onoff || need_highlight);
  need_define = need_onoff = need_highlight = FALSE;
}


// prepare the light for display ... regenerate command list if necessary
void Light::prepare(DisplayDevice *) {
  // if a flag has been set, change the settings
  if(need_define || need_onoff || need_highlight || needDefineCleanup)
    create_cmdlist();
}
