/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: GraphicsFormsObj.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.11 $	$Date: 1995/03/28 03:44:58 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The on-screen menu to control the coloring and display method of molecules.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: GraphicsFormsObj.C,v $
 * Revision 1.11  1995/03/28  03:44:58  billh
 * When the top molecule is changed, the menu resets to make that molecule
 * the one being edited.
 *
 * Revision 1.10  1995/03/24  18:50:08  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 *
 * Revision 1.9  1994/12/11  01:17:35  dalke
 * Added ribbons option
 *
 * Revision 1.8  1994/12/05  08:24:43  billh
 * Added multiple modes for changing reps; now includes second mode to
 * use the mouse to select atoms, with a browser for looking at the molecule
 * names.
 *
 * Revision 1.7  94/11/03  01:03:05  billh
 * Changed text and enum name for 'off' option in atom representation.
 * 
 * Revision 1.6  1994/11/02  23:32:44  billh
 * Added 'none' option to AtomRep to quickly turn off graphics reps
 *
 * Revision 1.5  1994/11/02  01:35:25  billh
 * Fixed numerous small bugs.
 *
 * Revision 1.4  1994/10/31  23:51:41  billh
 * Fixed some bugs.
 *
 * Revision 1.3  1994/10/31  20:32:01  billh
 * Fixed many bugs, set to always modify when something changes.
 *
 * Revision 1.2  1994/10/29  02:23:52  billh
 * Added code to generate commands to modify, add, or delete reps.
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /tmp_mnt/Home/h2/billh/projects/vmd/src/RCS/GraphicsFormsObj.C,v 1.11 1995/03/28 03:44:58 billh Exp $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include "GraphicsFormsObj.h"
#include "CmdMol.h"
#include "MoleculeList.h"
#include "Molecule.h"
#include "NameList.h"
#include "Global.h"
#include "AtomColor.h"
#include "AtomSel.h"
#include "AtomRep.h"

// commands we are interested in
static int numCmds = 8;
static int cmdList[8] = { Command::MOL_NEW,	Command::MOL_DEL,
	Command::MOL_ADDREP,		Command::MOL_MODREP,
	Command::MOL_DELREP,		Command::MOL_MODREPITEM,
	Command::REMOTE_ATTACH,		Command::REMOTE_RUN };

// type of editing the menu is doing currently
#define GREDIT_UNKNOWN		0
#define GREDIT_COLORDRAW	1
#define GREDIT_ATOMSEL		2
#define GREDIT_DEFAULT		GREDIT_COLORDRAW

// different operations, and the string names to use if they are to be
// inserted in the selection either at the beginning of an operation, or 
// in the middle of an operation
#define ATOMSEL_OP_AND		0
#define	ATOMSEL_OP_OR		1
#define ATOMSEL_OP_ANDNOT	2
#define ATOMSEL_OP_ORNOT	3
#define ATOMSEL_OP_ANDBLOCK	4
#define	ATOMSEL_OP_ORBLOCK	5
#define ATOMSEL_OP_ANDNOTBLOCK	6
#define ATOMSEL_OP_ORNOTBLOCK	7
#define ATOMSEL_OP_AMOUNT	8
static char *atomselOpStr[ATOMSEL_OP_AMOUNT] = { 
	"and",   "or",   "and not",   "or not",
	"and (", "or (", "and not (", "or not ("   };
static char *atomselOpStrBeg[ATOMSEL_OP_AMOUNT] = {
	"",  "",  "not",   "not",
	"(", "(", "not (", "not (" };

// collection of objects which are used for the different editing modes
static FL_OBJECT **colordraw_objs[] = { &color_method_box, &draw_method_box,
	&graphics_color, &graphics_cindex, &graphics_trans,
	&graphics_drawstyle, &graphics_bond_rad, &graphics_bond_res,
	&graphics_sphere_rad, &graphics_sphere_res, &graphics_line_thick };

static FL_OBJECT **atomsel_objs[] = { &atomsel_method_box,
	&atomsel_operator, &atomsel_end_block,
	&atomsel_clear_sel, &atomsel_enter_sel,
	&atomsel_which_list, &atomsel_namelist };


/////////////////////////  constructor  
GraphicsFormsObj::GraphicsFormsObj(UIList *uil, CommandQueue *cq,
	int sh,int bd,int *npos) : FormsObj("graphics", uil, cq, bd, npos) {
	
  // set values of common menu elements
  form = graphicsMenu;
  lightButton = graphics_menu_button;
  offButton = graphics_menu_off;
  
  // register which commands we are interested in
  for(int i=0; i < numCmds; command_wanted(cmdList[i++]));
  
  // turn on if required
  if(sh)
    On();

  // indicate we do not yet have a selected molecule
  molID = (-1);
  molRep = (-1);
  dmol = NULL;
  
  // set the current editing mode
  editMode = GREDIT_UNKNOWN;
}


/////////////////////////  private routines  

// fill the given string buffer with a string describing the mth rep of
// the current mol.  Returns a pointer to the string.  If error, string is
// returned empty.
char *GraphicsFormsObj::graphics_rep_string(int m, char *str) {
  DrawMolItem *component;

  // check if no rep available
  if(!dmol || !(component = dmol->component(m))) {
    str[0] = '\0';
  } else {
    // component is the item to get data from
    strncpy(str, (component->atomRep)->cmdStr, 11);
    str[11] = '\0';
    while(strlen(str) < 12)
      strcat(str, " ");
    strncat(str, (component->atomColor)->cmdStr, 11);
    str[23] = '\0';
    while(strlen(str) < 24)
      strcat(str, " ");
    strncat(str, (component->atomSel)->cmdStr, 20);
    str[44] = '\0';
  }
  
  return str;
}


// set the graphics editing mode to the given value
void GraphicsFormsObj::set_editing_mode(int newMode) {

  if(newMode == editMode)
    return;
  editMode = newMode;

  freeze();
    // turn off objects first
    if(newMode != GREDIT_COLORDRAW) {
      int amt = sizeof(colordraw_objs) / sizeof(FL_OBJECT **);
      for(int i=0; i < amt; i++)
        fl_hide_object( *(colordraw_objs[i]) );
    }
    if(newMode != GREDIT_ATOMSEL) {
      int amt = sizeof(atomsel_objs) / sizeof(FL_OBJECT **);
      for(int i=0; i < amt; i++)
        fl_hide_object( *(atomsel_objs[i]) );
    }
    
    // turn on objects last
    if(newMode == GREDIT_COLORDRAW) {
      fl_set_button(graphics_mode_colordraw, TRUE);
      int amt = sizeof(colordraw_objs) / sizeof(FL_OBJECT **);
      for(int i=0; i < amt; i++)
        fl_show_object( *(colordraw_objs[i]) );
      reset_coloring_method(FALSE);
      reset_drawing_method(FALSE);
      reset_selection_method(FALSE);
    } else if(newMode == GREDIT_ATOMSEL) {
      fl_set_button(graphics_mode_atomsel, TRUE);
      int amt = sizeof(atomsel_objs) / sizeof(FL_OBJECT **);
      for(int i=0; i < amt; i++)
        fl_show_object( *(atomsel_objs[i]) );
    }
  unfreeze();
}


// fill the namelist browser for atom selection with the proper names
void GraphicsFormsObj::reset_namelist_browser(void) {
  freeze();
    // clear the name browser
    fl_clear_browser(atomsel_namelist);

    // get current name list selection, and check for a category with names
    int list = fl_get_browser(atomsel_which_list) - 1;
  
    // only if a list has been selected, and is useful one, do we enter names
    // also, only need to fill browser if there is a molecule currently
    if(dmol && list >= 0) {
      NameList<int> *namelist = NULL;
      if(list == AtomSel::NAME)
        namelist = & (dmol->atomNames);
      else if(list == AtomSel::TYPE)
        namelist = & (dmol->atomTypes);
      else if(list == AtomSel::RESNAME)
        namelist = & (dmol->resNames);
      else if(list == AtomSel::RESID)
        namelist = & (dmol->resIds);
      else if(list == AtomSel::SEGNAME)
        namelist = & (dmol->segNames);
      if(namelist) {
        for(int i=0; i < namelist->num(); i++)
	  fl_add_browser_line(atomsel_namelist, namelist->sort_name(i));
      }
    }
  unfreeze();
}


// fill the graphics rep browser with the settings for the current mol
void GraphicsFormsObj::fill_graphics_reps_browser(void) {
  freeze();
    char strbuf[128];
    int N = (dmol ? dmol->components() : 0);
    fl_clear_browser(graphics_reps);
    if(N > 0) {
      for(int i=0; i < N; i++)
        fl_add_browser_line(graphics_reps, graphics_rep_string(i, strbuf));
	
      // by default, select the first item if no other option, else keep same
      // selection
      if(molRep < 0 || molRep >= N)
        molRep = 0;
      fl_select_browser_line(graphics_reps, molRep + 1);
    } else {
      molRep = (-1);
    }
    reset_edited_rep();
  unfreeze();
}


// reset the form to use the molecule with given ID as the current mol
void GraphicsFormsObj::reset_edited_molecule(int n) {
  freeze();
    fl_clear_choice(graphics_mol);

    if(moleculeList->num() < 1) {
      // no molecules; reset everything
      molID = (-1);
      dmol = NULL;
    } else {
      // select a molecule (possibly the same one as previously chosen)
      int newline = (-1), newID = (-1), topline = (-1), oldline = (-1);
      for(int i=0; i < moleculeList->num(); i++) {
        Molecule *m = moleculeList->molecule(i);
        fl_addto_choice(graphics_mol, m->name);
        if(m->id() == n)
          newline = i;
	if(m->id() == molID)
	  oldline = i;
	if(moleculeList->is_top(m))
	  topline = i;
      }
      if(newline < 0) {
        newline = (topline < 0 ? oldline : topline);
        if(newline < 0)		// STILL no new molecule chosen, so
	  newline = 0;		// just take the first one
      }

      fl_set_choice(graphics_mol, newline + 1);
      newID = (moleculeList->molecule(newline))->id();

      // a new molecule; reset other items
      molID = newID;
      dmol = moleculeList->molecule(newline);
    }

    // reset all the other browsers and options based on the new molecule
    molRep = (-1);			// no rep selected yet
    fill_graphics_reps_browser();	// put text in rep browser
    reset_namelist_browser();		// put text in namelist browser
  unfreeze();
}


// reset the form based on the current selections in the graphics rep browser
void GraphicsFormsObj::reset_edited_rep(void) {

  freeze();
    // scan through the rep browser until a line is found, and then
    // set the settings for that item.
    int N = (dmol ? dmol->components() : 0);
    for(int i=0; i < N; i++) {
      if(fl_isselected_browser_line(graphics_reps, i+1)) {
        molRep = i;
        break;
      }
    }
    
    // now reset all controls based on new molRep value
    reset_coloring_method(TRUE);
    reset_drawing_method(TRUE);
    reset_selection_method(TRUE);
  unfreeze();
}


// reset coloring method controls based on current rep, if any
void GraphicsFormsObj::reset_coloring_method(int setvals) {
  freeze();
    int newmethod = fl_get_choice(graphics_color) - 1;
    if(setvals && molRep >= 0) {
      AtomColor *ac = (dmol->component(molRep))->atomColor;
      newmethod = ac->method();
      fl_set_choice(graphics_color, newmethod + 1);
      fl_set_button(graphics_trans, ac->transparent());
      if(newmethod == AtomColor::INDEX)
        set_input_int(graphics_cindex, ac->color_index(), 3);
    }
    if(editMode == GREDIT_COLORDRAW) {
      if(newmethod == AtomColor::INDEX)
        fl_show_object(graphics_cindex);
      else
        fl_hide_object(graphics_cindex);
    }
  unfreeze();
}


// reset drawing method controls based on current rep, if any
void GraphicsFormsObj::reset_drawing_method(int setvals) {
  freeze();
    int newstyle = fl_get_choice(graphics_drawstyle) - 1;
    if(setvals && molRep >= 0) {
      AtomRep *ar = (dmol->component(molRep))->atomRep;
      newstyle = ar->method();
      fl_set_choice(graphics_drawstyle, newstyle + 1);

      if(newstyle == AtomRep::LINES) {
        fl_set_counter_value(graphics_line_thick, ar->line_thickness());
      } else if(newstyle == AtomRep::CPK) {
        fl_set_counter_value(graphics_bond_rad, ar->bond_rad());
        fl_set_counter_value(graphics_bond_res, ar->bond_res());
        fl_set_counter_value(graphics_sphere_rad, ar->sphere_rad());
        fl_set_counter_value(graphics_sphere_res, ar->sphere_res());
      } else if(newstyle == AtomRep::VDW || newstyle == AtomRep::DOTTED) {
        fl_set_counter_value(graphics_sphere_rad, ar->sphere_rad());
        fl_set_counter_value(graphics_sphere_res, ar->sphere_res());
      } else if(newstyle == AtomRep::LICORICE) {
        fl_set_counter_value(graphics_bond_rad, ar->bond_rad());
        fl_set_counter_value(graphics_bond_res, ar->bond_res());
        fl_set_counter_value(graphics_sphere_res, ar->sphere_res());
      } else if(newstyle == AtomRep::REPOFF ||
		 newstyle == AtomRep::RIBBONS) {
        // do nothing
      }
    }
    if(editMode == GREDIT_COLORDRAW) {
      if(newstyle == AtomRep::LINES) {
        fl_hide_object(graphics_bond_rad);
        fl_hide_object(graphics_bond_res);
        fl_hide_object(graphics_sphere_rad);
        fl_hide_object(graphics_sphere_res);
        fl_show_object(graphics_line_thick);
      } else if(newstyle == AtomRep::CPK) {
        fl_show_object(graphics_bond_rad);
        fl_show_object(graphics_bond_res);
        fl_show_object(graphics_sphere_rad);
        fl_show_object(graphics_sphere_res);
        fl_hide_object(graphics_line_thick);
      } else if(newstyle == AtomRep::VDW || newstyle == AtomRep::DOTTED) {
        fl_hide_object(graphics_bond_rad);
        fl_hide_object(graphics_bond_res);
        fl_show_object(graphics_sphere_rad);
        fl_show_object(graphics_sphere_res);
        fl_hide_object(graphics_line_thick);
      } else if(newstyle == AtomRep::LICORICE) {
        fl_show_object(graphics_bond_rad);
        fl_show_object(graphics_bond_res);
        fl_hide_object(graphics_sphere_rad);
        fl_show_object(graphics_sphere_res);
        fl_hide_object(graphics_line_thick);
      } else if(newstyle == AtomRep::REPOFF ||
		 newstyle == AtomRep::RIBBONS) {
        fl_hide_object(graphics_bond_rad);
        fl_hide_object(graphics_bond_res);
        fl_hide_object(graphics_sphere_rad);
        fl_hide_object(graphics_sphere_res);
        fl_hide_object(graphics_line_thick);
      }
    }
  unfreeze();
}


// reset atom selection controls based on current rep, if any
void GraphicsFormsObj::reset_selection_method(int setvals) {
  if(molRep < 0 || !setvals)
    return;

  freeze();
    AtomSel *as = (dmol->component(molRep))->atomSel;
    fl_set_input(graphics_sel, as->cmdStr);
  unfreeze();
}


// fill string with text command to set objects to proper color, rep, or sel.
// NOTE: this routine is NOT re-entrant.  But that should not matter here ...
char *GraphicsFormsObj::create_color_cmd(void) {
  static char colorStr[256];
  int colorVal = fl_get_choice(graphics_color) - 1;
  
  if(colorVal >= 0) {
    // create new color command string
    if(colorVal == AtomColor::INDEX)
      sprintf(colorStr, "%s %d", AtomColorName[colorVal],
	atoi(fl_get_input(graphics_cindex)));
    else
      strcpy(colorStr, AtomColorName[colorVal]);
    if(fl_get_button(graphics_trans))
      strcat(colorStr, " transparent");
  } else
    colorStr[0] = '\0';

  return colorStr;
}


// fill string with text command to set objects to proper color, rep, or sel.
// NOTE: this routine is NOT re-entrant.  But that should not matter here ...
char *GraphicsFormsObj::create_rep_cmd(void) {
  static char styleStr[256];
  int styleVal = fl_get_choice(graphics_drawstyle) - 1;

  if(styleVal >= 0) {
      // create new display method string
      if(styleVal == AtomRep::LINES) {
        sprintf(styleStr, "%s %d", AtomRepName[styleVal],
		(int)fl_get_counter_value(graphics_line_thick));

      } else if(styleVal == AtomRep::CPK) {
        sprintf(styleStr, "%s %f %f %d %d", AtomRepName[styleVal],
		fl_get_counter_value(graphics_sphere_rad),
		fl_get_counter_value(graphics_bond_rad),
		(int)fl_get_counter_value(graphics_sphere_res),
 		(int)fl_get_counter_value(graphics_bond_res));
 
      } else if(styleVal == AtomRep::VDW || styleVal == AtomRep::DOTTED) {
        sprintf(styleStr, "%s %f %d", AtomRepName[styleVal],
		fl_get_counter_value(graphics_sphere_rad),
		(int)fl_get_counter_value(graphics_sphere_res));

      } else if(styleVal == AtomRep::LICORICE) {
        sprintf(styleStr, "%s %f %d %d", AtomRepName[styleVal],
		fl_get_counter_value(graphics_bond_rad),
		(int)fl_get_counter_value(graphics_sphere_res),
 		(int)fl_get_counter_value(graphics_bond_res));
      } else if(styleVal == AtomRep::REPOFF ||
		 styleVal == AtomRep::RIBBONS ) {
        strcpy(styleStr, AtomRepName[styleVal]);
      } else
        styleStr[0] = '\0';
  } else
    styleStr[0] = '\0';

  return styleStr;
}


// fill string with text command to set objects to proper color, rep, or sel.
// NOTE: this routine is NOT re-entrant.  But that should not matter here ...
char *GraphicsFormsObj::create_sel_cmd(void) {
  static char selStr[256];
  char *selInput = fl_get_input(graphics_sel);
  
  if(selInput && strlen(selInput) > 0)
    return selInput;
  else
    return NULL;
}


//////////////////////  protected virtual routines  

// routine to check the main form for use
int GraphicsFormsObj::check(FL_OBJECT *obj) {
  char selbuf[256];

  if(obj == graphics_mol) {
    Molecule *mol = moleculeList->molecule(fl_get_choice(obj) - 1);
    if(mol)
      reset_edited_molecule(mol->id());
    else
      reset_edited_molecule(-1);

  } else if(obj == graphics_reps) {
    reset_edited_rep();

  } else if(obj == graphics_color || obj == graphics_cindex || 
  	obj == graphics_trans) {
    if(molRep < 0) {
      if(obj == graphics_color)
        reset_coloring_method(FALSE);
    } else {
      if(obj == graphics_color) {
        freeze();
          int colorVal = fl_get_choice(graphics_color) - 1;
          reset_coloring_method(TRUE);
          fl_set_choice(graphics_color, colorVal + 1);
          reset_coloring_method(FALSE);
        unfreeze();
      }

      // add command to just change the coloring of the current item
      addcommand(new CmdMolChangeRepItem(molRep, molID,
			CmdMolChangeRepItem::COLOR,create_color_cmd(),id()));
    }

  } else if(obj == graphics_drawstyle || obj == graphics_bond_rad ||
	obj == graphics_bond_res || obj == graphics_sphere_rad ||
	obj == graphics_sphere_res || obj == graphics_line_thick) {
    if(molRep < 0) {
      if(obj == graphics_drawstyle)
        reset_drawing_method(FALSE);
    } else {
      if(obj == graphics_drawstyle) {
        freeze();
          int styleVal = fl_get_choice(graphics_drawstyle) - 1;
          reset_drawing_method(TRUE);
          fl_set_choice(graphics_drawstyle, styleVal + 1);
          reset_drawing_method(FALSE);
        unfreeze();
      }

      // add command to just change the drawing method of the current item
      addcommand(new CmdMolChangeRepItem(molRep, molID,
			CmdMolChangeRepItem::REP, create_rep_cmd(), id()));
    }

  } else if(obj == graphics_sel || obj == atomsel_enter_sel) {
    if(molRep >= 0) {
      char *s = create_sel_cmd();
      if(s)
        addcommand(new CmdMolChangeRepItem(molRep, molID,
			CmdMolChangeRepItem::SEL, s, id()));
      else
        reset_selection_method(TRUE);
    }

  } else if(obj == graphics_add_active || obj == graphics_add_selected) {
    // put in commands to change current settings
    char *s = create_sel_cmd();
    if(s) {
      addcommand(new CmdMolColor(create_color_cmd(), id()));
      addcommand(new CmdMolRep(
		    AtomRepName[fl_get_choice(graphics_drawstyle)-1], id()));
      addcommand(new CmdMolSelect(s, id()));
      
      // put in command to add item
      addcommand(new CmdMolAddRep(
		   (obj == graphics_add_selected ? molID : (-1)), id()));
    } else
      reset_selection_method(TRUE);

  } else if(obj == graphics_delete) {
    if(molID >= 0 && molRep >= 0)
      addcommand(new CmdMolDeleteRep(molRep, molID, id()));

  } else if(obj == graphics_mode_colordraw) {
    set_editing_mode(GREDIT_COLORDRAW);
    
  } else if(obj == graphics_mode_atomsel) {
    set_editing_mode(GREDIT_ATOMSEL);

  } else if(obj == atomsel_end_block) {
    // only adds ) if ('s outnumber )'s
    char *oldsel = fl_get_input(graphics_sel);
    int no = 0, nc = 0;
    while(*oldsel != '\0') {
      if(*oldsel == '(')
        no++;
      else if(*oldsel == ')')
        nc++;
      oldsel++;
    }
    if(no > nc) {
      strcpy(selbuf, fl_get_input(graphics_sel));
      strcat(selbuf, ")");
      fl_set_input(graphics_sel, selbuf);
    }
    
  } else if(obj == atomsel_operator) {
    strcpy(selbuf, fl_get_input(graphics_sel));
    int atstart = TRUE, i = strlen(selbuf) - 1;
    char c;
    
    // first check if we're at the start of a new command, or appending to
    // a current one
    while(i >= 0) {
      c = selbuf[i];
      if(c != ' ' && c != '\t' && c != '(') {	// check for regular char
        atstart = FALSE;
	break;
      } else if(c == '(') {			// check for open paren
        break;
      }
      i--;
    }

    // append space, if necessary
    if(i >= 0)
      strcat(selbuf, " ");

    // append operator selection
    if(atstart)
      strcat(selbuf, atomselOpStrBeg[fl_get_browser(obj) - 1]);
    else
      strcat(selbuf, atomselOpStr[fl_get_browser(obj) - 1]);

    // see if a selection method has been set.  If so, append it
    int selmeth = fl_get_browser(atomsel_which_list);
    if(selmeth > 0) {
      // append another spac
      strcat(selbuf, " ");
      
      // append string
      strcat(selbuf, AtomSelName[selmeth - 1]);
    }
    
    // after all this, change the input string
    fl_set_input(graphics_sel, selbuf);

  } else if(obj == atomsel_namelist) {
    // append name to end of selection string
    strcpy(selbuf, fl_get_input(graphics_sel));
    strcat(selbuf, " ");
    strcat(selbuf, fl_get_browser_line(obj, fl_get_browser(obj)));
    fl_set_input(graphics_sel, selbuf);

  } else if(obj == atomsel_which_list) {
    // fill the namelist browser with the proper names
    reset_namelist_browser();

  } else if(obj == atomsel_clear_sel) {
    // just reset the current selection to be empty
    fl_set_input(graphics_sel, "");

  } else
    return FALSE;

  return TRUE;
}


// do form-specific acting on commands.  Return whether
// any action was taken on this command.
// Arguments are the command type, command object, and the 
// success of the command (T or F).
int GraphicsFormsObj::forms_act_on_command(int type, Command *, int) {

  MSGDEBUG(3,"GraphicsFormsObj: acting on command " << type << sendmsg);

  if(type == Command::MOL_NEW || type == Command::MOL_DEL ||
	type == Command::MOL_ADDREP || type == Command::MOL_MODREP ||
	type == Command::MOL_DELREP || 
	type == Command::REMOTE_ATTACH || type == Command::REMOTE_RUN) {
    freeze();
      // make the top molecule the active one here
      molID = (-1);
      reset();
      if(type == Command::MOL_ADDREP) {
        // select the last item in the representation browser, if any,
	// so that new reps are automatically selected
	int maxl = fl_get_browser_maxline(graphics_reps);
	if(maxl > 0) {
	  fl_deselect_browser(graphics_reps);
	  fl_select_browser_line(graphics_reps, maxl);
	  reset_edited_rep();
        }
      }
    unfreeze();

  } else if(type == Command::MOL_MODREPITEM) {
    fill_graphics_reps_browser();

  } else
    // unknown command type
    return FALSE;
    
  return TRUE;
}


//////////////////////  public virtual routines  

// initialize the user interface
void GraphicsFormsObj::init(void) {
  int i;

  freeze();
    fl_set_button(graphics_trans, FALSE);
    fl_set_browser_fontstyle(graphics_reps, FL_FIXED_STYLE);

    fl_clear_choice(graphics_color);
    for(i=0; i < AtomColor::TOTAL; i++)
      fl_addto_choice(graphics_color, AtomColorName[i]);
    fl_set_choice(graphics_color, DEFAULT_ATOMCOLOR + 1);

    fl_clear_choice(graphics_drawstyle);
    for(i=0; i < AtomRep::TOTAL; i++)
      fl_addto_choice(graphics_drawstyle, AtomRepName[i]);
    fl_set_choice(graphics_drawstyle, DEFAULT_ATOMREP + 1);

    fl_clear_browser(atomsel_operator);
    for(i=0; i < ATOMSEL_OP_AMOUNT; i++)
      fl_add_browser_line(atomsel_operator, atomselOpStr[i]);
      
    fl_clear_browser(atomsel_which_list);
    for(i=0; i < AtomSel::TOTAL; i++)
      fl_add_browser_line(atomsel_which_list, AtomSelName[i]);
    fl_select_browser_line(atomsel_which_list, AtomSel::ALL + 1);
   
    fl_clear_browser(atomsel_namelist);

    fl_set_counter_bounds(graphics_bond_rad, 0.0, 1000.0);
    fl_set_counter_bounds(graphics_sphere_rad, 0.0, 1000.0);
    fl_set_counter_bounds(graphics_bond_res, 1.0, 1000.0);
    fl_set_counter_bounds(graphics_sphere_res, 1.0, 1000.0);
    fl_set_counter_bounds(graphics_line_thick, 1.0, 1000.0);

    fl_set_counter_step(graphics_bond_rad, 0.01, 0.1);
    fl_set_counter_step(graphics_sphere_rad, 0.01, 0.1);
    fl_set_counter_step(graphics_bond_res, 1.0, 1.0);
    fl_set_counter_step(graphics_sphere_res, 1.0, 1.0);
    fl_set_counter_step(graphics_line_thick, 1.0, 1.0);

    fl_set_counter_precision(graphics_bond_rad, 2);
    fl_set_counter_precision(graphics_sphere_rad, 2);
    fl_set_counter_precision(graphics_bond_res, 0);
    fl_set_counter_precision(graphics_sphere_res, 0);
    fl_set_counter_precision(graphics_line_thick, 0);

    fl_set_counter_value(graphics_bond_rad, 0.3);
    fl_set_counter_value(graphics_sphere_rad, 1.0);
    fl_set_counter_value(graphics_bond_res, 8.0);
    fl_set_counter_value(graphics_sphere_res, 4.0);
    fl_set_counter_value(graphics_line_thick, 1.0);

    // do a reset to set initial values
    reset();

    // properly set the editing mode
    set_editing_mode(GREDIT_DEFAULT);
  unfreeze();
}


// reset the user interface
void GraphicsFormsObj::reset(void) {
  reset_edited_molecule(molID);
}

