/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: FilesFormsObj.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.6 $	$Date: 95/03/24 18:49:22 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The on-screen menu to load a new molecule from a set of files.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	FilesFormsObj.C,v $
 * Revision 1.6  95/03/24  18:49:22  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.5  1994/11/30  12:47:36  billh
 * Fixed bug; reading psf/pdb now correctly chooses coordinate file type.
 *
 * Revision 1.4  94/11/27  08:00:29  billh
 * Changed format dramatically; now selects type of struct/coor file
 * combination preferred, and displays "Please wait" message when reading
 * in a molecule.
 * 
 * Revision 1.3  1994/11/22  02:33:24  billh
 * Small change due to relocation of coor file format data to CoorFileData
 *
 * Revision 1.2  94/10/05  04:37:15  billh
 * Took out double backslash from text, even in comments.
 * 
 * Revision 1.1  1994/10/01  03:08:43  billh
 * Initial revision
 *
 * Revision 1.1  1994/08/24  03:10:37  billh
 * Initial revision
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /private/auto143000131/vmdsrc/vmd/billh/src/RCS/FilesFormsObj.C,v 1.6 95/03/24 18:49:22 billh Exp $";
#endif

#include <stdio.h>
#include "FilesFormsObj.h"
#include "CmdMenu.h"
#include "CmdMol.h"
#include "MoleculeFile.h"
#include "Global.h"
#include "utilities.h"

// different ways to read in a molecule
#define NEWMOL_PSF_PDB	0
#define NEWMOL_PSF_DCD	1
#define NEWMOL_PDB	2
#define NEWMOL_PDB_DCD	3
#define NEWMOL_RASTER3D	4
#define NEWMOL_EDM	5
#define NEWMOL_METHODS	6

// different file types supported here
#define NEWMOL_TYPE_PDB		0
#define NEWMOL_TYPE_PSF		1
#define NEWMOL_TYPE_DCD		2
#define NEWMOL_TYPE_RASTER3D	3
#define NEWMOL_TYPE_EDM		4
#define NEWMOL_TYPE_NONE	5
#define NEWMOL_TYPES		NEWMOL_TYPE_NONE

// static strings and flags for these molecule data file combinations
static int newmol_struct_types[NEWMOL_METHODS] = {
	NEWMOL_TYPE_PSF,	NEWMOL_TYPE_PSF,
	NEWMOL_TYPE_PDB,	NEWMOL_TYPE_PDB,
	NEWMOL_TYPE_RASTER3D,	NEWMOL_TYPE_EDM };
static int newmol_coor_types[NEWMOL_METHODS] = {
	NEWMOL_TYPE_PDB,	NEWMOL_TYPE_DCD,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_DCD,
	NEWMOL_TYPE_NONE,	NEWMOL_TYPE_NONE };
static char *newmol_method_names[NEWMOL_METHODS] = {
	"psf and pdb",		"psf and dcd",
	"pdb only",		"pdb and dcd",
	"Raster3D",		"electron density map" };
static int newmol_struct_codes[NEWMOL_METHODS] = {
	MoleculeFile::PSF,	MoleculeFile::PSF,
	MoleculeFile::PDB,	MoleculeFile::PDB,
	MoleculeFile::RASTER3D,	MoleculeFile::EDM };
static int newmol_coor_codes[NEWMOL_METHODS] = {
	CoorFileData::PDB,	CoorFileData::DCD,
	CoorFileData::UNKNOWN,	CoorFileData::DCD,
	CoorFileData::UNKNOWN,  CoorFileData::UNKNOWN };

// static strings and flags for these molecule data file types
static char *newmol_type_names[NEWMOL_TYPES] = {
	"pdb",			"psf",			"dcd",
	"r3D",			"edm" };
static char *newmol_type_masks[NEWMOL_TYPES] = {
	"*.[pP][dD][bB]",	"*.[pP][sS][fF]",	"*.[dD][cC][dD]",	
	"*",			"*" };


/////////////////////////  constructor  
FilesFormsObj::FilesFormsObj(UIList *uil, CommandQueue *cq,
	int sh,int bd,int *npos) : FormsObj("files", uil, cq, bd, npos) {
	
  // set values of common menu elements
  form = filesMenu;
  lightButton = files_menu_button;
  offButton = files_menu_off;
  
  // turn on if required
  if(sh)
    On();
}

/////////////////////////////  private routines  

// select a filename
void  FilesFormsObj::choose_filename(FL_OBJECT *txt, int ftype, char *prompt) {
  freeze();

  // get file name
  char *mask = newmol_type_masks[ftype];
  char *fname = fl_show_file_selector(prompt, "", mask, "");
  if(fname)
    fl_set_input(txt, fname);

  unfreeze();
}


void FilesFormsObj::reset_mol_type(int newtype) {
  char selectbuf[16];

  freeze();
    // set the structure file inputs properly
    strcpy(selectbuf, "Select ");
    strcat(selectbuf, newmol_type_names[newmol_struct_types[newtype]]);
    fl_set_object_label(struct_file_select, selectbuf);
    fl_set_input(struct_file_name, "");

    // set the coordinate file inputs properly, and show inputs if needed
    fl_set_input(coor_file_name, "");
    if(newmol_coor_types[newtype] != NEWMOL_TYPE_NONE) {
      fl_show_object(coor_file_text);
      fl_show_object(coor_file_select);
      fl_show_object(coor_file_name);
      strcpy(selectbuf, "Select ");
      strcat(selectbuf, newmol_type_names[newmol_coor_types[newtype]]);
      fl_set_object_label(coor_file_select, selectbuf);
    } else {
      fl_hide_object(coor_file_text);
      fl_hide_object(coor_file_select);
      fl_hide_object(coor_file_name);
    }
  unfreeze();
}


//////////////////////  protected virtual routines  

// routine to check the main form for use
int FilesFormsObj::check(FL_OBJECT *obj) {

  if(obj == struct_file_type) {
    reset_mol_type(fl_get_choice(obj) - 1);

  } else if(obj == struct_file_name || obj == coor_file_name) {
    // do nothing

  } else if(obj == struct_file_select) {
    // select was pressed, get new file name
    choose_filename(struct_file_name, 
	newmol_struct_types[fl_get_choice(struct_file_type) - 1],
      	"Select molecular structure file:");
    
  } else if(obj == coor_file_select) {
    // select was pressed, get new file name
    choose_filename(coor_file_name, 
	newmol_coor_types[fl_get_choice(struct_file_type) - 1],
      	"Select initial coordinate file:");
  
  } else if(obj == files_reset) {
    reset();
  
  } else if(obj == files_cancel) {
    reset();
    addcommand(new CmdMenuShow(id(),FALSE,id()));

  } else if(obj == files_load) {
    char *strfile = fl_get_input(struct_file_name);
    char *coorfile = fl_get_input(coor_file_name);
    int ftype = fl_get_choice(struct_file_type) - 1;

    if(strfile && strlen(strfile) < 1)
      strfile = NULL;
    if(coorfile && strlen(coorfile) < 1)
      coorfile = NULL;

    if(!strfile) {
      msgErr << "You must specify a structure file on the top line."<< sendmsg;
    } else if(newmol_coor_types[ftype] != NEWMOL_TYPE_NONE && !coorfile) {
      msgErr << "You must specify a coordinate file on the bottom line.";
      msgErr << sendmsg;
    } else {
      int sftype = newmol_struct_codes[ftype];
      int cftype = newmol_coor_codes[ftype];
      fl_hide_object(obj);
      fl_show_object(mol_wait_text);
      addcommand(new CmdMolNew(strfile, sftype, coorfile, cftype, id()));
    }

  } else
    return FALSE;

  return TRUE;
}


// do form-specific acting on commands.  Return whether
// any action was taken on this command.
// Arguments are the command type, command object, and the 
// success of the command (T or F).
int FilesFormsObj::forms_act_on_command(int type, Command *cmd, int suc) {

  MSGDEBUG(3,"FilesFormsObj: acting on command " << type << sendmsg);

  // check all the possible commands that we look for ...
  // base class already takes care of new, del, and top
  if(type == Command::MOL_NEW && cmd->getUIid() == id() && suc) {
    // this UI did the loading, and it worked ... turn off the menu
    addcommand(new CmdMenuShow(id(),FALSE,id()));

  } else
    // unknown command type
    return FALSE;
    
  return TRUE;
}


//////////////////////  public virtual routines  

// initialize the user interface
void FilesFormsObj::init(void) {
  int i;

  freeze();
    // set all the basic values for controllers, etc
    fl_clear_choice(struct_file_type);
    for(i=0; i < NEWMOL_METHODS; i++)
      fl_addto_choice(struct_file_type, newmol_method_names[i]);
    fl_set_choice(struct_file_type, NEWMOL_PSF_PDB + 1);
    
    // do a reset to set initial values
    reset();

  unfreeze();
}


// reset the user interface
void FilesFormsObj::reset(void) {
  freeze();
    fl_hide_object(mol_wait_text);
    fl_show_object(files_load);

    reset_mol_type(fl_get_choice(struct_file_type) - 1);
  unfreeze();
}

