/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: DisplayFormsObj.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 95/03/24 18:49:00 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The on-screen menu to control the state of the current display device.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	DisplayFormsObj.C,v $
 * Revision 1.4  95/03/24  18:49:00  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.3  1994/11/02  07:32:23  billh
 * Added controls and support to change screen height and distance from origin.
 *
 * Revision 1.2  94/10/05  04:37:15  billh
 * Took out double backslash from text, even in comments.
 * 
 * Revision 1.1  1994/09/29  20:58:48  billh
 * Initial revision
 *
 * Revision 1.1  1994/08/24  03:10:37  billh
 * Initial revision
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /private/auto143000131/vmdsrc/vmd/billh/src/RCS/DisplayFormsObj.C,v 1.4 95/03/24 18:49:00 billh Exp $";
#endif

#include <stdio.h>
#include "DisplayFormsObj.h"
#include "CmdDisplay.h"
#include "LightList.h"
#include "DisplayDevice.h"
#include "Axes.h"
#include "Stage.h"
#include "Global.h"

// commands we are interested in
static int numCmds = 11;
static int cmdList[11] = { Command::DISP_STEREO,
	Command::DISP_EYESEP,		Command::DISP_FOCALLEN,
	Command::DISP_LIGHT_ON,		Command::DISP_CLIP,
	Command::DISP_DEPTHCUE,		Command::DISP_ANTIALIAS,
	Command::DISP_SCRHEIGHT,	Command::DISP_SCRDIST,
	Command::CMD_AXES,		Command::CMD_STAGE };

/////////////////////////  constructor  
DisplayFormsObj::DisplayFormsObj(UIList *uil, CommandQueue *cq,
	int sh,int bd,int *npos) : FormsObj("display", uil, cq, bd, npos) {
	
  // set values of common menu elements
  form = displayMenu;
  lightButton = display_menu_button;
  offButton = display_menu_off;
  
  // register which commands we are interested in
  for(int i=0; i < numCmds; command_wanted(cmdList[i++]));
  
  // turn on if required
  if(sh)
    On();
}

/////////////////////////  private routines  

// set the state of the light browsers
void DisplayFormsObj::reset_light_controls(void) {
  int i;
  char numstr[8];
  
  if(!lights) {
    for(i=0; i < DISP_LIGHTS; i++)
      lightOn[i] = FALSE;
    return;
  }

  freeze();
  fl_clear_browser(light_on_browser);

  lights->reset();
  i = 0;
  while(lights->is_current()) {
    sprintf(numstr, "%-d", i);
    fl_add_browser_line(light_on_browser, numstr);
    lightOn[i] = (lights->current())->light_displayed();
    if(lightOn[i])
      fl_select_browser_line(light_on_browser, i+1);
    lights->next();
    i++;
  }
    
  unfreeze();
}

// set the state of the clipping plane and stereo controls
void DisplayFormsObj::reset_clip_stereo_controls(void) {
  if(!display)
    return;
  
  freeze();
  
  fl_set_counter_value(near_clip_val, display->near_clip());
  fl_set_counter_value(far_clip_val, display->far_clip());
  fl_set_counter_value(eye_sep_val, display->eyesep());
  fl_set_counter_value(focal_len_val, display->eye_dist());
  
  unfreeze();
}

// set the state of the stereo mode
void DisplayFormsObj::reset_stereo_mode(void) {
  freeze();

  fl_clear_choice(stereo_pos); 
  if(display) {
    for(int i=0; i < display->num_stereo_modes(); i++)
      fl_addto_choice(stereo_pos, display->stereo_name(i));
    fl_set_choice(stereo_pos, display->stereo_mode() + 1);
  }
  
  unfreeze();
}


// set the state of display toggles
void DisplayFormsObj::reset_toggle_controls(void) {
  if(!display)
    return;
  
  freeze();
  
  fl_set_button(depthcue_button, display->cueing_enabled());
  fl_set_button(aa_button, display->aa_enabled());

  unfreeze();
}


// set the state of the extra displayed items (axes and stage)
void DisplayFormsObj::reset_displayed_items(void) {
  freeze();
  
  fl_clear_choice(axes_pos);
  fl_clear_choice(stage_pos);
  if(axes) {
    for(int i=0; i < axes->locations(); i++)
      fl_addto_choice(axes_pos, axes->loc_description(i));
    fl_set_choice(axes_pos, axes->location() + 1);
  }
  
  if(stage) {
    for(int i=0; i < stage->locations(); i++)
      fl_addto_choice(stage_pos, stage->loc_description(i));
    fl_set_choice(stage_pos, stage->location() + 1);
  }

  unfreeze();
}


// set the state for text status
void DisplayFormsObj::reset_status_items(void) {
  if(!display)
    return;

  freeze();  
    fl_set_counter_value(screen_height_val, display->screen_height());
    fl_set_counter_value(screen_dist_val, display->distance_to_screen());
  unfreeze();
}


//////////////////////  protected virtual routines  

// routine to check the main form for use
int DisplayFormsObj::check(FL_OBJECT *obj) {

  if(obj == near_clip_val) {
    addcommand(new CmdDisplayClipNear(fl_get_counter_value(obj), id()));

  } else if(obj == far_clip_val) {
    addcommand(new CmdDisplayClipFar(fl_get_counter_value(obj), id()));
    
  } else if(obj == eye_sep_val) {
    addcommand(new CmdDisplayEyesep(fl_get_counter_value(obj), id()));
  
  } else if(obj == focal_len_val) {
    addcommand(new CmdDisplayFocallen(fl_get_counter_value(obj), id()));
  
  } else if(obj == screen_height_val) {
    addcommand(new CmdDisplayScreenHeight(fl_get_counter_value(obj), id()));
  
  } else if(obj == screen_dist_val) {
    addcommand(new CmdDisplayScreenDistance(fl_get_counter_value(obj), id()));
  
  } else if(obj == light_on_browser) {
    for(int i=0; i < DISP_LIGHTS; i++) {
      if(lightOn[i] != fl_isselected_browser_line(obj, i+1))
        addcommand(new CmdDisplayLightOn(i, !lightOn[i], id()));
    }

  } else if(obj == axes_pos) {
    addcommand(new CmdDisplayAxes(fl_get_choice(obj) -  1, id()));

  } else if(obj == stage_pos) {
    addcommand(new CmdDisplayStage(CmdDisplayStage::LOCATION,
	fl_get_choice(obj) -  1, id()));
  
  } else if(obj == stereo_pos) {
    addcommand(new CmdDisplayStereo(fl_get_choice(obj) - 1, id()));
  
  } else if(obj == depthcue_button) {
    addcommand(new CmdDisplayDepthcueOn(fl_get_button(obj), id()));
  
  } else if(obj == aa_button) {
    addcommand(new CmdDisplayAAOn(fl_get_button(obj), id()));
  
  } else
    return FALSE;

  return TRUE;
}


// do form-specific acting on commands.  Return whether
// any action was taken on this command.
// Arguments are the command type, command object, and the 
// success of the command (T or F).
// here, don't need this, just take parent class version
int DisplayFormsObj::forms_act_on_command(int type, Command *, int) {

  MSGDEBUG(3,"DisplayFormsObj: acting on command " << type << sendmsg);

  // check all the possible commands that we look for ...
  if(type == Command::DISP_STEREO) {
    reset_stereo_mode();

  } else if(type == Command::DISP_EYESEP || type == Command::DISP_FOCALLEN ||
  	type == Command::DISP_CLIP) {
    reset_clip_stereo_controls();

  } else if(type == Command::DISP_LIGHT_ON) {
    reset_light_controls();

  } else if(type == Command::DISP_DEPTHCUE ||type == Command::DISP_ANTIALIAS) {
    reset_toggle_controls();

  } else if(type == Command::CMD_AXES || type == Command::CMD_STAGE) {
    reset_displayed_items();

  } else if(type == Command::DISP_SCRHEIGHT || type == Command::DISP_SCRDIST) {
    reset_status_items();

  } else
    // unknown command type
    return FALSE;
    
  return TRUE;
}


//////////////////////  public virtual routines  

// initialize the user interface
void DisplayFormsObj::init(void) {
  freeze();
    // set all the basic values for controllers, etc
    fl_set_counter_bounds(near_clip_val, 0.001, 1000.0);
    fl_set_counter_bounds(far_clip_val, 0.001, 1000.0);
    fl_set_counter_bounds(eye_sep_val, 0.0, 100.0);
    fl_set_counter_bounds(focal_len_val, 0.001, 1000.0);
    fl_set_counter_bounds(screen_height_val, 0.0, 100000.0);
    fl_set_counter_bounds(screen_dist_val, -100000.0, 100000.0);
    fl_set_counter_step(near_clip_val, 0.05, 0.5);
    fl_set_counter_step(far_clip_val, 0.05, 0.5);
    fl_set_counter_step(eye_sep_val, 0.01, 0.1);
    fl_set_counter_step(focal_len_val, 0.05, 0.5);
    fl_set_counter_step(screen_height_val, 0.1, 1.0);
    fl_set_counter_step(screen_dist_val, 0.1, 1.0);
    fl_set_counter_precision(near_clip_val, 2);
    fl_set_counter_precision(far_clip_val, 2);
    fl_set_counter_precision(eye_sep_val, 2);
    fl_set_counter_precision(focal_len_val, 2);
    fl_set_counter_precision(screen_height_val, 1);
    fl_set_counter_precision(screen_dist_val, 1);
   
    // do a reset to set initial values
    reset();

  unfreeze();
}


// reset the user interface
void DisplayFormsObj::reset(void) {
  freeze();
    reset_light_controls();
    reset_clip_stereo_controls();
    reset_stereo_mode();
    reset_toggle_controls();
    reset_displayed_items();
    reset_status_items();
  unfreeze();
}
