/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CoorPDB.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 95/03/24 18:48:44 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * PDB file class - subclasses of ICoorFile and OCoorfile; used to read and
 * write PDB files.  When reading PDB files, the user may specify to read
 * structure info; if so, all the PDB info is read and stored.  If not,
 * only the coordinates are read.  When reading/writing PDB files, a
 * BaseMolecule must be specified, so that the structure can be properly
 * compared.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	CoorPDB.C,v $
 * Revision 1.4  95/03/24  18:48:44  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.3  1994/12/11  01:16:23  dalke
 * Added provision for other functions to reference the PDB write
 *
 * Revision 1.2  1994/11/22  02:32:55  billh
 * Fixed error with writing PDB file; now prints occup and beta values properly.
 *
 * Revision 1.1  94/09/17  09:11:36  billh
 * Initial revision
 * 
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include "CoorPDB.h"
#include "ReadPDB.h"
#include "BaseMolecule.h"
#include "Timestep.h"
#include "Inform.h"
#include "Atom.h"

/****************************  ICoorPDB routines  **************************/

// constructor
ICoorPDB::ICoorPDB(BaseMolecule *newmol) {
  // save the molecule pointer
  mol = newmol;

  // save number of atoms expected in file
  nAtoms = mol->nAtoms;
  
  // a PDB file can only store 1 coordinate set.
  nFrames = 1;
}


// close the input file; return success
ICoorPDB::~ICoorPDB(void) {
  if(opened())
    fclose(cf);
}


// initializer; reads header info and inits all variables.  Returns if file was
// initialized.
int ICoorPDB::init(char *fn) {
  char pdbstr[PDB_RECORD_LENGTH+2];
  int indx;
  
  // if file is already open, we can continue processing it.
  if(opened())
    return TRUE;

  // initially we assume init failed; also, must do base-class initialization
  Initialized = FALSE;
  ICoorFile::init(fn);

  // open the file
  if(!(cf = fopen(filename(),"r"))) {
    errorNo = COOR_ERR_NOTOPEN;
    return FALSE;
  }
  
  // must scan the file once to find number of atoms, if this is unknown.
  if(nAtoms <= 0) {
    nAtoms = 0;
    do {
      if((indx = read_pdb_record(cf, pdbstr)) == PDB_ATOM)
        nAtoms++;
    } while (indx != PDB_END && indx != PDB_EOF);
    rewind(cf);
  
    // if still no atoms, there is an error
    if(!nAtoms) {
      errorNo = COOR_ERR_NOATOMS;
      return FALSE;
    }
  }

  MSGDEBUG(1,"Opened PDB input file '" << filename() << "'." << sendmsg);
  MSGDEBUG(1,"   Atoms: " << nAtoms << sendmsg);

  return (Initialized = TRUE);
}


// read the next set of coordinates; return list of positions, or NULL if
// error
Timestep* ICoorPDB::read(void) {
  int indx, spos = 0, i = 0, j = 0;
  char pdbstr[PDB_RECORD_LENGTH+2];
  float *x, *y, *z, *occup, *beta;
  Timestep *pos;

  if(!opened() || (currFrames == nFrames))
    return NULL;

  // create a new position list to store the data
  pos = new Timestep(nAtoms, 0.0);
  x = pos->pos;
  y = x + 1;
  z = x + 2;
  occup = pos->data + ATOMOCCUP;
  beta = pos->data + ATOMBETA;
  
  // read any header info
  read_header();
  
  // read through the file, storing the data we need.
  do {
    indx = read_pdb_record(cf, pdbstr);
    if((indx == PDB_END || indx == PDB_EOF) && (i < nAtoms)) {
      errorNo = COOR_ERR_EARLYEOF;
      delete pos;
      return NULL;
    } else if(indx == PDB_REMARK) {
      add_comment(pdbstr);
    } else if(indx == PDB_ATOM) {
      if(i >= nAtoms) {
        errorNo = COOR_ERR_EXTRA;
	break;		// return frame, but with error
      }
      // just get the coordinates, and store them
      get_pdb_coordinates(pdbstr, x, y, z, occup, beta);
      x += ATOMCOORDS;
      y += ATOMCOORDS;
      z += ATOMCOORDS;
      occup += ATOMEXTRA;
      beta += ATOMEXTRA;
      
      // supply values for extra data
      Atom *atom = mol->atom(i);
      pos->data[j + ATOMCHARGE] = atom->extra[ATOMCHARGE];
      pos->data[j + ATOMMASS] = atom->extra[ATOMMASS];
      pos->data[j + ATOMRAD] = atom->extra[ATOMRAD];

      i++;
      j += ATOMEXTRA;
    }
  } while(!(indx == PDB_END || indx == PDB_EOF));

  currFrames++;
  return pos;
}


/****************************  OCoorPDB routines  **************************/
 
// constructor
OCoorPDB::OCoorPDB(BaseMolecule *newmol) {
  // save the molecule pointer
  mol = newmol;  
  nAtoms = mol->nAtoms;
  
  // a PDB file can only store 1 coordinate set.
  nFrames = 1;
}


// destructor
// close the output file; return success
OCoorPDB::~OCoorPDB(void) {
  if(opened()) {
    fprintf(cf,"END\n");
    fclose(cf);
  }
}


// initializer; writes header info and inits all variables.  Returns TRUE
// if file was initialized.
int OCoorPDB::init(char *fn) {  
  // if file is already open, we can continue processing it.
  if(opened())
    return TRUE;

  // initially we assume init failed; also, must do base-class initialization
  Initialized = FALSE;
  OCoorFile::init(fn);

  // open the file
  if(!(cf = fopen(filename(),"w"))) {
    errorNo = COOR_ERR_NOTOPEN;
    return FALSE;
  }
  
  // write header comments to beginning of file
  write_header();
  
  MSGDEBUG(1,"Opened PDB output file '" << filename() << "'." << sendmsg);
  MSGDEBUG(1,"   Atoms: " << nAtoms << sendmsg);
  
  return (Initialized = TRUE);
}


// write header comments
void OCoorPDB::write_header(void) {

  if(!cf || !mol)
    return;

  fprintf(cf,"REMARK File: %s\n",filename());
  fprintf(cf,"REMARK Atoms in molecule: %d\n",mol->nAtoms);
  fprintf(cf,"REMARK This file created by program 'vmd', user '%s'\n",user());
  fprintf(cf,"REMARK -------------------------------------------------\n");
}


// write the coordinates in the given coordinate set to the file.  Return
// total frames written so far, or (-1) if error.
int OCoorPDB::write(Timestep *pos) {
  int i;
  
  // check for errors
  if(!opened() || !pos || (currFrames == nFrames))
    return (-1);

  if(pos->num != nAtoms) {
    errorNo = COOR_ERR_A_NOT_EQ;
    return (-1);
  }
  // write the coordinate data
  for(i=0; i < nAtoms; i++) {
     write_pdb_record(mol, pos, i, cf);
  }
  return ++currFrames;
}


// This is not a member of the class so that other routines may access
// it, specifically, the 'ribbons' routine in DrawMolItem
// this may someday include the DSSP interface
// i is the atom index
void write_pdb_record(BaseMolecule *mol, Timestep *pos, int i, FILE *cf) {
  int p, resid;
  float *loc, *occup, *beta;
  Atom *atom = mol->atom(i);
  char name[6], resname[4], segname[5], *nameptr;

  loc = pos->pos + ATOMCOORDS * i;
  occup = pos->data + ATOMOCCUP + ATOMEXTRA * i;
  beta = pos->data + ATOMBETA + ATOMEXTRA * i;
  name[0] = ' ';
  strncpy(name+1, atom->namestr, 4);
  name[5] = '\0';
    
  // the name must be left-justified
  if(strlen(name) == 5) {
    nameptr = name + 1;
  } else {
    nameptr = name;
    while((p = strlen(name)) < 4) {
      name[p] = ' ';
      name[p+1] = '\0';
    }
  }
  strncpy(resname,atom->resnamestr,3);
  resname[3] = '\0';
  strncpy(segname,atom->segnamestr,4);
  segname[4] = '\0';
  resid = atoi(atom->residstr);

  write_raw_pdb_record(cf, "ATOM  ", i+1, nameptr, resname, resid,
	loc[0],loc[1],loc[2],*occup,*beta,segname);
}


void write_raw_pdb_record(FILE *outfile, char *recordname, int index, 
    char *atomname, char *resname, int resid, float x, float y, 
    float z, float occ, float beta, char *segname) {
    
  fprintf(outfile,
    "%s%5d %4s%c%3s %c%4d%c   %8.3f%8.3f%8.3f%6.2f%6.2f      %4s\n",
    	recordname, index, atomname, ' ', resname, ' ', resid, ' ',
    	x, y, z, occ, beta, segname);
}
