/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: CmdRemote.C,v $
 *      $Author: billh $        $Locker:  $                $State: Exp $
 *      $Revision: 1.5 $      $Date: 95/03/24 18:48:02 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Command objects for affecting remote connections.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	CmdRemote.C,v $
 * Revision 1.5  95/03/24  18:48:02  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.4  1994/11/09  02:50:28  billh
 * Added finished support for displaying patches for a remote molecule.
 *
 * Revision 1.3  94/10/21  03:50:45  billh
 * Updated for new simulation control commands, and remoteList object.
 * 
 * Revision 1.2  1994/10/05  04:37:15  billh
 * Took out double backslash from text, even in comments.
 *
 * Revision 1.1  94/10/01  11:04:31  billh
 * Initial revision
 * 
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /private/auto143000131/vmdsrc/vmd/billh/src/RCS/CmdRemote.C,v 1.5 95/03/24 18:48:02 billh Exp $";
#endif

#include "CmdRemote.h"
#include "CmdMol.h"
#include "MoleculeRemote.h"
#include "DrawPatch.h"
#include "RemoteList.h"
#include "Global.h"


//////////////////////  general commands used by these classes

  // print error message about no remote pointer
  void no_remote_error(void) {
    msgErr << "You must first initialize a new remote connection." << sendmsg;
  }
  
  void no_running_sim_error(void) {
    msgErr << "You must first have an active, running remote simulation.";
    msgErr << sendmsg;
  }

  // print out available application for current remote object.
  void print_avail_apps(void) {
    msgInfo << "Available applications to run on " << remote->host();
    msgInfo << ": " << remote->avail_apps() << sendmsg;
    for(int i=0; i < remote->avail_apps(); i++)
      msgInfo << i << ":  " << remote->avail_app_name(i) << sendmsg;
  }
  
  // print out available jobs on remote computer
  void print_avail_jobs(void) {
    int pid, uid;
    msgInfo << "Jobs currently running on " << remote->host();
    msgInfo << ": " << remote->running_jobs() << sendmsg;
    for(int i=0; i < remote->running_jobs(); i++) {
      remote->running_job_ids(i, pid, uid);
      msgInfo << i << ":  " << remote->running_job_name(i);
      msgInfo << " (pid=" << pid << ", uid=" << uid << ")" << sendmsg;
    }
  }
  
  // print out current parameter data for remote connection setup
  void print_parameters(void) {
    msgInfo << "Current parameters for remote application " << remote->app();
    msgInfo << sendmsg;
    for(int i=0; i < remote->options(); i++)
      msgInfo << i << ": " << remote->option_string(i) << sendmsg;
  }

  // print out all data in a summary
  void print_remote_status(CmdRemoteList::RemoteList whatToList) {
    msgInfo << "Current remote connection setup:" << sendmsg;
    msgInfo << "--------------------------------" << sendmsg;
    msgInfo << "Status: " << remote->status() << sendmsg;
    if(remote->error_status() != MDC_EOK)
      msgInfo << "Last error: " << remote->error_status_desc() << sendmsg;
    if(whatToList == CmdRemoteList::APPS || whatToList == CmdRemoteList::ALL)
      print_avail_apps();
    if(whatToList == CmdRemoteList::JOBS || whatToList == CmdRemoteList::ALL)
      print_avail_jobs();
    if(whatToList == CmdRemoteList::PARAMS||whatToList == CmdRemoteList::ALL) {
      if(remote->options() > 0)
        print_parameters();
      else
        msgInfo << "No parameters are yet available for editing." << sendmsg;
    }
  }
  
  // print out list of currently/previously running simulations
  void print_simulation_summary(void) {
    msgInfo << "Current and previous remote simulations:" << sendmsg;
    msgInfo << "----------------------------------------" << sendmsg;
    if(remoteList->num() > 0) {
      msgInfo << "ID    Connected to             Status" << sendmsg;
      for(int i=0; i < remoteList->num(); i++) {
        MoleculeRemote *rem = remoteList->remote(i);
	msgInfo << rem->id() << "   " << (rem->rem)->host() << "       ";
	msgInfo << (rem->rem)->status() << sendmsg;
      }
    } else {
      msgInfo << "No current or previous connections." << sendmsg;
    }
  }

///////////////////////////  CmdRemoteInit
///////////// initialize remote with a new connection
int CmdRemoteInit::do_execute(void) {
  if(remote == NULL) {
    // no remote, can initialize
    remote = new Remote(username, computer);
    if(remote->error_status() != MDC_EOK) {
      msgErr << "Cannot initialize remote connection." << sendmsg;
      delete remote;
      remote = NULL;
    } else {
      msgInfo << "Remote connection established." << sendmsg;
      print_remote_status(CmdRemoteList::ALL);
      return TRUE;
    }
  } else {
    msgErr << "There is already an existing remote connection in progress.";
    msgErr << sendmsg;
  }
  
  // if here, there was an error
  return FALSE;
}

CmdRemoteInit::CmdRemoteInit(char *cmptr, char *un, int newUIid)
	: Command(Command::REMOTE_INIT, newUIid) {
  computer = stringdup(cmptr);
  if(un)
    username = stringdup(un);
  else
    username = ::username();
  *cmdText << "remote init " << computer << " " << username << ends;
}

CmdRemoteInit::~CmdRemoteInit(void) {
  delete [] computer;
  delete [] username;
}


///////////////////////////  CmdRemoteCancel  
//////////// cancel the current connection
int CmdRemoteCancel::do_execute(void) {
  if(remote != NULL) {
    delete remote;
    remote = NULL;
  } else {
    no_remote_error();
    return FALSE;
  }
  return TRUE;
}

CmdRemoteCancel::CmdRemoteCancel(int newUIid)
	: Command(Command::REMOTE_CANCEL, newUIid) {
  *cmdText << "remote cancel" << ends;
}


///////////////////////////  CmdRemoteList  
/////////////// list the current remote objects settings
int CmdRemoteList::do_execute(void) {
  if(remote != NULL) {
    print_remote_status(whatToList);
  } else {
    no_remote_error();
    return FALSE;
  }
  return TRUE;
}

CmdRemoteList::CmdRemoteList(RemoteList whattodo, int newUIid)
	: Command(Command::REMOTE_LIST, newUIid) {
  whatToList = whattodo;
  *cmdText << "remote list ";
  if(whatToList == JOBS)
    *cmdText << "jobs";
  else if(whatToList == APPS)
    *cmdText << "apps";
  else if(whatToList == PARAMS)
    *cmdText << "parameters";
  *cmdText << ends;
}


///////////////////////////  CmdRemoteNew  
/////////////// set the remote connection setup to get the parameters
/////////////// to start a new job on the remote computer, using the selected
/////////////// app.  App is an index into the list of possible applications
/////////////// that can be started.
int CmdRemoteNew::do_execute(void) {
  if(remote != NULL) {
    if(remote->get_parameters(whichApp)) {
      // add command to print out listing
      CmdRemoteList listParams(CmdRemoteList::PARAMS);
      return listParams.execute();
    } else {
      // error getting parameters
      msgErr << "Cannot get parameters for the selected application.";
      msgErr << sendmsg;
    }
  } else {
    no_remote_error();
  }

  // if here, did not work
  return FALSE;
}

CmdRemoteNew::CmdRemoteNew(int newapp, int newUIid)
	: Command(Command::REMOTE_NEW, newUIid) {
  whichApp = newapp;
  *cmdText << "remote new " << whichApp << ends;
}


///////////////////////////  CmdRemoteAttach  
/////////////// have the remote simulation attach to the specified currently-
/////////////// running job ... a new molecule will be created if this is
/////////////// successful
int CmdRemoteAttach::do_execute(void) {
  if(remote != NULL) {
    if(remote->attach_to_job(whichJob)) {
      // attach was successful ... now create a new molecule
      CmdMolNew newMol((-1));		// constructor for creating remote mol
      return newMol.execute();		// return success of execution
    } else {
      // error attaching to job
      msgErr << "Cannot attach to the specified job.  Clearing remote setup.";
      msgErr << sendmsg;
    }
  } else {
    no_remote_error();
  }

  // if here, did not work
  return FALSE;
}

CmdRemoteAttach::CmdRemoteAttach(int newjob, int newUIid)
	: Command(Command::REMOTE_ATTACH, newUIid) {
  whichJob = newjob;
  *cmdText << "remote attach " << whichJob << ends;
}


///////////////////////////  CmdRemoteRun  
/////////////// creates a new simulation using previously established
/////////////// parameters.  Only works if a CmdRemoteNew command has been
/////////////// executed previously
int CmdRemoteRun::do_execute(void) {
  if(remote != NULL) {
    if(remote->start_new_simulation()) {
      // started successfully ... now create a new molecule
      CmdMolNew newMol((-1));		// constructor for creating remote mol
      return newMol.execute();		// return success of execution
    } else {
      // error starting new job
      msgErr << "Cannot start a new simulation.  Most recent error:\n";
      msgErr << "  " << remote->error_status_desc() << sendmsg;
    }
  } else {
    no_remote_error();
  }

  // if here, did not work
  return FALSE;
}

CmdRemoteRun::CmdRemoteRun(int newUIid)
	: Command(Command::REMOTE_RUN, newUIid) {
  *cmdText << "remote run" << ends;
}


///////////////////////////  CmdRemoteSetopt  
/////////////// sets the value for the Nth parameter in the current
/////////////// remote simulation being set up.  Only works if parameters
/////////////// are being edited.
int CmdRemoteSetopt::do_execute(void) {
  if(remote != NULL) {
    if(remote->set_option(keyword, paramVal)) {
      msgInfo << "Parameter " << keyword << " changed." << sendmsg;
      return TRUE;
    } else {
      msgErr << "Unable to change parameter value." << sendmsg;
    }
  } else {
    no_remote_error();
  }

  // if here, did not work
  return FALSE;
}

CmdRemoteSetopt::CmdRemoteSetopt(char *kw, char *pv, int newUIid)
	: Command(Command::REMOTE_SETOPT, newUIid) {
  keyword = stringdup(kw);
  paramVal = stringdup(pv);
  *cmdText << "remote setoption " << keyword << " " << paramVal << ends;
}

CmdRemoteSetopt::~CmdRemoteSetopt(void) {
  delete [] paramVal;
  delete [] keyword;
}

///////////////////////////  CmdRemoteReadopt  
/////////////// read in parameter values from the given file.
/////////////// Only works if a CmdRemoteNew command has been
/////////////// executed previously.
int CmdRemoteReadopt::do_execute(void) {
  FILE *f;

  if(remote != NULL) {
    if(remote->editing_parameters()) {
      if(!(f = fopen(fname, "r"))) {
        msgErr << "Cannot open parameter file for reading." << sendmsg;
      } else {
        if(! remote->read_param_file(f)) {
	  msgErr << "Problem settings parameters from file " << fname;
	  msgErr << sendmsg;
	}
	fclose(f);
	return TRUE;
      }
    } else {
      msgErr << "You must first be editing a set of parameters." << sendmsg;
    }
  } else {
    no_remote_error();
  }

  // if here, did not work
  return FALSE;
}

CmdRemoteReadopt::CmdRemoteReadopt(char *fn, int newUIid)
	: Command(Command::REMOTE_READOPT, newUIid) {
  fname = stringdup(fn);
  *cmdText << "remote readoptions " << fname << ends;
}

CmdRemoteReadopt::~CmdRemoteReadopt(void) {
  delete [] fname;
}


///////////////////////////  CmdRemoteReadopt  
/////////////// write current parameter values to the given file.
/////////////// This will work as long as the current remote object has
/////////////// any parameters to write.
int CmdRemoteWriteopt::do_execute(void) {
  FILE *f;

  if(remote != NULL) {
    if(remote->options() > 0) {
      if(!(f = fopen(fname, "w"))) {
        msgErr << "Cannot open parameter file for writing." << sendmsg;
      } else {
        if(! remote->write_param_file(f)) {
	  msgErr << "Problem writing parameters to file " << fname;
	  msgErr << sendmsg;
	}
	fclose(f);
	return TRUE;
      }
    } else {
      msgErr << "There are no available options to write." << sendmsg;
    }
  } else {
    no_remote_error();
  }

  // if here, did not work
  return FALSE;
}

CmdRemoteWriteopt::CmdRemoteWriteopt(char *fn, int newUIid)
	: Command(Command::REMOTE_WRITEOPT, newUIid) {
  fname = stringdup(fn);
  *cmdText << "remote writeoptions " << fname << ends;
}

CmdRemoteWriteopt::~CmdRemoteWriteopt(void) {
  delete [] fname;
}


///////////////////////////  CmdRemoteChangeParam  
/////////////// issue command to change parameter for top remote connection.
int CmdRemoteChangeParam::do_execute(void) {
  MoleculeRemote *rem;

  if((rem = remoteList->top()) != NULL && (rem->rem)->running_simulation()) {
    if((rem->rem)->change_setting(param, newval))
      return TRUE;
    else
      msgErr << "Bad parameter name '" << param << "'" << sendmsg;
  }

  // if here, did not work
  no_running_sim_error();
  return FALSE;
}

CmdRemoteChangeParam::CmdRemoteChangeParam(char *pm, char *nv, int newUIid)
	: Command(Command::SIM_SETPARAM, newUIid) {
  param = stringdup(pm);
  newval = stringdup(nv);
  *cmdText << "sim modify " << param << " " << newval << ends;
}

CmdRemoteChangeParam::~CmdRemoteChangeParam(void) {
  delete [] newval;
  delete [] param;
}


///////////////////////////  CmdRemotePatchDisplay  
/////////////// change how patches are displayed (if they are even available)
int CmdRemotePatchDisplay::do_execute(void) {
  MoleculeRemote *rem;

  if((rem = remoteList->top()) != NULL && newval >= 0)
    return (rem->patch_display_method(newval));
  
  // if here, did not work
  return FALSE;
}

CmdRemotePatchDisplay::CmdRemotePatchDisplay(char *nv, int newUIid)
	: Command(Command::SIM_PATCHDISP, newUIid) {
  int n = 0;
  newval = (-1);
  while(n < DrawPatch::TOTAL_PATCH_DISPLAY) {
    if(!strupncmp(nv, patchDisplayMethodName[n], CMDLEN)) {
      newval = n;
      break;
    }
    n++;
  }
  if(newval >= 0)
    *cmdText << "sim patch " << patchDisplayMethodName[newval];
  else
    msgErr << "Unknown patch display method " << nv << sendmsg;
  *cmdText << ends;
}

CmdRemotePatchDisplay::CmdRemotePatchDisplay(int nv, int newUIid)
	: Command(Command::SIM_PATCHDISP, newUIid) {
  newval = nv;
  *cmdText << "sim patch " << patchDisplayMethodName[newval] << ends;
}


///////////////////////////  CmdRemoteTop  
/////////////// set the specified molecule as the top remote connection
int CmdRemoteTop::do_execute(void) {
  if(remoteList)
    return remoteList->make_top(remoteList->rem_index_from_id(id));
  
  // if here, did not work
  msgErr << "Bad molecule ID " << id << " specified" << sendmsg;
  return FALSE;
}

CmdRemoteTop::CmdRemoteTop(int newid, int newUIid)
	: Command(Command::SIM_TOP, newUIid) {
  id = newid;
  *cmdText << "sim top " << id << ends;
}



///////////////////////////  CmdRemoteDetach  
/////////////// detach from a running simulation, but don't kill it
int CmdRemoteDetach::do_execute(void) {
  MoleculeRemote *rem;

  if((rem = remoteList->remote(remoteList->rem_index_from_id(id))) != NULL) {
    msgWarn << "Detaching from remote application - note that it is still";
    msgWarn << " running." << sendmsg;
    (rem->rem)->remote_close(TRUE);
    return TRUE;
  }
  
  // if here, did not work
  msgErr << "Bad molecule ID " << id << " specified" << sendmsg;
  return FALSE;
}

CmdRemoteDetach::CmdRemoteDetach(int newid, int newUIid)
	: Command(Command::SIM_DETACH, newUIid) {
  id = newid;
  *cmdText << "sim detach " << id << ends;
}


///////////////////////////  CmdRemoteKill  
/////////////// kill a running simulation
int CmdRemoteKill::do_execute(void) {
  MoleculeRemote *rem;

  if((rem = remoteList->remote(remoteList->rem_index_from_id(id))) != NULL) {
    msgWarn << "Killing remote application - note that it will no longer be";
    msgWarn << " running." << sendmsg;
    (rem->rem)->remote_close(FALSE);
    return TRUE;
  }
  
  // if here, did not work
  msgErr << "Bad molecule ID " << id << " specified" << sendmsg;
  return FALSE;
}

CmdRemoteKill::CmdRemoteKill(int newid, int newUIid)
	: Command(Command::SIM_KILL, newUIid) {
  id = newid;
  *cmdText << "sim kill " << id << ends;
}


///////////////////////////  CmdRemoteSimList  
/////////////// list all current and previous simulations
int CmdRemoteSimList::do_execute(void) {
  if(remoteList)
    print_simulation_summary();
  else
    return FALSE;
    
  return TRUE;
}

CmdRemoteSimList::CmdRemoteSimList(int newUIid)
	: Command(Command::SIM_LIST, newUIid) {
  *cmdText << "sim list" << ends;
}
