/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: CmdMol.h,v $
 *      $Author: billh $        $Locker:  $                $State: Exp $
 *      $Revision: 1.11 $      $Date: 95/03/24 18:47:57 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Command objects for affecting molecules.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	CmdMol.h,v $
 * Revision 1.11  95/03/24  18:47:57  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.10  1994/11/22  02:32:05  billh
 * Improved way new molecule command specifies structure and coordiante
 * file type ... can be used (mostly) unchanged when new types are added.
 *
 * Revision 1.9  94/10/26  23:21:34  billh
 * Added routines to change the settings for graphics representations
 * (MOL_MODREP, MOL_MODREPITEM).
 * 
 * Revision 1.8  94/10/03  01:39:42  dalke
 * Took code from .h to .C, leaving only prototypes (to reduce inlining)
 * 
 * Revision 1.7  1994/10/01  11:02:26  billh
 * Added commands to create a remotely-connected molecule.
 *
 * Revision 1.6  94/10/01  03:06:19  billh
 * Changed type of argument to 'create molecule' command (CmdMolNew).
 * 
 * Revision 1.5  1994/09/29  07:10:21  billh
 * Molecule commands updated to allow lists of indices.
 *
 * Revision 1.4  94/09/26  18:50:58  billh
 * Updated molecule commands for adding and listing.
 * 
 * Revision 1.3  94/09/24  20:13:29  billh
 * Put in call to 'moleculeList->add_color_names(newmol)' when a new
 * molecule is added, to initialize the color calculation.
 * 
 * Revision 1.2  94/09/23  00:43:13  billh
 * Added these commands:
 * MOL_FIX, MOL_ON, MOL_TOP, MOL_SELECT, MOL_REP, MOL_COLOR, MOL_ADDREP
 * 
 * Revision 1.1  1994/09/17  09:11:36  billh
 * Initial revision
 *
 ***************************************************************************/
#ifndef CMDMOL_H
#define CMDMOL_H

#include "Command.h"
#include "Global.h"
#include "MoleculeFile.h"
#include "MoleculeList.h"
#include "DrawMolItem.h"
#include "utilities.h"


// the following uses the Cmdtypes MOL_NEW, MOL_LIST, MOL_DEL, MOL_ACTIVE,
// MOL_FIX, MOL_ON, MOL_TOP, MOL_SELECT, MOL_REP, MOL_COLOR, MOL_ADDREP,
// MOL_MODREP, MOL_DELREP, MOL_MODREPITEM



/*
 * NOTES:
 *
 *	1) When referring to a molecule in a command by a number, the
 * unique molecule ID is used, NOT the relative index of the molecule into
 * the molecule list.  This is because the relative index changes as molecules
 * are added/removed, but the unique ID stays the same until the molecule is
 * deleted.
 */


///////// load a new molecule, either from a file, or from a remote connection
class CmdMolNew : public Command {
public:
  // id of new molecule (-1 if problem)
  int newID;

private:
  // method to read structure
  enum MolSource { MOLFILE, MOLREMOTE };

  // where is the molecule coming from?
  MolSource molSource;

  // structure file
  char *molStrFile;
  
  // what kind of structure file is this?
  int molStrFileType;

  // initial coordinate file ... if NULL, no coordinates loaded at start.
  char *molCoorFile;
  
  // type of coordinate file
  int molCoorFileType;

protected:
  virtual int do_execute(void) ;

public:
  // constructor A: read struct from file
  CmdMolNew(char *fname, int sftype, char *cfile, int cftype, int newUIid=(-1));
  
  // constructor B: read struct from remote.  Don't specify anything.
  CmdMolNew(int newUIid=(-1));
  
  // destructor
  virtual ~CmdMolNew(void) ;
};


///////////////  list the current set of loaded molecules
class CmdMolList : public Command {
private:
  // which molecule to print into for; if < 0, print summary of all
  int whichMol;		// ID of molecule to list

  // possible list of molecules to deal with; if NULL, use whichMol instead
  int *idList;
  int numList;

  // print out molecule summary line
  void print_mol_summary(int i) ;
  
  // print out atom rep summary line
  void print_arep_summary(Molecule *mol, int i) ;
  
protected:
  virtual int do_execute(void) ;

public:
  CmdMolList(int wm, int newUIid=(-1)) ;
  
  // constructor for a whole list of molecules.  The list will be deleted
  // when this is done, but not allocated, so the caller must new an array
  // of integers and pass it to this object.
  CmdMolList(int n, int *list, int newUIid=(-1));
  
  virtual ~CmdMolList(void) ;
};


///////////////  delete the Nth molecule in the molecule List
class CmdMolDelete : public Command {
public:
  int n;		// ID of molecule to delete

  // possible list of molecules to deal with; if NULL, use n instead
  int *idList;
  int numList;

  // peform actual action for molecule i.  return success.
  int do_action(int i) ;

protected:
  virtual int do_execute(void) ;
  
public:
  CmdMolDelete(int nmol,int newUIid=(-1));

  // alternate constructor where a list of molecules is given
  CmdMolDelete(int numn, int *list, int newUIid=(-1));
  
  virtual ~CmdMolDelete(void) ;
};


///////////////  make the Nth molecule 'active' or 'inactive'
class CmdMolActive : public Command {
public:
  int n;		// molecule to change
  int yn;		// new state for characteristic

  // possible list of molecules to deal with; if NULL, use n instead
  int *idList;
  int numList;

  // peform actual action for molecule i.  return success.
  int do_action(int i) ;

protected:
  virtual int do_execute(void) ;
  
public:
  CmdMolActive(int nmol, int newyn, int newUIid=(-1));

  // alternate constructor where a list of molecules is given
  CmdMolActive(int numn, int *list, int newyn, int newUIid=(-1));
  
  virtual ~CmdMolActive(void) ;
};


///////////////  make the Nth molecule 'fixed' or 'free'
class CmdMolFix : public Command {
public:
  int n;		// molecule to change
  int yn;		// new state for characteristic

  // possible list of molecules to deal with; if NULL, use n instead
  int *idList;
  int numList;

  // peform actual action for molecule i.  return success.
  int do_action(int i);
protected:
  virtual int do_execute(void) ;
  
public:
  CmdMolFix(int nmol, int newyn, int newUIid=(-1));

  // alternate constructor where a list of molecules is given
  CmdMolFix(int numn, int *list, int newyn, int newUIid=(-1));
  
  virtual ~CmdMolFix(void) ;
};



///////////////  make the Nth molecule 'on' or 'off'
class CmdMolOn : public Command {
public:
  int n;		// molecule to change
  int yn;		// new state for characteristic

  // possible list of molecules to deal with; if NULL, use n instead
  int *idList;
  int numList;

  // peform actual action for molecule i.  return success.
  int do_action(int i) ;

protected:
  virtual int do_execute(void) ;

public:
  CmdMolOn(int nmol, int newyn, int newUIid=(-1));

  // alternate constructor where a list of molecules is given
  CmdMolOn(int numn, int *list, int newyn, int newUIid=(-1));
  
  virtual ~CmdMolOn(void) ;
};



///////////////  make the Nth molecule 'top'
class CmdMolTop : public Command {
public:
  int n;		// ID of molecule to change

protected:
  virtual int do_execute(void) ;
  
public:
  CmdMolTop(int nmol, int newUIid=(-1));
};


///////////// set the current atom selection in moleculeList
class CmdMolSelect : public Command {
public:
  // new selection command (if NULL, print out current settings)
  char *sel;
  
protected:
  virtual int do_execute(void) ;

public:
  CmdMolSelect(char *newsel, int newUIid=(-1));
  
  virtual ~CmdMolSelect(void) ;
};


///////////// set the current atom representation in moleculeList
class CmdMolRep : public Command {
public:
  // new representation command (if NULL, print out current settings)
  char *sel;
  
protected:
  virtual int do_execute(void) ;

public:
  CmdMolRep(char *newsel, int newUIid=(-1));
  
  virtual ~CmdMolRep(void) ;
};


///////////// set the current atom coloring method in moleculeList
class CmdMolColor : public Command {
public:
  // new color command (if NULL, print out current settings)
  char *sel;
  
protected:
  virtual int do_execute(void) ;

public:
  CmdMolColor(char *newsel, int newUIid=(-1));
  
  virtual ~CmdMolColor(void) ;
};


///////////// add a new representation to the active molecules
class CmdMolAddRep : public Command {
public:
  // molecule to add rep to (if < 0, add to all active molecules)
  int n;
  
protected:
  virtual int do_execute(void) ;
public:
  CmdMolAddRep(int newmol, int newUIid=(-1));
};


///////////// change a representation for the specified molecule
class CmdMolChangeRep : public Command {
public:
  // molecule to change
  int n;
  
  // which rep to change
  int repn;

protected:
  virtual int do_execute(void) ;
public:
  CmdMolChangeRep(int rpos, int newmol, int newUIid=(-1));
};


///////////// change 1 representation characteristic for the specified molecule
class CmdMolChangeRepItem : public Command {
public:
  // characteristics that can be modified
  enum RepData { COLOR, REP, SEL };

  // molecule to change
  int n;
  
  // which rep to change
  int repn;

  // type of item to change
  RepData repData;
  
  // new value
  char *str;

protected:
  virtual int do_execute(void) ;
public:
  CmdMolChangeRepItem(int rpos, int newmol, RepData rd, char *s,
  	int newUIid=(-1));
  virtual ~CmdMolChangeRepItem(void);
};


///////////// delete a representation for the specified molecule
class CmdMolDeleteRep : public Command {
public:
  // molecule to change
  int n;
  
  // which rep to delete
  int repn;

protected:
  virtual int do_execute(void) ;
public:
  CmdMolDeleteRep(int rpos, int newmol, int newUIid=(-1));
};


#endif
