/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: UIText.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.28 $	$Date: 1995/03/04 05:14:26 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * This is the User Interface for text commands.  It reads characters from
 * the console, and executes the commands.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: UIText.C,v $
 * Revision 1.28  1995/03/04  05:14:26  billh
 * Added geomtry labelling commands.
 *
 * Revision 1.27  1995/02/22  08:44:28  dalke
 * added external interface commands
 *
 * Revision 1.26  1994/12/07  07:48:33  billh
 * Temporarily took out text commands to change alpha, shininess, etc.
 * of colors.  Should be updated and put in UIText2.C
 *
 * Revision 1.25  94/12/06  08:22:44  billh
 * Added color commands.
 * 
 * Revision 1.24  94/11/24  07:27:03  dalke
 * Added "render format filename" and "render list"
 * 
 * Revision 1.23  1994/11/22  02:35:24  billh
 * added anim read,write,delete text commands.
 *
 * Revision 1.22  94/11/07  07:00:26  dalke
 * Added some commands to push/ pop Tools
 * 
 * Revision 1.21  1994/11/03  03:41:05  dalke
 * fixed modselect bug
 * fixed "scale by" bug
 * made delay a float
 *
 * Revision 1.20  1994/11/03  00:14:25  dalke
 * Added the 'wait' command
 *
 * Revision 1.19  1994/11/02  07:31:00  billh
 * Added commands to change display screen size and distance to screen.
 *
 * Revision 1.18  94/10/26  23:21:34  billh
 * Added routines to change the settings for graphics representations
 * (MOL_MODREP, MOL_MODREPITEM).
 * 
 * Revision 1.17  94/10/20  01:32:41  billh
 * Added animation commands.
 * 
 * Revision 1.16  1994/10/05  05:33:01  billh
 * arguments changed to allow to compile on HP's.
 *
 * Revision 1.15  1994/10/03  08:32:14  dalke
 * Added CmdTool Commands
 *
 * Revision 1.14  1994/10/03  01:40:25  dalke
 * Added a #include
 *
 * Revision 1.13  1994/10/01  11:01:00  billh
 * Added remote connection commands, put menu parsing commands in a
 * separate routine.
 *
 * Revision 1.12  94/09/30  19:11:34  dalke
 * took out references to forms when not compiled with VMDFORMS
 * 
 * Revision 1.11  1994/09/29  07:10:21  billh
 * Molecule commands updated to allow lists of indices.
 *
 * Revision 1.10  1994/09/26  18:52:36  billh
 * Updated molecule commands for adding and listing.
 *
 * Revision 1.9  94/09/23  00:59:37  billh
 * Added many more molecule commands.
 * 
 * Revision 1.8  1994/09/17  09:03:30  billh
 * Put in initial molecule commands (load and list).
 *
 * Revision 1.7  94/09/15  07:03:00  dalke
 * Made tracker text output commands into Commands
 * 
 * Revision 1.6  1994/09/14  04:10:59  billh
 * Removed some excess debugging text.
 *
 * Revision 1.5  1994/09/12  20:53:16  billh
 * Added 'view reset' command.
 *
 * Revision 1.4  94/09/07  07:49:38  dalke
 * added commands to manipulate the trackers via text
 * 
 * Revision 1.3  1994/09/05  22:50:38  billh
 * Added complete support for commands to position the axes and stage.
 *
 * Revision 1.2  1994/08/26  00:02:55  billh
 * Added 'rot stop' command and fixed problem with stereo mode command.
 *
 * Revision 1.1  94/08/24  03:10:37  billh
 * Initial revision
 * 
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /Home/h2/billh/projects/vmd/src/RCS/UIText.C,v 1.28 1995/03/04 05:14:26 billh Exp $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#if defined(ARCH_IRIX4) || defined(ARCH_IRIX5)
#include <bstring.h>
#endif
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <ctype.h>
#include <fcntl.h>
#include "utilities.h"
#include "config.h"
#include "UIText.h"

#include "CommandQueue.h"  // needed in Global.h, but were externs
#include "Axes.h"
#include "Stage.h"

#include "CmdDisplay.h"  // used for "quickie" commands in act on command
#include "CmdMol.h"
#include "CmdTrans.h"
#include "CmdUtil.h"
#include "CmdRender.h"
#include "CmdExternal.h"


// commands we are interested in
static int numCmds = 3;
static int cmdList[3] = { Command::HELP, Command::SHELL, Command::TEXT_EVENT};


// class constructor
UIText::UIText(UIList *uil, CommandQueue *cq)
  : UIObject("Text Console",uil, cq), input_files(64) {

  MSGDEBUG(1,"Creating UIText ..." << sendmsg);

  // record which commands we want
  for(int i=0; i < numCmds; command_wanted(cmdList[i++]));

  // display system prompt
  need_prompt();
  
  // don't wait for anything
  delay = 0;
  mytimer.clear();
}


// display the prompt for the user
void UIText::prompt(void) {

  printf(VMD_CMD_PROMPT,myName);
  fflush(stdout);
  
  // tell Message objects they need a newline before next output message
  msgInfo.need_newline(TRUE);
  msgWarn.need_newline(TRUE);
  msgErr.need_newline(TRUE);
  msgDebug.need_newline(TRUE);
}


// specify new file to read commands from
void UIText::read_from_file(char *fname) {
  FILE *f;
  if((f = fopen(fname,"r")) != NULL) {
    msgInfo << "Reading commands from '" << fname << "'." << sendmsg;
    input_files.push(f);
  } else {
    msgErr << "Cannot open file '" << fname << "' for reading." << sendmsg;
  }
}


// print out help summary
void UIText::help(void) {

  // display some help - try to keep things in alphabetical order
  msgInfo << "Is hopefully on it's way soon ..." << sendmsg;
}


// check for an event; return TRUE if we found an event; FALSE otherwise
int UIText::check_event(void) {
  static char cmdbuffer[1024];
  fd_set readvec;
  int ret, stdin_fd;
  struct timeval timeout;


  // check to see if I've waited long enough
  if (delay > 0) {
    if (delay < mytimer.clock_time()) {
      delay = 0;
      mytimer.stop();
      mytimer.reset();
    } else {
      return FALSE;  // gotta wait some more
    }
  }

  // check for text from input device
  if(input_files.stack_size() < 1) {
    if(needPrompt) {
      prompt();
      needPrompt = FALSE;
    }
    timeout.tv_sec = 0;
    timeout.tv_usec = 0;
    stdin_fd = 0;
    FD_ZERO(&readvec);
    FD_SET(stdin_fd, &readvec);
#if defined(ARCH_IRIX4) || defined(ARCH_IRIX5)
    ret = select(16,&readvec,NULL,NULL,&timeout);
#else
    ret = select(16,(int *)(&readvec),NULL,NULL,&timeout);
#endif
  
    if (ret == -1) {
      // got an error
      msgErr << "Error from select, while attempting to read text input."
             << sendmsg;
      return FALSE;
    } else if (ret == 0) {
      // time out
      return FALSE;
    }

    if (fgets(cmdbuffer,1024,stdin) == NULL) {
      addcommand(new CmdQuit(FALSE,id()));
      return FALSE;
    }
    
  } else {
    // just read in next line from current file, if available
    if(!fgets(cmdbuffer, 1024, input_files.top())) {
      msgInfo << "EOF encountered for current input file." << sendmsg;
      input_files.pop();	// now go back to reading previous file
      return FALSE;
    }
  }

  // strip off trailing newline
  int cmdstrlen = strlen(cmdbuffer);
  if(cmdstrlen > 0 && cmdbuffer[cmdstrlen - 1] == '\n')
    cmdbuffer[cmdstrlen - 1] = '\0';

  MSGDEBUG(3, "Read command: " << cmdbuffer << sendmsg);

  // create a text event, and queue it; the event object must deallocate the
  // string space allocated to hold the text command.
  addcommand(new TextEvent(stringdup(cmdbuffer), id()));

  return TRUE;
}


// update the display due to a command being executed.  Return whether
// any action was taken on this command.
// Arguments are the command type, command object, and the 
// success of the command (T or F).
int UIText::act_on_command(int type, Command *cmd, int suc) {
  int argc, txterr = FALSE;
  char *argv[256], *syscmd;
  int size[2], loc[2], *actsize = NULL, *actloc = NULL;

  MSGDEBUG(3,"UIText: acting on command " << type << sendmsg);

  // check all the possible commands that we look for ...
  if(type == Command::SHELL) {
    if(suc);		// keep compiler happy
    need_prompt();

  } else if(type == Command::HELP) {
    if(((CmdHelp *)cmd)->helpLoc == CmdHelp::HELP_CONSOLE)
      need_prompt();
    else
      return FALSE;
      
  } else if(type == Command::TEXT_EVENT) {

    // it's a text event, so tokenize and process the command
    if(!str_tokenize(((TextEvent *)cmd)->cmd,&argc,argv)) {
      need_prompt();
      return FALSE;
    }

    // make sure there is at least ONE token in the command, and skip comments
    if(argc < 1 || argv[0][0] == '#') {
      ;		// we processed it, but did nothing
    
    } else if (!strupncmp(argv[0],"quit",CMDLEN) || 
        !strupncmp(argv[0],"stop",CMDLEN) ||
        !strupncmp(argv[0],"end",CMDLEN)) {
      addcommand(new CmdQuit(FALSE,id()));
    
    } else if (!strupncmp(argv[0],"sh",CMDLEN) ||
		!strupncmp(argv[0],"shell",CMDLEN)) {
      int sl, i;
      for(sl = 0, i=1; i < argc; i++)  sl += strlen(argv[i]);
      syscmd = new char[sl + 8 + argc];		// extra buffer added
      for(i=1; i < argc; i++) {
        strcat(syscmd,argv[i]);  strcat(syscmd," ");
      }
      msgWarn << "About to execute shell command '" << syscmd << "' ..."
              << sendmsg;
      addcommand(new CmdShell(syscmd,id()));
      delete [] syscmd;
  
    } else if (!strupncmp(argv[0],"help",CMDLEN) ||
		!strupncmp(argv[0],"?",CMDLEN)) {
      addcommand(new CmdHelp(CmdHelp::HELP_CONSOLE,id()));
      
    } else if (!strupncmp(argv[0],"wait",CMDLEN) && (argc == 2)) {
      delay = atof(argv[1]);
      if (delay <= 0) {
        txterr = TRUE;
        delay = 0;
      } else {
        mytimer.start();
      }
    } else if (!strupncmp(argv[0],"debug",CMDLEN)) {
      if(argc == 2) {
	addcommand(new CmdDebug(CmdDebug::TOGGLE,
		!strupncmp(argv[1],"on",CMDLEN), id()));
      } else if(argc == 3) {
        if(!strupncmp(argv[1], "level",CMDLEN))
	  addcommand(new CmdDebug(CmdDebug::LEVEL, atoi(argv[2]), id()));
	else
	  txterr = TRUE;
      } else
        txterr = TRUE;
    } else if (!strupncmp(argv[0],"rot",CMDLEN) && (argc == 2 || argc == 4)) {
    
      if(argc == 2) {
        if(!strupncmp(argv[1],"stop",CMDLEN)) {
          addcommand(new CmdStopRot(id()));
        } else {
	  txterr = TRUE;
	}
      } else {
        char axis = (char)(tolower(argv[1][0]));
        float deg = atof(argv[3]);
        int rotby;
        if(axis < 'x' || axis > 'z') {
          deg = 0.0;
	  rotby = CmdRotate::BY;
        } else
          rotby = ( !strupcmp(argv[2],"by") ? CmdRotate::BY : CmdRotate::TO );
        addcommand(new CmdRotate(deg, axis, rotby, id()));
      }
      
    } else if (!strupncmp(argv[0],"trans",CMDLEN) && argc == 4) {
      char axis = (char)(tolower(argv[1][0]));
      float deg = atof(argv[3]);
      int trby = ( !strupcmp(argv[2],"by") ? 
      		CmdTranslate::BY : CmdTranslate::TO );
      float x = 0.0, y = 0.0, z = 0.0;
      if(axis == 'x')
        x = deg;
      else if(axis == 'y')
        y = deg;
      else if(axis == 'z')
        z = deg;
      addcommand(new CmdTranslate(x, y, z, trby, id()));
      
    } else if (!strupncmp(argv[0],"scale",CMDLEN) && argc == 3) {
      float deg = atof(argv[2]);
      int scby = ( !strupcmp(argv[1],"by") ? CmdScale::BY : CmdScale::TO );
      addcommand(new CmdScale(deg, scby, id()));
      
    } else if (!strupncmp(argv[0],"rock",CMDLEN) && 
    	(argc == 2 || (argc >= 4 && argc <= 5)) ) {
      if(argc == 2) {
        if(!strupncmp(argv[1],"off",CMDLEN))
	  addcommand(new CmdRockOff(id()));
	else
	  txterr = TRUE;
      } else {
        char axis = (char)(tolower(argv[1][0]));
        float deg = atof(argv[3]);
        int steps = (-1);
        if(argc == 5)
          steps = atoi(argv[4]);
        addcommand(new CmdRockOn(deg, axis, steps, id()));
      }

    } else if (!strupncmp(argv[0],"axes",CMDLEN) && argc == 3 && axes) {
      if(!strupncmp(argv[1],"location",CMDLEN)) {
        for(int i=0; i < axes->locations(); i++) {
	  if(!strupncmp(argv[2],axes->loc_description(i),CMDLEN)) {
            addcommand(new CmdDisplayAxes(i, id()));
	    break;
	  }
	}
	if(i == axes->locations())
	  txterr = TRUE;
      } else
        txterr = TRUE;

    } else if (!strupncmp(argv[0],"stage",CMDLEN) && argc == 3 && stage) {
      if(!strupncmp(argv[1],"location",CMDLEN)) {
        for(int i=0; i < stage->locations(); i++) {
	  if(!strupncmp(argv[2],stage->loc_description(i),CMDLEN)) {
            addcommand(new CmdDisplayStage(CmdDisplayStage::LOCATION,
	    	i, id()));
	    break;
	  }
	}
	if(i == stage->locations())
	  txterr = TRUE;
      } else if(!strupncmp(argv[1],"panels",CMDLEN)) {
        addcommand(new CmdDisplayStage(CmdDisplayStage::PANELS,
		atoi(argv[2]), id()));
      } else
        txterr = TRUE;

    } else if (!strupncmp(argv[0],"light",CMDLEN) && argc >= 2) {
      // commands to change lights
      int n = atoi(argv[1]);
      
      if(argc == 3) {
        if(!strupncmp(argv[2],"on",CMDLEN))
	  addcommand(new CmdDisplayLightOn(n, TRUE, id()));
	else if(!strupncmp(argv[2],"off",CMDLEN))
	  addcommand(new CmdDisplayLightOn(n, FALSE, id()));
        else if(!strupncmp(argv[2],"highlight",CMDLEN))
	  addcommand(new CmdDisplayLightHL(n, TRUE, id()));
	else if(!strupncmp(argv[2],"unhighlight",CMDLEN))
	  addcommand(new CmdDisplayLightHL(n, FALSE, id()));
        else
	  txterr = TRUE;
      } else if(argc == 5 && !strupncmp(argv[2],"rot",CMDLEN)) {
        char axis = (char)(tolower(argv[3][0]));
        float deg = atof(argv[4]);
	addcommand(new CmdDisplayLightRot(n, deg, axis, id()));
      } else
        txterr = TRUE;

/*
 *  SHOULD BE REDONE, USING NEW COLOR COMMANDS IN CmdColor.[Ch] ....
 *  BETTER TEXT FORMAT WOULD BE 'color change <name> <name2>', or
 *                              'color change <name> <f1> <f2> <f3> for color,
 *                              'color change <name> alpha <f1> ,
 *                              'color change <name> shininess <f1> ,
 *                              'color change <name> ambient <f1> <f2> <f3> ,
 *                              etc
    } else if (!strupncmp(argv[0],"color",CMDLEN) && argc >= 3) {
      // commands to change colors
      int n = atoi(argv[1]);

      if(argc == 4) {
        float d = atof(argv[3]);
        if(!strupncmp(argv[2], "alpha", CMDLEN))
	  addcommand(new CmdDisplayMaterialsChange(n, ALPHA_INDEX, &d, id()));
	else if(!strupncmp(argv[2], "shininess", CMDLEN))
	  addcommand(new
	  	CmdDisplayMaterialsChange(n,SHININESS_INDEX,&d,id()));
        else
	  txterr = TRUE;
      } else if(argc == 6) {
        float d[3];
	for(int i=0; i < 3; i++)  d[i] = atof(argv[3+i]);
	if(!strupncmp(argv[2], "color", CMDLEN))
	  addcommand(new CmdDisplayMaterialsChange(n,COLOR_INDEX,d,id()));
	else if(!strupncmp(argv[2], "ambient", CMDLEN))
	  addcommand(new CmdDisplayMaterialsChange(n,AMBIENT_INDEX,d,id()));
	else if(!strupncmp(argv[2], "diffuse", CMDLEN))
	  addcommand(new CmdDisplayMaterialsChange(n,DIFFUSE_INDEX,d,id()));
	else if(!strupncmp(argv[2], "specular", CMDLEN))
	  addcommand(new CmdDisplayMaterialsChange(n,SPECULAR_INDEX,d,id()));
	else
	  txterr = TRUE;
      } else
        txterr = TRUE;
 *
 *
 */
    } else if (!strupncmp(argv[0],"journal",CMDLEN) ||
		!strupncmp(argv[0],"log",CMDLEN) ||
		!strupncmp(argv[0],">>",CMDLEN)) {
      // log file on/off commands
      if(argc == 2) {
        if (!strupncmp(argv[1],"off",CMDLEN))
          cmdQueue->log_off();
        else
          cmdQueue->log_on(argv[1]);
      }	else
	txterr = TRUE;

    } else if (!strupncmp(argv[0],"play",CMDLEN) ||
	       !strupncmp(argv[0],"<<",CMDLEN)) {
      // read commands from a file
      if(argc == 2) {
        read_from_file(argv[1]);
      } else
        txterr = TRUE;

    } else if (!strupncmp(argv[0], "tracker", CMDLEN)) {
      txterr = texteval_tracker(argc, argv);  // makes things less cluttered

    } else if (!strupncmp(argv[0], "render",CMDLEN)) {
      // render postscript output.ps
      if (argc == 3) {
        addcommand(new CmdRender(argv[1], argv[2], id()));
      } else if (argc == 2 && !strupncmp(argv[1], "list", CMDLEN)) {
        addcommand(new CmdRenderList(id()));
      } else
        txterr = TRUE;
    } else if (!strupncmp(argv[0], "view",CMDLEN)) {
      if(argc == 2 && !strupncmp(argv[1], "reset",CMDLEN))
        addcommand(new CmdResetView(id()));
      else
        txterr = TRUE;

    } else if (!strupncmp(argv[0],"display",CMDLEN)) {
      // display commands

      if(argc == 2 && !strupncmp(argv[1],"reshape",CMDLEN)) {
        addcommand(new CmdDisplayReshape(id()));

      } else if(argc == 3) {
        if(!strupncmp(argv[1],"eyesep",CMDLEN)) {
	  addcommand(new CmdDisplayEyesep(atof(argv[2]), id()));
	} else if(!strupncmp(argv[1],"focallength",CMDLEN)) {
	  addcommand(new CmdDisplayFocallen(atof(argv[2]), id()));
	} else if(!strupncmp(argv[1],"height",CMDLEN)) {
	  addcommand(new CmdDisplayScreenHeight(atof(argv[2]), id()));
	} else if(!strupncmp(argv[1],"distance",CMDLEN)) {
	  addcommand(new CmdDisplayScreenDistance(atof(argv[2]), id()));
 	} else if(!strupncmp(argv[1],"antialias",CMDLEN)) {
	  addcommand(new CmdDisplayAAOn(!strupcmp(argv[2],"on"), id()));
 	} else if(!strupncmp(argv[1],"depthcue",CMDLEN)) {
	  addcommand(new CmdDisplayDepthcueOn(!strupcmp(argv[2],"on"), id()));
	} else if(!strupncmp(argv[1],"stereo",CMDLEN)) {
	  int i, modes = display->num_stereo_modes();
	  for(i=0; i < modes; i++)
	    if(!strupncmp(argv[2],display->stereo_name(i),3)) {
	      addcommand(new CmdDisplayStereo(i, id()));
	      break;
	    }
	  txterr = (i == modes);
        } else
	  txterr = TRUE;

      } else if(argc == 4) {
        if(!strupncmp(argv[1],"nearclip",CMDLEN)) {
	  if(!strupncmp(argv[2],"set",CMDLEN))
	    addcommand(new CmdDisplayClipNear(atof(argv[3]), id()));
	  else if(!strupncmp(argv[2],"add",CMDLEN))
	    addcommand(new CmdDisplayClipNearRel(atof(argv[3]), id()));
	  else txterr = TRUE;
        } else if(!strupncmp(argv[1],"farclip",CMDLEN)) {
 	  if(!strupncmp(argv[2],"set",CMDLEN))
	    addcommand(new CmdDisplayClipFar(atof(argv[3]), id()));
	  else if(!strupncmp(argv[2],"add",CMDLEN))
	    addcommand(new CmdDisplayClipFarRel(atof(argv[3]), id()));
	  else txterr = TRUE;
	} else
	  txterr = TRUE;
      }

    } else if (!strupncmp(argv[0],"newmol",CMDLEN)) {
      // new molecule commands
      if (argc==3 || argc==5) {
        int sftype = MoleculeFile::UNKNOWN;
	int cftype = CoorFileData::UNKNOWN;
	char *cfstr = NULL;
	int i;

	// find structure file type
	for(i=0; i < MoleculeFile::STRTYPES; i++) {
          if(!strupncmp(argv[1], structureFileTypeNames[i], CMDLEN))
	    sftype = i;
	}
	if(sftype == MoleculeFile::UNKNOWN) {
	  msgErr << "Unknown structure file type " << argv[1]  << sendmsg;
	}
	
	if(argc == 5) {
	  // find coordinate file type
	  for(i=0; i < CoorFileData::COORTYPES; i++) {
            if(!strupncmp(argv[3], CoorFileSuffix[i], CMDLEN))
	      cftype = i;
	  }
	  if(cftype == CoorFileData::UNKNOWN) {
	    msgErr << "Unknown coordinate file type " << argv[3] << sendmsg;
	  } else {
	    cfstr = argv[4];
	  }
	}
	
	if(sftype != MoleculeFile::UNKNOWN)
	  addcommand(new CmdMolNew(argv[2],sftype,cfstr,cftype,id()));
	else
	  txterr = TRUE;
      } else
        txterr = TRUE;

   } else if (!strupncmp(argv[0], "external", CMDLEN) && argc == 2) {
      if (!strupncmp(argv[1], "on", CMDLEN) ||
	  !strupncmp(argv[1], "begin", CMDLEN) ||
	  !strupncmp(argv[1], "start", CMDLEN)) {
	 addcommand(new CmdExternalStart(id()));
      } else if (!strupncmp(argv[1], "off", CMDLEN) ||
		 !strupncmp(argv[1], "end", CMDLEN) ||
		 !strupncmp(argv[1], "stop", CMDLEN)) {
	 addcommand(new CmdExternalEnd(id()));
      } else {
	 txterr = TRUE;
      }

    } else if (!strupncmp(argv[0],"mol",CMDLEN) && argc > 1) {
      // general molecule commands
      txterr = texteval_molecule(argc, argv);  // makes things less cluttered

    } else if (!strupncmp(argv[0],"remote",CMDLEN) && argc > 1) {
      // general remote connection setup commands
      txterr = texteval_remote(argc, argv);

    } else if (!strupncmp(argv[0],"menu",CMDLEN) && argc >= 3) {
      // general menu commands
      txterr = texteval_menu(argc, argv);

    } else if (!strupncmp(argv[0],"anim",CMDLEN) && argc >= 2) {
      // general animation commands
      txterr = texteval_anim(argc, argv);

    } else if (!strupncmp(argv[0],"color",CMDLEN) && argc >= 2) {
      // general color editing commands
      txterr = texteval_color(argc, argv);

    } else if (!strupncmp(argv[0],"tool",CMDLEN)) {
      txterr = texteval_tool(argc, argv);

    } else if (!strupncmp(argv[0],"label",CMDLEN) && argc > 1) {
      // general labelling commands
      txterr = texteval_label(argc, argv);  // makes things less cluttered

    } else
      // did not understand command ... error
      txterr = TRUE;
    
    if(txterr)
      // must be an unknown command
      msgErr << "Invalid command '" << ((TextEvent *)cmd)->cmd << "'."
             << sendmsg;

    // display the prompt again
    need_prompt();
    
  } else
    // unknown command type
    return FALSE;

  return TRUE;
}
