/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Scene.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.7 $	$Date: 1994/12/06 08:26:45 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Scene object, which maintains a list of Displayable objects and
 * draws them to a DisplayDevice.
 * Each Scene has a list of Displayable objects, and also a list of display
 * commands.  The command lists are used to draw the objects, the Displayable
 * objects to prepare and update objects for drawing.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: Scene.C,v $
 * Revision 1.7  1994/12/06  08:26:45  billh
 * Added routine color_changed, which simply calls this same routine
 * for all registered objects.  color_changed is called when a color in a
 * specific category is changed.
 *
 * Revision 1.6  94/11/24  07:27:03  dalke
 * Added filedraw for writing the scene to some external graphics format
 * 
 * Revision 1.5  1994/10/05  04:38:23  billh
 * Took out double backslash from text, even in comments.
 *
 * Revision 1.4  1994/10/01  03:06:53  billh
 * New format; Scene now a more general base class, from which are derived
 * NormalScene and CaveScene (for starters ...)
 *
 * Revision 1.3  94/09/29  19:27:11  dalke
 * Added mem. alloc/free functions so I can use shared mem.
 * Split draw routine up into prepare/draw/update
 * 
 * Revision 1.2  1994/09/14  03:55:10  billh
 * Changed due to restructuring of Displayable ... now calls
 * 'draw_prepare' and 'draw_update' for Displayable's.
 *
 * Revision 1.1  94/08/24  03:10:37  billh
 * Initial revision
 * 
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /Home/h2/billh/projects/vmd/src/RCS/Scene.C,v 1.7 1994/12/06 08:26:45 billh Exp $";
#endif

#include "Scene.h"
#include "DisplayDevice.h"
#include "Inform.h"
#include "DispCmds.h"
#include "utilities.h"
//#include "RenderList.h"

///////////////////////////  constructor 
Scene::Scene(void) {

  MSGDEBUG(1,"Creating a Scene ..." << sendmsg);

  numCmdLists2D = numCmdLists3D = 0;
  numDisplayable2D = numDisplayable3D = 0;
  Rocking = FALSE;
  rockSteps = currRockStep = 0;
  rockAmount = 0.0;
  rockAxis = 'y';
}

///////////////////////////  public routines 

// clear the entire Scene, i.e. remove all displayable objects
void Scene::clear(void) {
  // go through list of Displayables, ask them all to unregister with this
  // Scene object
  while(num_displayable2D() > 0)
    (displayable2D(0))->unRegister(this);
  while(num_displayable3D() > 0)
    (displayable3D(0))->unRegister(this);
}


// do action due to the fact that a color for the given ColorList for
// the specified category has changed
void Scene::color_changed(ColorList *changelist, int clr) {
  int i;

  // just call this routine for all the registered displayables
  int num3D = num_displayable3D();
  for(i=0; i < num3D; i++)
    (displayable3D(i))->color_changed(changelist, clr);

  int num2D = num_displayable2D();
  for(i=0; i < num2D; i++)
    (displayable2D(i))->color_changed(changelist, clr);
}


// prepare all registered Displayables
void Scene::prepare(DisplayDevice *display) {
  int i;

  if (!display)
    return;

  // rock the scene, if necessary
  if(Rocking) {
    add_rot(rockAmount, rockAxis);
    if(rockSteps >= 0 && ++currRockStep >= rockSteps) {
      currRockStep = 0;
      rockAmount *= -1.0;		// reverse direction of rocking
    }
  }

  // prepare all registered Displayables
  int num3D = num_displayable3D();
  MSGDEBUG(3,"Scene: preparing " << num3D << " 3D objects ..." << sendmsg);
  for(i=0; i < num3D; i++)
    (displayable3D(i))->draw_prepare(display);

  int num2D = num_displayable2D();
  MSGDEBUG(3,"Scene: preparing " << num2D << " 2D objects ..." << sendmsg);
  for(i=0; i < num2D; i++)
    (displayable2D(i))->draw_prepare(display);
}


// draws the scene to the given DisplayDevice
// this is the only Scene which tells the display to do graphics commands
void Scene::draw(DisplayDevice *display) {
  int i;
  char *cmdlist;

  if(!display)
    return;

  // draw 3D objects first ... prepare the display
  display->prepare3D(TRUE);

  // draw left eye first, if stereo
  if(display->stereo_mode())
    display->left();
  else
    display->normal();
      
  // draw all the objects for left eye position
  MSGDEBUG(3,"Scene: drawing (L) " << num_disp_list3D() << " 3D objects ...");
  MSGDEBUG(3,sendmsg);
  for(i=0; i < num_disp_list3D(); i++) {
    cmdlist = disp_list3D(i);
    if( ((int *)cmdlist)[1] )
      display->render((void *)((int *)cmdlist + 3));
  }
    
  // now draw right eye, if in stereo
  if(display->stereo_mode()) {
    display->right();
    MSGDEBUG(3,"Scene: drawing (R)" << num_disp_list3D() << " 3D objects ...");
    MSGDEBUG(3,sendmsg);
    for(i=0; i < num_disp_list3D(); i++) {
      cmdlist = disp_list3D(i);
      if( ((int *)cmdlist)[1] )
        display->render((void *)((int *)cmdlist + 3));
    }
  }

  // draw 2D objects last
  if(display->has_2D() && num_disp_list2D() > 0) {
    display->prepare2D(FALSE);
    display->normal();
      
    MSGDEBUG(3,"Scene: drawing " << num_disp_list2D() << " 2D objects ...");
    MSGDEBUG(3,sendmsg);
    // draw all the objects for left eye position
    for(i=0; i < num_disp_list2D(); i++) {
      cmdlist = disp_list2D(i);
      if( ((int *)cmdlist)[1] )
        display->render((void *)((int *)cmdlist + 3));
    }
  }
  
  // update the display
  display->update(TRUE);
}

// draw the scene to a file in a given format, trying to match the
// view of the given DisplayDevice as closely as possible
// returns TRUE if successful, FALSE if not
int Scene::filedraw(char *method, char *filename, DisplayDevice *display){
//  FileRender *render = fileRenderList.find(rendername);
//  if (!render) { 
//    msgErr << "I don't know how to do the output format ";
//    msgErr << method << "." << sendmsg;
//    if (fileRenderList.num() == 0) {
//     msgInfo << "In fact, there are no possible output formats." << sendmsg;
//    } else {
//     msgInfo << "The valid output formats are:" << sendmsg;
//     for (int i=0; i<fileRenderList.num(); i++) {
//      msgInfo << i << "  " << renderList[i] << sendmsg;
//    }
//    return FALSE;
//  }
//  render -> init(filename, display);
//  for(i=0; i < num_disp_list3D(); i++) {
//    cmdlist = disp_list3D(i);
//      if( ((int *)cmdlist)[1] )
//        render -> render ((void *)((int *)cmdlist + 3));
//  }
//  render -> done();
    msgInfo << "Scene::filedraw not yet implemented" << sendmsg;
    return TRUE;
}

///////////////////  routines to affect all Displayables in the Scene 

// just change the transformation to the one given
void Scene::load_transformation(Matrix4 &m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->load_transformation(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->load_transformation(m);
} 

// reset to identity the transformation
void Scene::reset_transformation(void) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->reset_transformation();
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->reset_transformation();
}


void Scene::set_scale(float s) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->set_scale(s);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->set_scale(s);
}

void Scene::mult_scale(float s) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->mult_scale(s);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->mult_scale(s);
}

void Scene::set_scale(Matrix4 &m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->set_scale(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->set_scale(m);
}

void Scene::mult_scale(Matrix4 &m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->mult_scale(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->mult_scale(m);
}

void Scene::add_rot(float x, char axis) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->add_rot(x, axis);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->add_rot(x, axis);
}

void Scene::add_rot(Matrix4 &m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->add_rot(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->add_rot(m);
}

void Scene::set_rot(Matrix4 &m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->set_rot(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->set_rot(m);
}

void Scene::set_glob_trans(float x, float y, float z) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->set_glob_trans(x, y, z);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->set_glob_trans(x, y, z);
}

void Scene::set_glob_trans(Matrix4& m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->set_glob_trans(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->set_glob_trans(m);
}

void Scene::add_glob_trans(float x, float y, float z) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->add_glob_trans(x, y, z);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->add_glob_trans(x, y, z);
}

void Scene::add_glob_trans(Matrix4& m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->add_glob_trans(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->add_glob_trans(m);
}

void Scene::set_cent_trans(float x, float y, float z) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->set_cent_trans(x, y, z);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->set_cent_trans(x, y, z);
}

void Scene::set_cent_trans(Matrix4& m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->set_cent_trans(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->set_cent_trans(m);
}

void Scene::add_cent_trans(float x, float y, float z) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->add_cent_trans(x, y, z);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->add_cent_trans(x, y, z);
}

void Scene::add_cent_trans(Matrix4& m) {
  int i;
  for(i=0; i < num_displayable2D(); i++)
    (displayable2D(i))->add_cent_trans(m);
  for(i=0; i < num_displayable3D(); i++)
    (displayable3D(i))->add_cent_trans(m);
}
