/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Pickable.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.1 $	$Date: 1995/02/22 03:56:04 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A Pickable object is one which contains data which may be selected by
 * using a pointer device in an open DisplayDevice.  Each Pickable registers
 * itself with a PickList object which indicates it has items in its
 * drawing list which may be picked, and which indicates it should be told
 * when something has been successfully selected.  There are one or many
 * picking modes, which are represented by derivations of the PickMode class.
 * When picking is done, it is done while in a current pick mode, which
 * controls what to do with the information.
 *
 * A Pickable must provide versions of virtual routines which tell what
 * pick modes the object is interested in.
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: Pickable.C,v $
 * Revision 1.1  1995/02/22  03:56:04  billh
 * Initial revision
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /Home/h2/billh/projects/vmd/src/RCS/Pickable.C,v 1.1 1995/02/22 03:56:04 billh Exp $";
#endif

#include "Pickable.h"
#include "PickList.h"


//////////////////////////// constructor  ////////////////////////////
Pickable::Pickable(PickList *p) {

  // init picklist pointer
  pickList = NULL;
  
  // possibly register with this pickList
  register_with_picklist(p);
}


// destructor: unregister with the picklist
Pickable::~Pickable(void) {
  if(pickList)
    pickList->remove_pickable(this);
}


//////////////////////////// public routines  ////////////////////////////

// register with the given PickList.  If already registered, deregisters
// with the old one first
void Pickable::register_with_picklist(PickList * p) {

  // only do anything if p is not NULL
  if(p) {
    // deregister from picklist if necessary first
    if(pickList)
      pickList->remove_pickable(this);
      
    // register with new picklist
    pickList = p;
    pickList->add_pickable(this);
  }
}


// check to see if a PickMode with the given name has been created yet
// for the PickList.  If so, return the index of the mode.  If not,
// this routine calls the virtual create_pick_mode with the 2nd argument,
// which is a code for the derived class to know what kind of PickMode
// object to create.  Returns (-1) if some sort of error occurs.
int Pickable::add_pick_mode(char *name, int mcode) {

  // can only work if we have a current pick list
  if(!pickList)
    return (-1);

  // for our current PickList object, check if mode already exists
  int indx = pickList->pick_mode_num(name);

  // if it does not exist, create a new one first
  if(indx < 0) {
    PickMode *pm = create_pick_mode(mcode);
    if(!pm)
      return (-1);
    indx = pickList->add_pick_mode(name, pm);
  }
  
  // return the index of the mode
  return indx;
}
