/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Mouse.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.8 $	$Date: 1995/02/22 03:57:58 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Mouse UI object, which maintains the current state of the 
 * mouse, including what it is currently used for, and how much it has moved
 * from one measurement to the next.  This also deals with any pop-up or
 * pull-down menus available by using the mouse, as well as picking objects.
 *
 * A three-button mouse is assumed here, with the following usage:
 *	1) Buttons 1 and 2 : manipulation and picking.
 *	2) Button 3 (right): pop-up menu
 *
 * This is the general base class definition; specific versions for each
 * windowing/graphics system may be supplied.  This class can function as
 * it is, however, and will just be a 'zero' mouse - the buttons will never
 * be pressed, and the cursor will always be at 0,0.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: Mouse.h,v $
 * Revision 1.8  1995/02/22  03:57:58  billh
 * Now supports picking objects with a 2D or 3D pointer.
 *
 * Revision 1.7  94/11/09  03:43:18  billh
 * Added support to check state of function keys as if they were mouse buttons.
 * 
 * Revision 1.6  1994/10/28  18:33:29  billh
 * Converted to a general base class (non-GL-specific) which can be used
 * throughout the program.  A GL-specific Mouse is in GLMouse.h
 *
 * Revision 1.5  1994/10/19  20:22:20  billh
 * Added pop-up menu commands to turn forms on and off.
 *
 * Revision 1.4  1994/10/05  04:38:23  billh
 * Took out double backslash from text, even in comments.
 *
 * Revision 1.3  1994/10/04  20:29:22  billh
 * Changes to get to compile on HP's ... not there yet,though.
 *
 * Revision 1.2  1994/08/26  00:02:04  billh
 * Improved pop-up menu; now produces submenus to attach mouse to lights,
 * set the stereo mode, toggle lights on/off, and a couple other utilities.
 *
 * Revision 1.1  1994/08/24  03:10:37  billh
 * Initial revision
 *
 ***************************************************************************/
#ifndef MOUSE_OBJ_H
#define MOUSE_OBJ_H

#include "UIObject.h"
#include "Command.h"


class Mouse : public UIObject {

public:
  // different available mouse movement modes
  enum MoveMode { ROTATION, TRANSLATION, SCALING, LIGHT, PICKING };

  // different buttons of the mouse
  enum Buttons { B_LEFT, B_MIDDLE, B_RIGHT, B_F1, B_F2, B_F3, B_F4, B_F5,
  	B_F6, B_F7, B_F8, B_F9, B_F10, TOTAL_BUTTONS };
  
protected:
  // the current move mode
  MoveMode moveMode;
  
  // are we currently trying to pick something?
  // If so, set this to picking mode in use.
  int activePickMode;

  // flag for whether an active pick is in progress.
  // if 0, no pick.  if > 0, pickInProgress = pick-button - 1
  int pickInProgress;

  // which particular object the mouse is affecting (if mode == LIGHT)
  int moveObj;

  // the current button modes, down or up, and whether the mouse moved while
  // they were pressed
  int b1Down, b2Down;
  int b1Moved, b2Moved;
  
  // current & last position of the mouse, and last angular velocity 
  int currX, currY, oldX, oldY;
  
  // increment for mouse translation, rotation
  float transInc, rotInc, scaleInc;

  // rotational velocities, and scaling factor, and flag for whether rotating
  float xRotVel, yRotVel, zRotVel, scaling, RotVelScale;

public:
  // constructor/destructor
  Mouse(UIList *, CommandQueue *);
  virtual ~Mouse(void);
  
  //
  // general mouse actions
  //

  // stop rotation of object
  void stop_rotation(void);

  //
  // routines to query about the status of the mouse
  //

  // return the current x or y position of the mouse
  int x(void) { return currX; }
  int y(void) { return currY; }

  // return whether the given button is down (TRUE) or up (FALSE)
  virtual int button_down(int) { return FALSE; }

  //
  // routines used to deal with the current mouse mode
  //

  // set the current move mode
  virtual int move_mode(MoveMode, int = 0);
  
  // return the current move mode
  MoveMode curr_move_mode(void) { return moveMode; }

  // string name for the given move mode
  char *move_mode_name(MoveMode);

  //
  // routines used to deal with the current picking mode
  //

  // set the active pick mode for this mouse
  // (does things like change cursor style)
  // return success.
  virtual int pick_mode(int);

  // return the current active pick mode
  int curr_pick_mode(void) { return activePickMode; }

  // return whether we are currently picking something
  int picking(void) { return pickInProgress; }

  //
  // virtual routines for UI init/display
  // THE DERIVED CLASS, IF NECESSARY, SHOULD PROVIDE THEIR OWN VERSIONS,
  // AND THEN CALL THESE ROUTINES AS WELL
  //
   
  // initialize this user interface
  virtual void init(void) { reset(); }
  
  // reset the user interface (force update of all info displays)
  virtual void reset(void);
  
  // update the displays
  virtual void update(void) { }

  //
  // virtual routines for entering/processing commands and events
  // MUST BE SUPPLIED BY DERIVED CLASS: act_on_command and check_event
  //
  
};


//////////////////////  Windowing System Event Object 
class WinEvent : public Command {

public:
  // possible window events
  enum EventCodes { WIN_REDRAW, WIN_LEFT, WIN_MIDDLE, WIN_RIGHT,
  	WIN_MOUSEX, WIN_MOUSEY, WIN_KEYBD,
	WIN_UPDATE, WIN_INPUTCHANGE };

  // the type of event
  int eventCode;
  
  // the return value for the event
  int val;

  // constructor; must specify what the event type code and value are
  WinEvent(int d, int v, int newUIid=(-1))
    : Command(Command::WIN_EVENT, newUIid) {
    eventCode = d;
    val = v;
  }

};

#endif
