/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CommandQueue.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.5 $	$Date: 1995/01/08 22:08:03 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * This stores all the Commands to be run in a queue.  The idea is that
 * the various events add Commands to the command queue then they
 * are read off the queue and the UIs are notified.
 *
 * Commands may be logged to a file, if desired.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: CommandQueue.h,v $
 * Revision 1.5  1995/01/08  22:08:03  billh
 * Uses config.h instead of version.h
 *
 * Revision 1.4  1994/10/05  06:58:16  billh
 * Converted to use a ResizeArray instead of a DLinkList in the command queue
 * and ui list.
 *
 * Revision 1.3  1994/10/04  20:29:22  billh
 * Changes to get to compile on HP's ... not there yet,though.
 *
 * Revision 1.2  1994/09/12  20:52:37  billh
 * Made some debugging messages L3 instead of L2 (to reduce L2 verbosity).
 *
 * Revision 1.1  94/08/24  03:10:37  billh
 * Initial revision
 *  
 ***************************************************************************/
#ifndef COMMANDQUEUE_H
#define COMMANDQUEUE_H

#include <stdio.h>
#include <string.h>
#include "ResizeArray.h"
#include "Command.h"
#include "UIList.h"
#include "utilities.h"
#include "config.h"


/* NOTES:
 *	1) To add new commands to queue, use routine 'append(Command *)',
 *		inherited since this is a ResizeArray<> object.
 *	2) To do something, use 'execute' routine, with a given UIList obj.
 *		This will execute the top command and inform all the UIs in the
 *		UIList.
 *	3) 'execute_all' will do all the commands until the queue is empty.
 */

class CommandQueue {

  private:
    // the command list itself
    ResizeArray<Command *> cmdlist;

    // delete the current element in the linked list.  Needed since we
    // must delete the Command object stored as well as the list element.
    void delete_current(void) {
      if(cmdlist.num() > 0) {

        MSGDEBUG(3,"CQ: Deleting '" << *(cmdlist[0]) << "' ..." << sendmsg);
	
        delete cmdlist[0];	// delete the Command object stored
	cmdlist.remove(0);
      }
    }

    // are we logging commands to a file?  If so, which one?
    int loggingCmds;
    FILE *logfile;

  public:
    // constructor
    CommandQueue( void) : cmdlist(128) {
      loggingCmds = FALSE;
      logfile = NULL;
    }
    
    // destructor ... we must remove all commands, and delete them as well.
    // if logging, must close file
    ~CommandQueue( void) {
      while(cmdlist.num() > 0)
        delete_current();
      if(logging())
        log_off();
    }

    //
    // log file routines
    //
    
    // check if logging
    int logging(void) { return loggingCmds; }
    
    // turn on logging ... open file, and write header comments
    void log_on(char *fname) {
      if(!logging()) {
        if((logfile = fopen(fname, "w")) != NULL) {
	  msgInfo << "Logging commands to '" << fname << "'." << sendmsg;
	  loggingCmds = TRUE;
          fprintf(logfile,"# %s\n", VERSION_MSG);
	  fprintf(logfile,"# Log file '%s', created by user %s\n", fname,
		  username());
        } else {
          msgErr << "Cannot open log file '" << fname << "' for writing."
	         << sendmsg;
        }
      } else {
        msgErr << "Cannot open log file '" << fname << "' ... already "
	       << "logging commands to a file." << sendmsg;
      }
    }
    
    // turn off logging ... close the file
    void log_off(void) {
      if(logging()) {
        fprintf(logfile,"# %s\n", VERSION_MSG);
	fprintf(logfile,"# end of log file.\n");
        fclose(logfile);
	loggingCmds = FALSE;
	msgInfo << "Log file closed." << sendmsg;
      } else {
        msgErr << "Not currently logging commands." << sendmsg;
      }
    }

    //
    // command processing routines
    //
    
    // add a new command to the list
    void append(Command *cmd) {
      cmdlist.append(cmd);
    }

    // return number of commands yet to execute
    int num(void) {
      return cmdlist.num();
    }

    // execute the first command in the queue
    void execute( UIList *UIL) {
      if(cmdlist.num() > 0) {
      
        MSGDEBUG(3,"CQ: Executing '" << *(cmdlist[0]) << "' ..." << sendmsg);
	
        // if logging, write to log file
	if(logging() && strlen((cmdlist[0])->cmdtextbuf) > 0)
	  fprintf(logfile,"%s\n",(cmdlist[0])->cmdtextbuf);

        // do the command, and report to UIList the action has been done
        UIL->act_on_command_UI( (cmdlist[0])->gettype(), cmdlist[0], 
                                (cmdlist[0])->execute());

        // remove this command
        delete_current();
      }
    }

    // execute ALL the commands in the queue
    void execute_all( UIList *UIL) {
      if (UIL)
        while (cmdlist.num() > 0)
          execute(UIL);
    }
};

#endif
